<?php
declare(strict_types=1);

require_once "inc/config.php";
require_once "inc/functions.php";
require_once "inc/keyboards.php";
require_once "inc/database.php";
require_once "inc/auth.php";
require_once "inc/validators.php";
require_once "inc/logger.php";
require_once "inc/update-processor.php";
require_once "inc/handlers/upload-handler.php";
require_once "inc/handlers/admin-handler.php";
require_once "inc/handlers/channel-handler.php";
require_once "inc/handlers/point-handler.php";
require_once "inc/handlers/referral-handler.php";
require_once "inc/handlers/auto-delete-handler.php";
require_once "inc/handlers/analytics-handler.php";
require_once "inc/handlers/payment-handler.php";
require_once "inc/handlers/broadcast-handler.php";
require_once "inc/handlers/advertisement-handler.php";
require_once "inc/handlers/media-handler.php";
require_once "inc/handlers/user-handler.php";
require_once "inc/handlers/settings-handler.php";
require_once "inc/handlers/daily-spin-handler.php";
require_once "inc/handlers/gambling-handler.php";

initDatabase();

$processed = processUpdate();

if ($processed === false) {
    exit;
}

$update = $processed['update'];
$message = $processed['message'];
$callback_query = $processed['callback_query'];
$pre_checkout_query = $processed['pre_checkout_query'];
$message_reaction = $processed['message_reaction'];
$chat_id = $processed['chat_id'];
$user_id = $processed['user_id'];
$user_data = $processed['user_data'];
$text = $processed['text'];
$chat_type = $processed['chat_type'];
$callback_message_id = $processed['callback_message_id'] ?? null;

if ($message_reaction) {
    require_once "inc/handlers/channel-handler.php";
    handleMessageReaction($message_reaction);
    exit;
}

if ($pre_checkout_query) {
    handleSuccessfulPayment($pre_checkout_query);
    exit;
}

if ($callback_query) {
    $callback_data = $callback_query['data'] ?? '';
    
    bot('answerCallbackQuery', [
        'callback_query_id' => $callback_query['id']
    ]);
    
    switch ($callback_data) {
        case 'media':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "📁 مدیریت مدیا:", getInlineKeyboard('media'));
                } else {
                    sendMessage($chat_id, "📁 مدیریت مدیا:", getInlineKeyboard('media'));
                }
            }
            break;
            
        case 'upload':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "📤 لطفاً فایل را آپلود کنید.", getInlineKeyboard('back_only', 1, ['back_to' => 'media']));
                    $state_data = json_encode(['upload_message_id' => $callback_message_id]);
                    dbExecute(
                        "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'uploading', ?) 
                         ON DUPLICATE KEY UPDATE state = 'uploading', state_data = ?",
                        [$user_id, $state_data, $state_data]
                    );
                } else {
                    $result = sendMessage($chat_id, "📤 لطفاً فایل را آپلود کنید.", getInlineKeyboard('back_only', 1, ['back_to' => 'media']));
                    if ($result !== false && isset($result['ok']) && $result['ok']) {
                        $sent_message_id = (int)$result['result']['message_id'];
                        $state_data = json_encode(['upload_message_id' => $sent_message_id]);
                        dbExecute(
                            "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'uploading', ?) 
                             ON DUPLICATE KEY UPDATE state = 'uploading', state_data = ?",
                            [$user_id, $state_data, $state_data]
                        );
                    }
                }
            } else {
                sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند آپلود کنند.");
            }
            break;
            
        case 'list_media':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/upload-handler.php";
                require_once "inc/config.php";
                displayMediaList($chat_id, $user_id, 1, $callback_message_id);
            }
            break;
            
        case 'admins':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "👥 مدیریت ادمین‌ها:", getInlineKeyboard('admins'));
                } else {
                    sendMessage($chat_id, "👥 مدیریت ادمین‌ها:", getInlineKeyboard('admins'));
                }
            }
            break;
            
        case 'add_admin':
            if (isMainAdmin($user_id)) {
                $msg = "➕ لطفاً کاربر را forward کنید یا شناسه کاربر را ارسال کنید.";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                $state_data = json_encode(['action' => 'add']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_admin', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_admin', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'remove_admin':
            if (isMainAdmin($user_id)) {
                $msg = "➖ لطفاً کاربر را forward کنید یا شناسه کاربر را ارسال کنید.";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                $state_data = json_encode(['action' => 'remove']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_admin', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_admin', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'list_admins':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    handleListAdmins($chat_id, $callback_message_id);
                } else {
                    handleListAdmins($chat_id);
                }
            }
            break;
            
        case 'stats':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    displayAnalytics($chat_id, $callback_message_id);
                } else {
                    displayAnalytics($chat_id);
                }
            }
            break;
            
        case 'channel_settings':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "🏛 تنظیمات کانال:", getInlineKeyboard('channel_settings'));
                } else {
                    sendMessage($chat_id, "🏛 تنظیمات کانال:", getInlineKeyboard('channel_settings'));
                }
            }
            break;
            
        case 'backup_bot':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/backup-handler.php";
                $msg = "🤖 ربات بک آپ\n\n📝 لطفاً لینک ربات دیگر را ارسال کنید:\n\n💡 مثال: https://t.me/example_bot";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['step' => 'bot_link']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'creating_backup', ?) 
                     ON DUPLICATE KEY UPDATE state = 'creating_backup', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'verify_join':
            require_once "inc/handlers/channel-handler.php";
            
            $user_state = dbQueryOne(
                "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'pending_file_download'",
                [$user_id]
            );
            
            $pending_file_id = null;
            if ($user_state !== false && !empty($user_state['state_data'])) {
                $state_data = json_decode($user_state['state_data'], true);
                $pending_file_id = $state_data['file_id'] ?? null;
            }
            
            if ($pending_file_id !== null && $pending_file_id > 0) {
                $required_channels = getFileForcedJoinChannels($pending_file_id);
            } else {
                $required_channels = getFileForcedJoinChannels(0);
            }
            
            if (empty($required_channels)) {
                sendMessage($chat_id, "✅ هیچ کانال اجباری تنظیم نشده است.");
                break;
            }
            
            $membership = checkChannelMembership($user_id, $required_channels);
            
            if (!$membership['is_member']) {
                checkForcedJoinAndNotify($user_id, $chat_id, $required_channels);
            } else {
                $reaction_channels = getActiveReactionChannels();
                
                if ($reaction_channels === false || empty($reaction_channels)) {
                    if ($pending_file_id !== null && $pending_file_id > 0) {
                        require_once "inc/handlers/point-handler.php";
                        require_once "inc/handlers/user-handler.php";
                        
                        $is_admin = isAdmin($user_id);
                        $is_vip = isUserVIP($user_id);
                        
                        if (!$is_admin && !$is_vip) {
                            if (!hasEnoughPoints($user_id, 1)) {
                                sendMessage($chat_id, "❌ امتیاز شما کافی نیست.\n💡 برای کسب امتیاز از سیستم معرف استفاده کنید یا با Telegram Stars خریداری کنید.");
                                break;
                            }
                            
                            if (!deductUserPoints($user_id, 1)) {
                                sendMessage($chat_id, "❌ خطا در کسر امتیاز.");
                                break;
                            }
                            
                            $remaining_points = getUserPoints($user_id);
                            sendMessage($chat_id, "💎 1 امتیاز کسر شد!\n\n💎 امتیاز باقی‌مانده: {$remaining_points}");
                        }
                        
                        dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'pending_file_download'", [$user_id]);
                        
                        $result = handleFileDownload($user_id, $pending_file_id, $chat_id, true);
                        
                        if ($result['success']) {
                            sendMessage($chat_id, $result['message']);
                        } else {
                            sendMessage($chat_id, $result['message']);
                        }
                    } else {
                        sendMessage($chat_id, "✅ شما می‌توانید ادامه دهید!");
                    }
                } else {
                    showReactionMessage($user_id, $chat_id, $pending_file_id);
                }
            }
            break;
            
        case 'verify_reaction':
            require_once "inc/handlers/channel-handler.php";
            
            $user_state = dbQueryOne(
                "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'reaction_check'",
                [$user_id]
            );
            
            if ($user_state === false || empty($user_state['state_data'])) {
                sendMessage($chat_id, "❌ خطا: اطلاعات یافت نشد. لطفاً دوباره تلاش کنید.");
                break;
            }
            
            $state_data = json_decode($user_state['state_data'], true);
            $reaction_check_time = $state_data['reaction_check_time'] ?? 0;
            $pending_file_id = $state_data['pending_file_id'] ?? null;
            
            if (time() < $reaction_check_time) {
                $remaining = $reaction_check_time - time();
                sendMessage($chat_id, "⏱️ لطفاً {$remaining} ثانیه دیگر صبر کنید.");
                break;
            }
            
            $reaction_channels = getActiveReactionChannels();
            
            if ($reaction_channels === false || empty($reaction_channels)) {
                dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'reaction_check'", [$user_id]);
                dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'pending_file_download'", [$user_id]);
                
                if ($pending_file_id !== null && $pending_file_id > 0) {
                    require_once "inc/handlers/point-handler.php";
                    require_once "inc/handlers/user-handler.php";
                    
                    $is_admin = isAdmin($user_id);
                    $is_vip = isUserVIP($user_id);
                    
                    if (!$is_admin && !$is_vip) {
                        if (!hasEnoughPoints($user_id, 1)) {
                            sendMessage($chat_id, "❌ امتیاز شما کافی نیست.\n💡 برای کسب امتیاز از سیستم معرف استفاده کنید یا با Telegram Stars خریداری کنید.");
                            break;
                        }
                        
                        if (!deductUserPoints($user_id, 1)) {
                            sendMessage($chat_id, "❌ خطا در کسر امتیاز.");
                            break;
                        }
                        
                        $remaining_points = getUserPoints($user_id);
                        sendMessage($chat_id, "💎 1 امتیاز کسر شد!\n\n💎 امتیاز باقی‌مانده: {$remaining_points}");
                    }
                    
                    $result = handleFileDownload($user_id, $pending_file_id, $chat_id, true);
                    
                    if ($result['success']) {
                        sendMessage($chat_id, $result['message']);
                    } else {
                        sendMessage($chat_id, $result['message']);
                    }
                } else {
                    sendMessage($chat_id, "✅ شما می‌توانید ادامه دهید!");
                }
                break;
            }
            
            $reaction_channel_ids = array_column($reaction_channels, 'channel_id');
            $membership = checkChannelMembership($user_id, $reaction_channel_ids);
            
            if (!$membership['is_member']) {
                sendMessage($chat_id, "❌ لطفاً ابتدا در کانال‌های ری‌اکشن عضو شوید.");
                break;
            }
            
            $has_reactions = checkUserReactions($user_id, $reaction_channels, 2);
            
            if (!$has_reactions) {
                sendMessage($chat_id, "❌ لطفاً به 2 پست آخر هر کانال ری‌اکشن بزنید.");
                break;
            }
            
            dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'reaction_check'", [$user_id]);
            dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'pending_file_download'", [$user_id]);
            
            if ($pending_file_id !== null && $pending_file_id > 0) {
                require_once "inc/handlers/point-handler.php";
                require_once "inc/handlers/user-handler.php";
                
                $is_admin = isAdmin($user_id);
                $is_vip = isUserVIP($user_id);
                
                if (!$is_admin && !$is_vip) {
                    if (!hasEnoughPoints($user_id, 1)) {
                        sendMessage($chat_id, "❌ امتیاز شما کافی نیست.\n💡 برای کسب امتیاز از سیستم معرف استفاده کنید یا با Telegram Stars خریداری کنید.");
                        break;
                    }
                    
                    if (!deductUserPoints($user_id, 1)) {
                        sendMessage($chat_id, "❌ خطا در کسر امتیاز.");
                        break;
                    }
                    
                    $remaining_points = getUserPoints($user_id);
                    sendMessage($chat_id, "💎 1 امتیاز کسر شد!\n\n💎 امتیاز باقی‌مانده: {$remaining_points}");
                }
                
                $result = handleFileDownload($user_id, $pending_file_id, $chat_id, true);
                
                if ($result['success']) {
                    sendMessage($chat_id, $result['message']);
                } else {
                    sendMessage($chat_id, $result['message']);
                }
            } else {
                sendMessage($chat_id, "✅ شما می‌توانید ادامه دهید!");
            }
            break;
            
        case 'reaction_channels':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "💬 مدیریت کانال‌های ری‌اکشن:", getInlineKeyboard('reaction_channels'));
                } else {
                    sendMessage($chat_id, "💬 مدیریت کانال‌های ری‌اکشن:", getInlineKeyboard('reaction_channels'));
                }
            }
            break;
            
        case 'add_reaction_channel':
            if (isAdmin($user_id)) {
                $msg = "➕ لطفاً یک پیام از کانال ری‌اکشن را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['action' => 'add']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_reaction_channel', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_reaction_channel', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'remove_reaction_channel':
            if (isAdmin($user_id)) {
                $msg = "🗑️ لطفاً شناسه یا یوزرنیم کانال ری‌اکشن را برای حذف ارسال کنید.";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['action' => 'remove']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_reaction_channel', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_reaction_channel', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'list_reaction_channels':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/channel-handler.php";
                $channels = getActiveReactionChannels();
                
                if ($channels === false || empty($channels)) {
                    $msg = "📑 هیچ کانال ری‌اکشنی تنظیم نشده است.";
                } else {
                    $msg = "📑 لیست کانال‌های ری‌اکشن:\n\n";
                    foreach ($channels as $index => $channel) {
                        $status = $channel['is_active'] ? '✅' : '❌';
                        $link = $channel['channel_link'] ?: ($channel['channel_username'] ? 'https://t.me/' . $channel['channel_username'] : 'بدون لینک');
                        $msg .= ($index + 1) . ". {$status} {$channel['channel_title']}\n";
                        $msg .= "   🔗 {$link}\n\n";
                    }
                }
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, getInlineKeyboard('reaction_channels'));
                } else {
                    sendMessage($chat_id, $msg, getInlineKeyboard('reaction_channels'));
                }
            }
            break;
            
        case 'add_channel':
            if (isAdmin($user_id)) {
                $msg = "➕ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.";
                // Always send new message for input requests
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['action' => 'add']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_channel', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_channel', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'remove_channel':
            if (isAdmin($user_id)) {
                $msg = "➖ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.";
                // Always send new message for input requests
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['action' => 'remove']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_channel', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_channel', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'list_channels':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    handleListChannels($chat_id, $callback_message_id);
                } else {
                    handleListChannels($chat_id);
                }
            }
            break;
            
        case 'start':
            // Handle start callback - show appropriate menu based on user type
            if (isAdmin($user_id)) {
                // Check if admin is in user mode
                if (isRealAdmin($user_id) && isInUserMode($user_id)) {
                    // Check forced join for admin in user mode
                    require_once "inc/handlers/channel-handler.php";
                    if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                        break; // Message already sent
                    }
                    
                    // Admin in user mode - show user menu with exit button
                    $keyboard = getInlineKeyboard('user_menu', 1, ['user_id' => $user_id]);
                    if ($callback_message_id) {
                        editMessage($chat_id, $callback_message_id, "🏠 منوی اصلی:", $keyboard);
                    } else {
                        sendMessage($chat_id, "🏠 منوی اصلی:", $keyboard);
                    }
                } else {
                    // Normal admin menu
                    if ($callback_message_id) {
                        editMessage($chat_id, $callback_message_id, "🏠 منوی اصلی:", getInlineKeyboard('main', 1));
                    } else {
                        sendMessage($chat_id, "🏠 منوی اصلی:", getInlineKeyboard('main', 1));
                    }
                }
            } else {
                // Check forced join for regular users
                require_once "inc/handlers/channel-handler.php";
                if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                    break; // Message already sent
                }
                
                // Regular users get user menu (or admin in user mode)
                $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                    ? getInlineKeyboard('user_menu', 1, ['user_id' => $user_id])
                    : getInlineKeyboard('user_menu');
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "🏠 منوی اصلی:", $keyboard);
                } else {
                    sendMessage($chat_id, "🏠 منوی اصلی:", $keyboard);
                }
            }
            break;
            
        case 'back_main':
            // Check if admin is in user mode
            if (isRealAdmin($user_id) && isInUserMode($user_id)) {
                // Check forced join for admin in user mode
                require_once "inc/handlers/channel-handler.php";
                if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                    break; // Message already sent
                }
                
                // Admin in user mode - show user menu with exit button
                $keyboard = getInlineKeyboard('user_menu', 1, ['user_id' => $user_id]);
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "🏠 منوی اصلی:", $keyboard);
                } else {
                    sendMessage($chat_id, "🏠 منوی اصلی:", $keyboard);
                }
                // Clear user state when returning to main menu (except user_mode)
                dbExecute("DELETE FROM user_states WHERE user_id = ? AND state != 'user_mode'", [$user_id]);
            } elseif (isAdmin($user_id)) {
                // Normal admin menu
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "🏠 منوی اصلی:", getInlineKeyboard('main', 1));
                } else {
                    sendMessage($chat_id, "🏠 منوی اصلی:", getInlineKeyboard('main', 1));
                }
                // Clear user state when returning to main menu (except user_mode)
                dbExecute("DELETE FROM user_states WHERE user_id = ? AND state != 'user_mode'", [$user_id]);
            } else {
                // Check forced join for regular users
                require_once "inc/handlers/channel-handler.php";
                if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                    break; // Message already sent
                }
                
                // Regular users get user menu
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "🏠 منوی اصلی:", getInlineKeyboard('user_menu'));
                } else {
                    sendMessage($chat_id, "🏠 منوی اصلی:", getInlineKeyboard('user_menu'));
                }
            }
            break;
            
        case 'toggle_user_mode':
            if (isRealAdmin($user_id)) {
                require_once "inc/handlers/point-handler.php";
                $current_mode = isInUserMode($user_id);
                $new_mode = !$current_mode;
                
                if ($new_mode) {
                    // Enable user mode
                    $state_data = json_encode(['enabled' => true]);
                    $result = dbExecute(
                        "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'user_mode', ?) 
                         ON DUPLICATE KEY UPDATE state = 'user_mode', state_data = ?",
                        [$user_id, $state_data, $state_data]
                    );
                    
                    if ($result !== false) {
                        $msg = "✅ حالت کاربر فعال شد!\n\n👤 شما اکنون به عنوان کاربر عادی عمل می‌کنید.\n\n💎 امتیاز شما: " . getUserPoints($user_id);
                        // Force user_menu_with_exit since we just enabled user mode
                        $keyboard = getInlineKeyboard('user_menu_with_exit', 1);
                    } else {
                        $msg = "❌ خطا در فعال‌سازی حالت کاربر.";
                        $keyboard = getInlineKeyboard('main', 1);
                    }
                } else {
                    // Disable user mode
                    $result = dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'user_mode'", [$user_id]);
                    
                    if ($result !== false) {
                        $msg = "✅ حالت کاربر غیرفعال شد!\n\n👨‍💼 شما اکنون به عنوان ادمین عمل می‌کنید.";
                        $keyboard = getInlineKeyboard('main', 1);
                    } else {
                        $msg = "❌ خطا در غیرفعال‌سازی حالت کاربر.";
                        $keyboard = getInlineKeyboard('main', 1);
                    }
                }
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
            }
            break;
            
        case 'download_file':
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            // User wants to download a file - show file list or handle download
            $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                ? getInlineKeyboard('user_download_menu', 1, ['user_id' => $user_id])
                : getInlineKeyboard('user_download_menu');
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, "📥 برای دانلود فایل، لطفاً شناسه فایل را ارسال کنید یا از منوی فایل‌ها استفاده کنید.", $keyboard);
            } else {
                sendMessage($chat_id, "📥 برای دانلود فایل، لطفاً شناسه فایل را ارسال کنید یا از منوی فایل‌ها استفاده کنید.", $keyboard);
            }
            break;
            
        case 'my_downloads':
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            // Get user's downloaded files
            require_once "inc/handlers/upload-handler.php";
            $downloads = dbQuery(
                "SELECT d.*, f.file_type, f.file_size, f.created_at as file_created_at 
                 FROM downloads d 
                 INNER JOIN files f ON d.file_id = f.id 
                 WHERE d.user_id = ? 
                 ORDER BY d.created_at DESC 
                 LIMIT 50",
                [$user_id]
            );
            
            if ($downloads === false || empty($downloads)) {
                $msg = "📥 شما هنوز هیچ فایلی دانلود نکرده‌اید.";
                $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                    ? getInlineKeyboard('user_menu', 1, ['user_id' => $user_id])
                    : getInlineKeyboard('user_menu');
            } else {
                $msg = "📥 فایل‌های دانلود شده شما:\n\n";
                $msg .= "💡 روی هر فایل کلیک کنید تا دوباره دانلود کنید.\n\n";
                
                // Build inline keyboard with glass buttons for each downloaded file
                $keyboard_buttons = [];
                
                $file_type_emoji = [
                    'photo' => '🖼️',
                    'video' => '🎥',
                    'document' => '📄',
                    'audio' => '🎵',
                    'voice' => '🎤',
                    'video_note' => '📹',
                    'sticker' => '😀',
                    'animation' => '🎬'
                ];
                
                foreach ($downloads as $index => $download) {
                    $emoji = $file_type_emoji[$download['file_type']] ?? '📁';
                    $file_id = (int)$download['file_id'];
                    
                    $msg .= ($index + 1) . ". {$emoji} فایل #{$file_id}\n";
                    $msg .= "   📅 " . date('Y-m-d H:i', strtotime($download['created_at'])) . "\n\n";
                    
                    // Add glass button for this file
                    $keyboard_buttons[] = [
                        [
                            'text' => "{$emoji} فایل #{$file_id}",
                            'callback_data' => 'download_' . $file_id
                        ]
                    ];
                }
                
                // Add back button
                $keyboard_buttons[] = [
                    ['text' => '🏠 منوی اصلی', 'callback_data' => 'start']
                ];
                
                $keyboard = [
                    'inline_keyboard' => $keyboard_buttons
                ];
            }
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, $msg, $keyboard);
            } else {
                sendMessage($chat_id, $msg, $keyboard);
            }
            break;
            
        case 'my_points':
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            if ($callback_message_id) {
                displayUserPoints($chat_id, $user_id, $callback_message_id);
            } else {
                displayUserPoints($chat_id, $user_id);
            }
            break;
            
        case 'my_referral':
            // Check if referral system is enabled
            require_once "inc/handlers/settings-handler.php";
            $referral_enabled = (bool)getSettingValue('referral_enabled', true);
            
            if (!$referral_enabled) {
                $msg = "❌ سیستم معرف غیرفعال است.";
                $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                    ? getInlineKeyboard('user_menu', 1, ['user_id' => $user_id])
                    : getInlineKeyboard('user_menu');
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
                break;
            }
            
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            if ($callback_message_id) {
                displayReferralInfo($chat_id, $user_id, $callback_message_id);
            } else {
                displayReferralInfo($chat_id, $user_id);
            }
            break;
            
        case 'file_sorting_menu':
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            require_once "inc/handlers/upload-handler.php";
            displayFileList($chat_id, $user_id, 'date', 1, $callback_message_id);
            break;
            
        case 'recommended_files':
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            require_once "inc/handlers/upload-handler.php";
            $recommended = getRecommendedFiles($user_id, 20);
            
            if ($recommended === false || empty($recommended)) {
                $msg = "⭐ هیچ فایل پیشنهادی یافت نشد.";
                $keyboard = getInlineKeyboard('file_sorting_menu');
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
                break;
            }
            
            $msg = "⭐ فایل‌های پیشنهادی برای شما:\n\n";
            $msg .= "💡 روی هر فایل کلیک کنید تا دانلود کنید.\n\n";
            
            $keyboard_buttons = [];
            
            foreach ($recommended as $index => $file) {
                $file_type_emoji = [
                    'photo' => '🖼️',
                    'video' => '🎥',
                    'document' => '📄',
                    'audio' => '🎵',
                    'voice' => '🎤',
                    'video_note' => '📹',
                    'sticker' => '😀',
                    'animation' => '🎬'
                ];
                $emoji = $file_type_emoji[$file['file_type']] ?? '📁';
                
                $msg .= ($index + 1) . ". {$emoji} فایل #{$file['id']}\n";
                $msg .= "   📊 دانلود: {$file['download_count']}\n";
                $popularity = number_format((float)($file['popularity_score'] ?? 0), 2);
                $msg .= "   ⭐ محبوبیت: {$popularity}\n\n";
                
                $keyboard_buttons[] = [
                    [
                        'text' => "{$emoji} فایل #{$file['id']}",
                        'callback_data' => 'download_file_' . $file['id']
                    ]
                ];
            }
            
            $keyboard_buttons[] = [
                ['text' => '⬅️ بازگشت', 'callback_data' => 'download_file']
            ];
            
            $keyboard = [
                'inline_keyboard' => $keyboard_buttons
            ];
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, $msg, $keyboard);
            } else {
                sendMessage($chat_id, $msg, $keyboard);
            }
            break;
            
        case 'gambling':
            // Check if gambling is enabled
            require_once "inc/handlers/gambling-handler.php";
            if (!checkGamblingEnabled()) {
                $msg = "❌ سیستم بازی و شرط‌بندی غیرفعال است.";
                $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                    ? getInlineKeyboard('user_menu', 1, ['user_id' => $user_id])
                    : getInlineKeyboard('user_menu');
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
                break;
            }
            
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            $msg = "🎲 بازی و شرط‌بندی:\n\n";
            $msg .= "💡 یک بازی انتخاب کنید و شرط ببندید!\n\n";
            
            require_once "inc/handlers/gambling-handler.php";
            require_once "inc/handlers/settings-handler.php";
            $min_bet = (int)getSettingValue('gambling_min_bet', 1);
            $max_bet = (int)getSettingValue('gambling_max_bet', 100);
            $multiplier = (float)getSettingValue('gambling_win_multiplier', 2.0);
            
            $msg .= "💰 حداقل شرط: {$min_bet} امتیاز\n";
            $msg .= "💰 حداکثر شرط: {$max_bet} امتیاز\n";
            $msg .= "🎯 نرخ برد: {$multiplier}x\n\n";
            $msg .= "💎 امتیاز شما: " . getUserPoints($user_id);
            
            $keyboard = getInlineKeyboard('gambling_menu');
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, $msg, $keyboard);
            } else {
                sendMessage($chat_id, $msg, $keyboard);
            }
            break;
            
        case 'gambling_stats':
            require_once "inc/handlers/gambling-handler.php";
            displayGamblingStats($chat_id, $user_id, $callback_message_id);
            break;
            
        case 'gambling_number_guess':
            // Check if gambling is enabled
            require_once "inc/handlers/gambling-handler.php";
            if (!checkGamblingEnabled()) {
                sendMessage($chat_id, "❌ سیستم بازی و شرط‌بندی غیرفعال است.");
                break;
            }
            
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            // Check daily limit
            $daily_check = checkDailyLimit($user_id);
            if (!$daily_check['valid']) {
                sendMessage($chat_id, $daily_check['message']);
                break;
            }
            
            // Check cooldown
            $cooldown_check = checkCooldown($user_id);
            if (!$cooldown_check['valid']) {
                sendMessage($chat_id, $cooldown_check['message']);
                break;
            }
            
            $msg = "🎯 بازی حدس عدد:\n\n";
            $msg .= "💡 یک مبلغ شرط انتخاب کنید:\n\n";
            $msg .= "💎 امتیاز شما: " . getUserPoints($user_id);
            
            $keyboard = getInlineKeyboard('gambling_bet_menu');
            
            // Save state for number guess game
            $state_data = json_encode(['game_type' => 'number_guess']);
            dbExecute(
                "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'gambling_bet', ?) 
                 ON DUPLICATE KEY UPDATE state = 'gambling_bet', state_data = ?",
                [$user_id, $state_data, $state_data]
            );
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, $msg, $keyboard);
            } else {
                sendMessage($chat_id, $msg, $keyboard);
            }
            break;
            
        case 'gambling_coin_flip':
            // Check if gambling is enabled
            require_once "inc/handlers/gambling-handler.php";
            if (!checkGamblingEnabled()) {
                sendMessage($chat_id, "❌ سیستم بازی و شرط‌بندی غیرفعال است.");
                break;
            }
            
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            // Check daily limit
            $daily_check = checkDailyLimit($user_id);
            if (!$daily_check['valid']) {
                sendMessage($chat_id, $daily_check['message']);
                break;
            }
            
            // Check cooldown
            $cooldown_check = checkCooldown($user_id);
            if (!$cooldown_check['valid']) {
                sendMessage($chat_id, $cooldown_check['message']);
                break;
            }
            
            $msg = "🪙 بازی سکه:\n\n";
            $msg .= "💡 یک مبلغ شرط انتخاب کنید:\n\n";
            $msg .= "💎 امتیاز شما: " . getUserPoints($user_id);
            
            $keyboard = getInlineKeyboard('gambling_bet_menu');
            
            // Save state for coin flip game
            $state_data = json_encode(['game_type' => 'coin_flip']);
            dbExecute(
                "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'gambling_bet', ?) 
                 ON DUPLICATE KEY UPDATE state = 'gambling_bet', state_data = ?",
                [$user_id, $state_data, $state_data]
            );
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, $msg, $keyboard);
            } else {
                sendMessage($chat_id, $msg, $keyboard);
            }
            break;
            
        case 'gambling_dice_roll':
            // Check if gambling is enabled
            require_once "inc/handlers/gambling-handler.php";
            if (!checkGamblingEnabled()) {
                sendMessage($chat_id, "❌ سیستم بازی و شرط‌بندی غیرفعال است.");
                break;
            }
            
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            // Check daily limit
            $daily_check = checkDailyLimit($user_id);
            if (!$daily_check['valid']) {
                sendMessage($chat_id, $daily_check['message']);
                break;
            }
            
            // Check cooldown
            $cooldown_check = checkCooldown($user_id);
            if (!$cooldown_check['valid']) {
                sendMessage($chat_id, $cooldown_check['message']);
                break;
            }
            
            $msg = "🎲 بازی تاس:\n\n";
            $msg .= "💡 یک مبلغ شرط انتخاب کنید:\n\n";
            $msg .= "💎 امتیاز شما: " . getUserPoints($user_id);
            
            $keyboard = getInlineKeyboard('gambling_bet_menu');
            
            // Save state for dice roll game
            $state_data = json_encode(['game_type' => 'dice_roll']);
            dbExecute(
                "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'gambling_bet', ?) 
                 ON DUPLICATE KEY UPDATE state = 'gambling_bet', state_data = ?",
                [$user_id, $state_data, $state_data]
            );
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, $msg, $keyboard);
            } else {
                sendMessage($chat_id, $msg, $keyboard);
            }
            break;
            
        default:
            // Handle sorting callbacks: sort_files_{sort_by}_page_{page}
            if (preg_match('/^sort_files_(popular|date|downloads)_page_(\d+)$/', $callback_data, $matches)) {
                $sort_by = $matches[1];
                $page = (int)$matches[2];
                
                // Check forced join first
                require_once "inc/handlers/channel-handler.php";
                if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                    break; // Message already sent
                }
                
                require_once "inc/handlers/upload-handler.php";
                displayFileList($chat_id, $user_id, $sort_by, $page, $callback_message_id);
                break;
            }
            
            // Handle download_file_{file_id} callback
            if (preg_match('/^download_file_(\d+)$/', $callback_data, $matches)) {
                $file_id = (int)$matches[1];
                
                // Check forced join first
                require_once "inc/handlers/channel-handler.php";
                if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                    break; // Message already sent
                }
                
                require_once "inc/handlers/point-handler.php";
                require_once "inc/handlers/user-handler.php";
                
                $is_admin = isAdmin($user_id);
                $is_vip = isUserVIP($user_id);
                
                // Check if user has enough points (only for non-admin, non-VIP users)
                if (!$is_admin && !$is_vip) {
                    if (!hasEnoughPoints($user_id, 1)) {
                        sendMessage($chat_id, "❌ امتیاز شما کافی نیست.\n💡 برای کسب امتیاز از سیستم معرف استفاده کنید یا با Telegram Stars خریداری کنید.");
                        break;
                    }
                    
                    // Deduct 1 point
                    if (!deductUserPoints($user_id, 1)) {
                        sendMessage($chat_id, "❌ خطا در کسر امتیاز.");
                        break;
                    }
                    
                    // Show deduction message
                    $remaining_points = getUserPoints($user_id);
                    sendMessage($chat_id, "💎 1 امتیاز کسر شد!\n\n💎 امتیاز باقی‌مانده: {$remaining_points}");
                }
                
                // Download the file (skip forced join check as already verified)
                $result = handleFileDownload($user_id, $file_id, $chat_id, true);
                
                if ($result['success']) {
                    sendMessage($chat_id, $result['message']);
                } else {
                    sendMessage($chat_id, $result['message']);
                }
                break;
            }
            
            // Handle gambling_bet_{amount} callback
            if (preg_match('/^gambling_bet_(\d+)$/', $callback_data, $matches)) {
                $bet_amount = (int)$matches[1];
                
                // Check if gambling is enabled
                require_once "inc/handlers/gambling-handler.php";
                if (!checkGamblingEnabled()) {
                    sendMessage($chat_id, "❌ سیستم بازی و شرط‌بندی غیرفعال است.");
                    break;
                }
                
                // Check forced join first
                require_once "inc/handlers/channel-handler.php";
                if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                    break; // Message already sent
                }
                
                // Check bet limits
                $bet_check = checkBetLimits($bet_amount);
                if (!$bet_check['valid']) {
                    sendMessage($chat_id, $bet_check['message']);
                    break;
                }
                
                // Get game type from state
                $user_state = dbQueryOne(
                    "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'gambling_bet'",
                    [$user_id]
                );
                
                if ($user_state === false || empty($user_state['state_data'])) {
                    sendMessage($chat_id, "❌ خطا در دریافت اطلاعات بازی. لطفاً دوباره شروع کنید.");
                    break;
                }
                
                $state_data = json_decode($user_state['state_data'], true);
                $game_type = $state_data['game_type'] ?? '';
                
                // Update state with bet amount
                $state_data['bet_amount'] = $bet_amount;
                dbExecute(
                    "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                    [json_encode($state_data), $user_id]
                );
                
                // Show appropriate menu based on game type
                if ($game_type === 'number_guess') {
                    $msg = "🎯 بازی حدس عدد:\n\n";
                    $msg .= "💰 شرط شما: {$bet_amount} امتیاز\n";
                    $msg .= "💡 یک عدد بین 1 تا 10 انتخاب کنید:";
                    $keyboard = getInlineKeyboard('gambling_number_menu');
                } elseif ($game_type === 'coin_flip') {
                    $msg = "🪙 بازی سکه:\n\n";
                    $msg .= "💰 شرط شما: {$bet_amount} امتیاز\n";
                    $msg .= "💡 شیر یا خط انتخاب کنید:";
                    $keyboard = getInlineKeyboard('gambling_coin_menu');
                } elseif ($game_type === 'dice_roll') {
                    $msg = "🎲 بازی تاس:\n\n";
                    $msg .= "💰 شرط شما: {$bet_amount} امتیاز\n";
                    $msg .= "💡 یک عدد بین 1 تا 6 انتخاب کنید:";
                    $keyboard = getInlineKeyboard('gambling_dice_menu');
                } else {
                    sendMessage($chat_id, "❌ نوع بازی نامعتبر است.");
                    break;
                }
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
                break;
            }
            
            // Handle gambling_number_{number} callback
            if (preg_match('/^gambling_number_(\d+)$/', $callback_data, $matches)) {
                $guessed_number = (int)$matches[1];
                
                // Check if gambling is enabled
                require_once "inc/handlers/gambling-handler.php";
                if (!checkGamblingEnabled()) {
                    sendMessage($chat_id, "❌ سیستم بازی و شرط‌بندی غیرفعال است.");
                    break;
                }
                
                // Get game state
                $user_state = dbQueryOne(
                    "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'gambling_bet'",
                    [$user_id]
                );
                
                if ($user_state === false || empty($user_state['state_data'])) {
                    sendMessage($chat_id, "❌ خطا در دریافت اطلاعات بازی. لطفاً دوباره شروع کنید.");
                    break;
                }
                
                $state_data = json_decode($user_state['state_data'], true);
                $bet_amount = (int)($state_data['bet_amount'] ?? 0);
                
                if ($bet_amount <= 0) {
                    sendMessage($chat_id, "❌ مبلغ شرط نامعتبر است.");
                    break;
                }
                
                // Play game
                $result = playNumberGuess($user_id, $bet_amount, $guessed_number);
                
                // Clear state
                dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'gambling_bet'", [$user_id]);
                
                // Show result
                $keyboard = getInlineKeyboard('gambling_menu');
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $result['message'], $keyboard);
                } else {
                    sendMessage($chat_id, $result['message'], $keyboard);
                }
                break;
            }
            
            // Handle gambling_dice_{number} callback
            if (preg_match('/^gambling_dice_(\d+)$/', $callback_data, $matches)) {
                $guessed_number = (int)$matches[1];
                
                // Check if gambling is enabled
                require_once "inc/handlers/gambling-handler.php";
                if (!checkGamblingEnabled()) {
                    sendMessage($chat_id, "❌ سیستم بازی و شرط‌بندی غیرفعال است.");
                    break;
                }
                
                // Get game state
                $user_state = dbQueryOne(
                    "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'gambling_bet'",
                    [$user_id]
                );
                
                if ($user_state === false || empty($user_state['state_data'])) {
                    sendMessage($chat_id, "❌ خطا در دریافت اطلاعات بازی. لطفاً دوباره شروع کنید.");
                    break;
                }
                
                $state_data = json_decode($user_state['state_data'], true);
                $bet_amount = (int)($state_data['bet_amount'] ?? 0);
                
                if ($bet_amount <= 0) {
                    sendMessage($chat_id, "❌ مبلغ شرط نامعتبر است.");
                    break;
                }
                
                // Play game
                $result = playDiceRoll($user_id, $bet_amount, $guessed_number);
                
                // Clear state
                dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'gambling_bet'", [$user_id]);
                
                // Show result
                $keyboard = getInlineKeyboard('gambling_menu');
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $result['message'], $keyboard);
                } else {
                    sendMessage($chat_id, $result['message'], $keyboard);
                }
                break;
            }
            
            // Handle gambling_coin_{choice} callback
            if (preg_match('/^gambling_coin_(heads|tails)$/', $callback_data, $matches)) {
                $choice = $matches[1];
                
                // Check if gambling is enabled
                require_once "inc/handlers/gambling-handler.php";
                if (!checkGamblingEnabled()) {
                    sendMessage($chat_id, "❌ سیستم بازی و شرط‌بندی غیرفعال است.");
                    break;
                }
                
                // Get game state
                $user_state = dbQueryOne(
                    "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'gambling_bet'",
                    [$user_id]
                );
                
                if ($user_state === false || empty($user_state['state_data'])) {
                    sendMessage($chat_id, "❌ خطا در دریافت اطلاعات بازی. لطفاً دوباره شروع کنید.");
                    break;
                }
                
                $state_data = json_decode($user_state['state_data'], true);
                $bet_amount = (int)($state_data['bet_amount'] ?? 0);
                
                if ($bet_amount <= 0) {
                    sendMessage($chat_id, "❌ مبلغ شرط نامعتبر است.");
                    break;
                }
                
                // Play game
                $result = playCoinFlip($user_id, $bet_amount, $choice);
                
                // Clear state
                dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'gambling_bet'", [$user_id]);
                
                // Show result
                $keyboard = getInlineKeyboard('gambling_menu');
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $result['message'], $keyboard);
                } else {
                    sendMessage($chat_id, $result['message'], $keyboard);
                }
                break;
            }
            
            // Handle file-specific auto-delete callback (format: file_autodelete_<file_id>)
            if (strpos($callback_data, 'file_autodelete_') === 0) {
                if (isAdmin($user_id)) {
                    $file_id = (int)str_replace('file_autodelete_', '', $callback_data);
                    $msg = "⏰ تنظیم تایم حذف خودکار برای فایل #{$file_id}\n\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:";
                    $keyboard = [
                        'inline_keyboard' => [
                            [
                                ['text' => '⏱️ 5s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_5'],
                                ['text' => '⏱️ 10s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_10']
                            ],
                            [
                                ['text' => '⏱️ 15s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_15'],
                                ['text' => '⏱️ 20s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_20']
                            ],
                            [
                                ['text' => '⏱️ 25s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_25'],
                                ['text' => '⏱️ 30s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_30']
                            ],
                            [
                                ['text' => '⏱️ 35s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_35'],
                                ['text' => '⏱️ 40s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_40']
                            ],
                            [
                                ['text' => '⏱️ 45s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_45'],
                                ['text' => '⏱️ 50s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_50']
                            ],
                            [
                                ['text' => '⏱️ 55s', 'callback_data' => 'set_file_autodelete_' . $file_id . '_55'],
                                ['text' => '⏱️ 1m', 'callback_data' => 'set_file_autodelete_' . $file_id . '_60']
                            ],
                            [
                                ['text' => '🔢 عدد دلخواه', 'callback_data' => 'set_file_autodelete_custom_' . $file_id]
                            ],
                            [
                                ['text' => '❌ غیرفعال', 'callback_data' => 'set_file_autodelete_' . $file_id . '_0']
                            ],
                            [
                                ['text' => '⬅️ بازگشت', 'callback_data' => 'back_main']
                            ]
                        ]
                    ];
                    if ($callback_message_id) {
                        editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                    } else {
                        sendMessage($chat_id, $msg, $keyboard);
                    }
                }
                break;
            }
            
            // Handle set file auto-delete (format: set_file_autodelete_<file_id>_<seconds>)
            if (strpos($callback_data, 'set_file_autodelete_') === 0) {
                if (isAdmin($user_id)) {
                    require_once "inc/handlers/auto-delete-handler.php";
                    $parts = explode('_', $callback_data);
                    
                    if (strpos($callback_data, '_custom_') !== false) {
                        // Custom seconds
                        $file_id = (int)end($parts);
                        $msg = "🔢 لطفاً تعداد ثانیه را ارسال کنید:\n\nمثال: 200";
                        sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                        // Save state
                        $state_data = json_encode(['type' => 'file_autodelete_custom', 'file_id' => $file_id]);
                        dbExecute(
                            "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_autodelete', ?) 
                             ON DUPLICATE KEY UPDATE state = 'setting_autodelete', state_data = ?",
                            [$user_id, $state_data, $state_data]
                        );
                    } else {
                        // Fixed seconds
                        $file_id = (int)$parts[count($parts) - 2];
                        $seconds = (int)end($parts);
                        
                        if ($seconds === 0) {
                            // Disable auto-delete
                            $result = setFileAutoDelete($file_id, 0);
                            $msg = $result ? "✅ تایم حذف خودکار برای فایل #{$file_id} غیرفعال شد." : "❌ خطا در غیرفعال کردن تایم حذف.";
                        } else {
                            $result = setFileAutoDelete($file_id, $seconds);
                            $time_str = $seconds . " ثانیه";
                            $msg = $result ? "✅ تایم حذف خودکار برای فایل #{$file_id} روی {$time_str} تنظیم شد." : "❌ خطا در تنظیم تایم حذف.";
                        }
                        
                        if ($callback_message_id) {
                            editMessage($chat_id, $callback_message_id, $msg, getInlineKeyboard('back_only'));
                        } else {
                            sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                        }
                    }
                }
                break;
            }
            
            // Handle list_media_page callback (format: list_media_page_<page>)
            if (strpos($callback_data, 'list_media_page_') === 0) {
                if (isAdmin($user_id)) {
                    require_once "inc/handlers/media-handler.php";
                    $page = (int)str_replace('list_media_page_', '', $callback_data);
                    if ($page < 1) {
                        $page = 1;
                    }
                    displayMediaList($chat_id, $user_id, $page, $callback_message_id);
                }
                break;
            }
            
            // Handle media info callback (format: media_info_<file_id>)
            if (strpos($callback_data, 'media_info_') === 0) {
                if (isAdmin($user_id)) {
                    require_once "inc/handlers/upload-handler.php";
                    require_once "inc/handlers/media-handler.php";
                    require_once "inc/config.php";
                    
                    $file_id = (int)str_replace('media_info_', '', $callback_data);
                    
                    if ($file_id <= 0) {
                        sendMessage($chat_id, "❌ شناسه فایل نامعتبر است.");
                        break;
                    }
                    
                    $file = getFileById($file_id);
                    
                    if ($file === false) {
                        sendMessage($chat_id, "❌ فایل یافت نشد.");
                        break;
                    }
                    
                    // Build start link using file_code (short and clean like old version)
                    $file_code = $file['file_code'] ?? '';
                    if (empty($file_code)) {
                        sendMessage($chat_id, "❌ فایل کد یافت نشد.");
                        break;
                    }
                    $start_link = "https://t.me/" . BOT_USERNAME . "?start={$file_code}";
                    
                    // Get file type emoji
                    $file_type_emoji = [
                        'photo' => '🖼️',
                        'video' => '🎥',
                        'document' => '📄',
                        'audio' => '🎵',
                        'voice' => '🎤',
                        'video_note' => '📹',
                        'sticker' => '😀',
                        'animation' => '🎬'
                    ];
                    $emoji = $file_type_emoji[$file['file_type']] ?? '📁';
                    
                    // Build message with file info
                    $msg = "📋 اطلاعات فایل\n\n";
                    $msg .= "{$emoji} نوع: {$file['file_type']}\n";
                    $msg .= "🆔 شناسه: #{$file['id']}\n";
                    $msg .= "📊 تعداد دانلود: {$file['download_count']}\n";
                    
                    if (!empty($file['file_size'])) {
                        $msg .= "📦 حجم: " . formatFileSize((int)$file['file_size']) . "\n";
                    }
                    
                    if (!empty($file['auto_delete_at'])) {
                        $msg .= "⏱️ حذف خودکار: {$file['auto_delete_at']} ثانیه\n";
                    } else {
                        $msg .= "⏱️ حذف خودکار: غیرفعال\n";
                    }
                    
                    if (!empty($file['created_at'])) {
                        $msg .= "📅 تاریخ ایجاد: {$file['created_at']}\n";
                    }
                    
                    // Get forced join channels if any
                    if (!empty($file['forced_join_channels'])) {
                        $forced_channels = json_decode($file['forced_join_channels'], true);
                        if (is_array($forced_channels) && !empty($forced_channels)) {
                            require_once "inc/handlers/channel-handler.php";
                            $placeholders = implode(',', array_fill(0, count($forced_channels), '?'));
                            $channels = dbQuery(
                                "SELECT channel_title FROM channels WHERE channel_id IN ({$placeholders})",
                                $forced_channels
                            );
                            if ($channels !== false && !empty($channels)) {
                                $msg .= "\n📢 کانال‌های جوین اجباری:\n";
                                foreach ($channels as $channel) {
                                    $msg .= "   • {$channel['channel_title']}\n";
                                }
                            }
                        }
                    }
                    
                    $msg .= "\n🔗 لینک استارت:\n";
                    $msg .= "{$start_link}";
                    
                    // Build keyboard with start link button and auto-delete settings
                    $keyboard = [
                        'inline_keyboard' => [
                            [
                                [
                                    'text' => '🔗 باز کردن لینک استارت',
                                    'url' => $start_link
                                ]
                            ],
                            [
                                ['text' => '⏱️ تنظیم تایم حذف', 'callback_data' => 'file_autodelete_' . $file_id]
                            ],
                            [
                                ['text' => '⬅️ بازگشت به لیست', 'callback_data' => 'list_media']
                            ]
                        ]
                    ];
                    
                    if ($callback_message_id) {
                        editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                    } else {
                        sendMessage($chat_id, $msg, $keyboard);
                    }
                }
                break;
            }
            
            if (strpos($callback_data, 'main_page_') === 0) {
                // Only admins can access main menu pagination
                if (!isAdmin($user_id)) {
                    break;
                }
                
                $page_str = str_replace('main_page_', '', $callback_data);
                $page = (int)$page_str;
                if ($page > 0) {
                    if ($callback_message_id) {
                        editMessage($chat_id, $callback_message_id, "🏠 منوی اصلی:", getInlineKeyboard('main', $page));
                    } else {
                        sendMessage($chat_id, "🏠 منوی اصلی:", getInlineKeyboard('main', $page));
                    }
                }
                break;
            }
            
            // Check if callback is for downloading a specific file (format: download_<file_id>)
            if (strpos($callback_data, 'download_') === 0) {
                $file_id = (int)str_replace('download_', '', $callback_data);
                
                // Check forced join channels first (file-specific or global)
                require_once "inc/handlers/channel-handler.php";
                $required_channels = getFileForcedJoinChannels($file_id);
                
                if (!empty($required_channels)) {
                    // Use checkForcedJoinAndNotify for consistent glass keyboard
                    if (!checkForcedJoinAndNotify($user_id, $chat_id, $required_channels)) {
                        exit;
                    }
                }
                
                // Handle download
                $result = handleFileDownload($user_id, $file_id, $chat_id);
                
                if ($result['success']) {
                    sendMessage($chat_id, $result['message']);
                } else {
                    sendMessage($chat_id, $result['message']);
                }
            }
            break;
            
        case 'daily_spin':
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            require_once "inc/handlers/daily-spin-handler.php";
            $result = performDailySpin($user_id, $chat_id);
            
            // Use appropriate keyboard for spin result
            $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                ? getInlineKeyboard('user_spin_menu', 1, ['user_id' => $user_id])
                : getInlineKeyboard('user_spin_menu');
            
            if ($result['success']) {
                // Show result message
                sendMessage($chat_id, $result['message'], $keyboard);
            } else {
                // Show error (already spun today)
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $result['message'], $keyboard);
                } else {
                    sendMessage($chat_id, $result['message'], $keyboard);
                }
            }
            break;
            
        case 'use_promo_code':
            // Check if promo codes are enabled
            require_once "inc/handlers/settings-handler.php";
            $promo_codes_enabled = (bool)getSettingValue('promo_codes_enabled', true);
            
            if (!$promo_codes_enabled) {
                $msg = "❌ کدهای جایزه غیرفعال است.";
                $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                    ? getInlineKeyboard('user_menu', 1, ['user_id' => $user_id])
                    : getInlineKeyboard('user_menu');
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
                break;
            }
            
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            // User wants to enter promo code
            $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                ? getInlineKeyboard('user_promo_code_menu', 1, ['user_id' => $user_id])
                : getInlineKeyboard('user_promo_code_menu');
            
            $msg = "🎫 لطفاً کد جایزه/تخفیف/سکه خود را ارسال کنید:\n\n💡 کد را به صورت متنی ارسال کنید (مثلاً: GIFT100)";
            
            if ($callback_message_id) {
                editMessage($chat_id, $callback_message_id, $msg, $keyboard);
            } else {
                sendMessage($chat_id, $msg, $keyboard);
            }
            break;
            
        case 'delete_time':
            if (isAdmin($user_id)) {
                $msg = "⏰ تنظیم تایم حذف خودکار\n\n🔹 برای فایل خاص: به پیام فایل reply کنید و دکمه را بزنید\n🔹 برای تمام فایل‌ها (Global): یکی از دکمه‌های زیر را بزنید:";
                $keyboard = [
                    'inline_keyboard' => [
                        [
                            ['text' => '⏱️ 5s', 'callback_data' => 'global_autodelete_5s'],
                            ['text' => '⏱️ 10s', 'callback_data' => 'global_autodelete_10s']
                        ],
                        [
                            ['text' => '⏱️ 15s', 'callback_data' => 'global_autodelete_15s'],
                            ['text' => '⏱️ 20s', 'callback_data' => 'global_autodelete_20s']
                        ],
                        [
                            ['text' => '⏱️ 25s', 'callback_data' => 'global_autodelete_25s'],
                            ['text' => '⏱️ 30s', 'callback_data' => 'global_autodelete_30s']
                        ],
                        [
                            ['text' => '⏱️ 35s', 'callback_data' => 'global_autodelete_35s'],
                            ['text' => '⏱️ 40s', 'callback_data' => 'global_autodelete_40s']
                        ],
                        [
                            ['text' => '⏱️ 45s', 'callback_data' => 'global_autodelete_45s'],
                            ['text' => '⏱️ 50s', 'callback_data' => 'global_autodelete_50s']
                        ],
                        [
                            ['text' => '⏱️ 55s', 'callback_data' => 'global_autodelete_55s'],
                            ['text' => '⏱️ 1m', 'callback_data' => 'global_autodelete_60s']
                        ],
                        [
                            ['text' => '🔢 عدد دلخواه', 'callback_data' => 'global_autodelete_custom']
                        ],
                        [
                            ['text' => '❌ غیرفعال', 'callback_data' => 'global_autodelete_off']
                        ],
                        [
                            ['text' => '⬅️ بازگشت', 'callback_data' => 'media']
                        ]
                    ]
                ];
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
            }
            break;
            
        // Global auto-delete callbacks
        case 'global_autodelete_5s':
        case 'global_autodelete_10s':
        case 'global_autodelete_15s':
        case 'global_autodelete_20s':
        case 'global_autodelete_25s':
        case 'global_autodelete_30s':
        case 'global_autodelete_35s':
        case 'global_autodelete_40s':
        case 'global_autodelete_45s':
        case 'global_autodelete_50s':
        case 'global_autodelete_55s':
        case 'global_autodelete_60s':
        case 'global_autodelete_off':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/auto-delete-handler.php";
                $seconds_map = [
                    'global_autodelete_5s' => 5,
                    'global_autodelete_10s' => 10,
                    'global_autodelete_15s' => 15,
                    'global_autodelete_20s' => 20,
                    'global_autodelete_25s' => 25,
                    'global_autodelete_30s' => 30,
                    'global_autodelete_35s' => 35,
                    'global_autodelete_40s' => 40,
                    'global_autodelete_45s' => 45,
                    'global_autodelete_50s' => 50,
                    'global_autodelete_55s' => 55,
                    'global_autodelete_60s' => 60
                ];
                
                if ($callback_data === 'global_autodelete_off') {
                    $seconds = 0;
                } else {
                    $seconds = $seconds_map[$callback_data] ?? 0;
                }
                
                $msg = handleSetGlobalAutoDelete($seconds);
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, getInlineKeyboard('back_only'));
                } else {
                    sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                }
            }
            break;
            
        case 'global_autodelete_custom':
            if (isAdmin($user_id)) {
                $msg = "🔢 لطفاً تعداد ثانیه را ارسال کنید:\n\nمثال: 200";
                // Always send new message for input requests
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'global_autodelete_custom']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_autodelete', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_autodelete', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'block':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "🚫 مدیریت بلاک کاربران:", getInlineKeyboard('block_menu'));
                } else {
                    sendMessage($chat_id, "🚫 مدیریت بلاک کاربران:", getInlineKeyboard('block_menu'));
                }
            }
            break;
            
        case 'search_user':
            if (isAdmin($user_id)) {
                // Always send new message for input requests
                sendMessage($chat_id, "🔍 لطفاً یوزرنیم، نام یا شناسه کاربر را ارسال کنید.", getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'search_user']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'searching_user', ?) 
                     ON DUPLICATE KEY UPDATE state = 'searching_user', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'set_vip':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/user-handler.php";
                $msg = "⭐ لطفاً کاربر را forward کنید یا شناسه کاربر را ارسال کنید.";
                // Always send new message for input requests
                sendMessage($chat_id, $msg, getInlineKeyboard('block_menu'));
            }
            break;
            
        case 'remove_vip':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/user-handler.php";
                $msg = "❌ لطفاً کاربر را forward کنید یا شناسه کاربر را ارسال کنید.";
                // Always send new message for input requests
                sendMessage($chat_id, $msg, getInlineKeyboard('block_menu'));
            }
            break;
            
        case 'point_management':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "💎 مدیریت امتیاز:", getInlineKeyboard('point_management'));
                } else {
                    sendMessage($chat_id, "💎 مدیریت امتیاز:", getInlineKeyboard('point_management'));
                }
            }
            break;
            
        case 'add_points':
        case 'deduct_points':
        case 'set_points':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/point-handler.php";
                $action = str_replace('_points', '', $callback_data);
                $action_names = ['add' => 'افزودن', 'deduct' => 'کسر', 'set' => 'تنظیم'];
                $action_name = $action_names[$action] ?? 'افزودن';
                $msg = "💎 لطفاً کاربر را forward کنید یا شناسه کاربر را ارسال کنید.\n\nسپس مقدار امتیاز را برای {$action_name} وارد کنید.";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['action' => $action]);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_points', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_points', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'block_user':
        case 'unblock_user':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/user-handler.php";
                $action = str_replace('_user', '', $callback_data);
                $action_name = $action === 'block' ? 'بلاک' : 'آنبلاک';
                $msg = "🚫 لطفاً کاربر را forward کنید یا شناسه کاربر را ارسال کنید.";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['action' => $action]);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'managing_block', ?) 
                     ON DUPLICATE KEY UPDATE state = 'managing_block', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'add_points_user':
        case 'deduct_points_user':
        case 'set_points_user':
            if (isAdmin($user_id)) {
                // Get target user_id from state
                $user_state = dbQueryOne("SELECT state_data FROM user_states WHERE user_id = ? AND state = 'viewing_user'", [$user_id]);
                if ($user_state && !empty($user_state['state_data'])) {
                    $state_data = json_decode($user_state['state_data'], true);
                    $target_user_id = isset($state_data['target_user_id']) ? (int)$state_data['target_user_id'] : null;
                    
                    if ($target_user_id !== null) {
                        // Ask for points value
                        require_once "inc/handlers/point-handler.php";
                        $action = str_replace('_points_user', '', $callback_data);
                        $action_names = ['add' => 'افزودن', 'deduct' => 'کسر', 'set' => 'تنظیم'];
                        $action_name = $action_names[$action] ?? 'افزودن';
                        $msg = "💎 لطفاً مقدار امتیاز را برای {$action_name} وارد کنید:";
                        sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                        // Save state with action and target_user_id
                        $state_data = json_encode(['action' => $action, 'target_user_id' => $target_user_id]);
                        dbExecute(
                            "UPDATE user_states SET state = 'managing_points', state_data = ? WHERE user_id = ?",
                            [$state_data, $user_id]
                        );
                    } else {
                        sendMessage($chat_id, "❌ خطا: کاربر یافت نشد.");
                    }
                } else {
                    sendMessage($chat_id, "❌ لطفاً ابتدا کاربر را جستجو کنید.");
                }
            }
            break;
            
        case 'block_user_info':
        case 'unblock_user_info':
            if (isAdmin($user_id)) {
                // Get target user_id from state
                $user_state = dbQueryOne("SELECT state_data FROM user_states WHERE user_id = ? AND state = 'viewing_user'", [$user_id]);
                if ($user_state && !empty($user_state['state_data'])) {
                    $state_data = json_decode($user_state['state_data'], true);
                    $target_user_id = isset($state_data['target_user_id']) ? (int)$state_data['target_user_id'] : null;
                    
                    if ($target_user_id !== null) {
                        require_once "inc/handlers/user-handler.php";
                        $action = str_replace('_user_info', '', $callback_data);
                        
                        if ($action === 'block') {
                            handleBlockUserFromPanel($chat_id, $user_id, $target_user_id);
                        } else {
                            handleUnblockUserFromPanel($chat_id, $user_id, $target_user_id);
                        }
                        
                        // Refresh user info
                        displayUserInfo($chat_id, $target_user_id, null, true);
                    } else {
                        sendMessage($chat_id, "❌ خطا: کاربر یافت نشد.");
                    }
                } else {
                    sendMessage($chat_id, "❌ لطفاً ابتدا کاربر را جستجو کنید.");
                }
            }
            break;
            
        case 'set_vip_user':
        case 'remove_vip_user':
            if (isAdmin($user_id)) {
                // Get target user_id from state
                $user_state = dbQueryOne("SELECT state_data FROM user_states WHERE user_id = ? AND state = 'viewing_user'", [$user_id]);
                if ($user_state && !empty($user_state['state_data'])) {
                    $state_data = json_decode($user_state['state_data'], true);
                    $target_user_id = isset($state_data['target_user_id']) ? (int)$state_data['target_user_id'] : null;
                    
                    if ($target_user_id !== null) {
                        require_once "inc/handlers/user-handler.php";
                        $action = str_replace('_vip_user', '', $callback_data);
                        
                        if ($action === 'set') {
                            if (setUserVIP($target_user_id)) {
                                sendMessage($chat_id, "✅ کاربر با موفقیت VIP شد!");
                            } else {
                                sendMessage($chat_id, "❌ خطا در VIP کردن کاربر.");
                            }
                        } else {
                            if (removeUserVIP($target_user_id)) {
                                sendMessage($chat_id, "✅ VIP کاربر با موفقیت حذف شد!");
                            } else {
                                sendMessage($chat_id, "❌ خطا در حذف VIP کاربر.");
                            }
                        }
                        
                        // Refresh user info
                        displayUserInfo($chat_id, $target_user_id, null, true);
                    } else {
                        sendMessage($chat_id, "❌ خطا: کاربر یافت نشد.");
                    }
                } else {
                    sendMessage($chat_id, "❌ لطفاً ابتدا کاربر را جستجو کنید.");
                }
            }
            break;
            
        case 'advanced_settings':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "⚙️ تنظیمات پیشرفته:", getInlineKeyboard('advanced_settings'));
                } else {
                    sendMessage($chat_id, "⚙️ تنظیمات پیشرفته:", getInlineKeyboard('advanced_settings'));
                }
            }
            break;
            
        case 'set_stars_rate':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "⭐ لطفاً نرخ Stars به امتیاز را وارد کنید:\n\nمثال: 1 (یعنی 1 Star = 1 امتیاز)";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'stars_rate']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'set_referral_points':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "🎁 لطفاً امتیازهای معرف را به فرمت زیر وارد کنید:\n\nمعرف|معرف‌شده\n\nمثال:\n10|5";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'referral_points']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'set_referral_multilevel_points':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "🎁 لطفاً امتیازهای رفرال چند سطحی را به فرمت زیر وارد کنید:\n\nسطح1|سطح2|سطح3\n\nمثال:\n5|3|1";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'referral_multilevel_points']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'show_referral_tree':
            // Check if referral system is enabled
            require_once "inc/handlers/settings-handler.php";
            $referral_enabled = (bool)getSettingValue('referral_enabled', true);
            
            if (!$referral_enabled) {
                $msg = "❌ سیستم معرف غیرفعال است.";
                $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                    ? getInlineKeyboard('user_menu', 1, ['user_id' => $user_id])
                    : getInlineKeyboard('user_menu');
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
                break;
            }
            
            // Check forced join first
            require_once "inc/handlers/channel-handler.php";
            if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
                break; // Message already sent
            }
            
            require_once "inc/handlers/referral-handler.php";
            displayReferralTree($chat_id, $user_id, $callback_message_id);
            break;
            
        case 'set_forced_join_text':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "📢 لطفاً متن جدید Forced Join را ارسال کنید:";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'forced_join_text']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'gambling_settings':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $gambling_enabled = (bool)getSettingValue('gambling_enabled', true);
                $gambling_min_bet = (int)getSettingValue('gambling_min_bet', 1);
                $gambling_max_bet = (int)getSettingValue('gambling_max_bet', 100);
                $gambling_win_multiplier = (float)getSettingValue('gambling_win_multiplier', 2.0);
                $gambling_daily_limit = (int)getSettingValue('gambling_daily_limit', 0);
                $gambling_cooldown_seconds = (int)getSettingValue('gambling_cooldown_seconds', 0);
                
                $msg = "🎲 تنظیمات بازی و شرط‌بندی:\n\n";
                $msg .= "✅ فعال: " . ($gambling_enabled ? '✅' : '❌') . "\n";
                $msg .= "💰 حداقل شرط: {$gambling_min_bet} امتیاز\n";
                $msg .= "💰 حداکثر شرط: {$gambling_max_bet} امتیاز\n";
                $msg .= "🎯 نرخ برد: {$gambling_win_multiplier}x\n";
                $msg .= "📅 محدودیت روزانه: " . ($gambling_daily_limit > 0 ? $gambling_daily_limit . ' بازی' : 'بدون محدودیت') . "\n";
                $msg .= "⏱️ زمان انتظار: " . ($gambling_cooldown_seconds > 0 ? $gambling_cooldown_seconds . ' ثانیه' : 'بدون انتظار') . "\n\n";
                $msg .= "💡 برای تغییر هر تنظیم، روی دکمه مربوطه کلیک کنید.";
                
                $keyboard = getInlineKeyboard('gambling_settings');
                
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, $keyboard);
                } else {
                    sendMessage($chat_id, $msg, $keyboard);
                }
            }
            break;
            
        case 'set_gambling_min_bet':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "💰 لطفاً حداقل شرط را ارسال کنید (عدد):\n\nمثال: 5";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'gambling_min_bet']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'set_gambling_max_bet':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "💰 لطفاً حداکثر شرط را ارسال کنید (عدد):\n\nمثال: 1000";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'gambling_max_bet']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'set_gambling_multiplier':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "🎯 لطفاً نرخ برد را ارسال کنید (عدد اعشاری):\n\nمثال: 2.0 (برای 2 برابر)\nمثال: 1.5 (برای 1.5 برابر)";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'gambling_multiplier']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'set_gambling_daily_limit':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "📅 لطفاً محدودیت تعداد بازی در روز را ارسال کنید:\n\nعدد (مثلاً 10) یا 0 برای بدون محدودیت";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'gambling_daily_limit']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'set_gambling_cooldown':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $msg = "⏱️ لطفاً زمان انتظار بین بازی‌ها را به ثانیه ارسال کنید:\n\nعدد (مثلاً 300 برای 5 دقیقه) یا 0 برای بدون انتظار";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['type' => 'gambling_cooldown']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'setting_advanced', ?) 
                     ON DUPLICATE KEY UPDATE state = 'setting_advanced', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'send_message':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "✉️ ارسال پیام:", getInlineKeyboard('send_message'));
                } else {
                    sendMessage($chat_id, "✉️ ارسال پیام:", getInlineKeyboard('send_message'));
                }
            }
            break;
            
        case 'broadcast_normal':
        case 'broadcast_forward':
            if (isAdmin($user_id)) {
                $type = strpos($callback_data, 'forward') !== false ? 'forward' : 'normal';
                // Always send new message for input requests
                sendMessage($chat_id, "📢 لطفاً پیامی که می‌خواهید broadcast کنید را ارسال کنید.", getInlineKeyboard('back_only'));
                // Save state for broadcast
                $state_data = json_encode(['broadcast_type' => $type, 'last_message_id' => $callback_message_id]);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'broadcasting', ?) 
                     ON DUPLICATE KEY UPDATE state = 'broadcasting', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'advertisement':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "📣 مدیریت تبلیغات:", getInlineKeyboard('advertisement'));
                } else {
                    sendMessage($chat_id, "📣 مدیریت تبلیغات:", getInlineKeyboard('advertisement'));
                }
            }
            break;
            
        case 'add_advertisement':
        case 'remove_advertisement':
        case 'list_advertisements':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/advertisement-handler.php";
                if ($callback_data === 'list_advertisements') {
                    listAdvertisements($chat_id, $callback_message_id);
                } elseif ($callback_data === 'add_advertisement') {
                    $msg = "➕ لطفاً محتوای تبلیغ را ارسال کنید.\n\n💡 می‌توانید متن، عکس، ویدیو یا فایل ارسال کنید.";
                    sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                    // Save state
                    $state_data = json_encode(['step' => 'content']);
                    dbExecute(
                        "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'adding_advertisement', ?) 
                         ON DUPLICATE KEY UPDATE state = 'adding_advertisement', state_data = ?",
                        [$user_id, $state_data, $state_data]
                    );
                } else {
                    $msg = "➖ لطفاً شناسه تبلیغ را ارسال کنید.";
                    sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                    // Save state
                    $state_data = json_encode(['step' => 'remove']);
                    dbExecute(
                        "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'removing_advertisement', ?) 
                         ON DUPLICATE KEY UPDATE state = 'removing_advertisement', state_data = ?",
                        [$user_id, $state_data, $state_data]
                    );
                }
            }
            break;
            
        case 'advertisement_channels':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "📢 مدیریت کانال‌های تبلیغاتی:", getInlineKeyboard('advertisement_channels'));
                } else {
                    sendMessage($chat_id, "📢 مدیریت کانال‌های تبلیغاتی:", getInlineKeyboard('advertisement_channels'));
                }
            }
            break;
            
        case 'add_ad_channel':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/channel-handler.php";
                require_once "inc/handlers/advertisement-handler.php";
                $msg = "➕ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.\n\nسپس متن دکمه را وارد کنید (اختیاری، پیش‌فرض: 📢 کانال ما)";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['step' => 'channel_info']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'adding_ad_channel', ?) 
                     ON DUPLICATE KEY UPDATE state = 'adding_ad_channel', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'remove_ad_channel':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/advertisement-handler.php";
                $msg = "🗑️ لطفاً شناسه کانال تبلیغاتی را ارسال کنید.";
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                // Save state
                $state_data = json_encode(['step' => 'remove']);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'removing_ad_channel', ?) 
                     ON DUPLICATE KEY UPDATE state = 'removing_ad_channel', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'list_ad_channels':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/advertisement-handler.php";
                if ($callback_message_id) {
                    listAdvertisementChannels($chat_id, $callback_message_id);
                } else {
                    listAdvertisementChannels($chat_id);
                }
            }
            break;
            
        case 'toggle_ad_enabled':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                $current = (bool)getSettingValue('advertisement_enabled', true);
                $new = !$current;
                setSettingValue('advertisement_enabled', $new);
                $msg = "✅ تبلیغات کانال: " . ($new ? 'فعال' : 'غیرفعال');
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, getInlineKeyboard('advertisement_channels'));
                } else {
                    sendMessage($chat_id, $msg, getInlineKeyboard('advertisement_channels'));
                }
            }
            break;
            
        case 'settings':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    displaySettings($chat_id, $callback_message_id);
                } else {
                    displaySettings($chat_id);
                }
            }
            break;
            
        case 'features_list':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                if ($callback_message_id) {
                    displayFeaturesList($chat_id, $callback_message_id);
                } else {
                    displayFeaturesList($chat_id);
                }
            }
            break;
            
        case 'toggle_feature_paid_mode':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('is_paid_mode', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_advertisement':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('advertisement_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_forced_join':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('forced_join_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_daily_spin':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('daily_spin_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_referral':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('referral_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_mandatory_seen':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('mandatory_seen_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_reaction_channels':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('reaction_channels_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_promo_codes':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('promo_codes_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'toggle_feature_gambling':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/settings-handler.php";
                toggleFeature('gambling_enabled', $chat_id, $callback_message_id);
            }
            break;
            
        case 'send_channel':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "📢 لطفاً پیامی که می‌خواهید به کانال ارسال کنید را forward کنید.", getInlineKeyboard('back_only'));
                } else {
                    sendMessage($chat_id, "📢 لطفاً پیامی که می‌خواهید به کانال ارسال کنید را forward کنید.", getInlineKeyboard('back_only'));
                }
            }
            break;
            
        case 'media_settings':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "⚙️ تنظیمات مدیا:", getInlineKeyboard('media_settings'));
                } else {
                    sendMessage($chat_id, "⚙️ تنظیمات مدیا:", getInlineKeyboard('media_settings'));
                }
            }
            break;
            
        case 'media_info':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/media-handler.php";
                if ($callback_message_id) {
                    displayMediaInfo($chat_id, $callback_message_id);
                } else {
                    displayMediaInfo($chat_id);
                }
            }
            break;
            
        // Auto-delete callbacks
        case 'autodelete_30s':
        case 'autodelete_1m':
        case 'autodelete_5m':
        case 'autodelete_10m':
        case 'autodelete_30m':
        case 'autodelete_1h':
        case 'autodelete_6h':
        case 'autodelete_24h':
        case 'autodelete_off':
            if (isAdmin($user_id)) {
                $seconds_map = [
                    'autodelete_30s' => 30,
                    'autodelete_1m' => 60,
                    'autodelete_5m' => 300,
                    'autodelete_10m' => 600,
                    'autodelete_30m' => 1800,
                    'autodelete_1h' => 3600,
                    'autodelete_6h' => 21600,
                    'autodelete_24h' => 86400
                ];
                
                if ($callback_data === 'autodelete_off') {
                    $msg = "❌ لطفاً به پیام فایلی که می‌خواهید timer را غیرفعال کنید reply کنید.";
                } else {
                    $seconds = $seconds_map[$callback_data] ?? 0;
                    $msg = "⏰ لطفاً به پیام فایلی که می‌خواهید timer {$seconds} ثانیه‌ای برای آن تنظیم کنید reply کنید.";
                }
                
                // Always send new message for input requests
                sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
            }
            break;
            
        case 'promo_codes':
            if (isAdmin($user_id)) {
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, "🎁 مدیریت کدهای جایزه/تخفیف/سکه:", getInlineKeyboard('promo_codes'));
                } else {
                    sendMessage($chat_id, "🎁 مدیریت کدهای جایزه/تخفیف/سکه:", getInlineKeyboard('promo_codes'));
                }
            }
            break;
            
        case 'add_promo_gift':
        case 'add_promo_discount':
        case 'add_promo_coin':
            if (isAdmin($user_id)) {
                $promo_type = str_replace('add_promo_', '', $callback_data);
                $type_names = ['gift' => 'جایزه', 'discount' => 'تخفیف', 'coin' => 'سکه'];
                $type_name = $type_names[$promo_type] ?? 'جایزه';
                $msg = "🎁 لطفاً اطلاعات کد {$type_name} را به فرمت زیر ارسال کنید:\n\nکد|مقدار|تعداد استفاده (0=نامحدود)|تاریخ انقضا (اختیاری)\n\nمثال:\nGIFT100|100|10|2025-12-31";
                // Always send new message for input requests
                $result = sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                $sent_message_id = null;
                if ($result !== false && isset($result['ok']) && $result['ok']) {
                    $sent_message_id = (int)$result['result']['message_id'];
                }
                // Save state
                $state_data = json_encode(['promo_type' => $promo_type, 'last_message_id' => $sent_message_id]);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'adding_promo', ?) 
                     ON DUPLICATE KEY UPDATE state = 'adding_promo', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
            
        case 'list_promo_codes':
            if (isAdmin($user_id)) {
                require_once "inc/handlers/promo-code-handler.php";
                $codes = listPromoCodes();
                if ($codes === false || empty($codes)) {
                    $msg = "📑 لیست کدها خالی است.";
                } else {
                    $msg = "📑 لیست کدها:\n\n";
                    foreach ($codes as $index => $code) {
                        $type_names = ['gift' => 'جایزه', 'discount' => 'تخفیف', 'coin' => 'سکه'];
                        $type_name = $type_names[$code['type']] ?? $code['type'];
                        $status = $code['is_active'] ? '✅' : '❌';
                        $usage = $code['usage_limit'] > 0 ? "{$code['usage_count']}/{$code['usage_limit']}" : "{$code['usage_count']}/∞";
                        $expires = $code['expires_at'] ? date('Y-m-d', strtotime($code['expires_at'])) : 'نامحدود';
                        $msg .= ($index + 1) . ". {$status} {$code['code']} ({$type_name})\n";
                        $msg .= "   💰 مقدار: {$code['value']}\n";
                        $msg .= "   📊 استفاده: {$usage}\n";
                        $msg .= "   ⏰ انقضا: {$expires}\n\n";
                    }
                }
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, getInlineKeyboard('promo_codes'));
                } else {
                    sendMessage($chat_id, $msg, getInlineKeyboard('promo_codes'));
                }
            }
            break;
            
        case 'delete_promo_code':
            if (isAdmin($user_id)) {
                $msg = "🗑️ لطفاً شناسه کد را ارسال کنید.";
                if ($callback_message_id) {
                    editMessage($chat_id, $callback_message_id, $msg, getInlineKeyboard('back_only'));
                } else {
                    sendMessage($chat_id, $msg, getInlineKeyboard('back_only'));
                }
                // Save state
                $state_data = json_encode(['last_message_id' => $callback_message_id]);
                dbExecute(
                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'deleting_promo', ?) 
                     ON DUPLICATE KEY UPDATE state = 'deleting_promo', state_data = ?",
                    [$user_id, $state_data, $state_data]
                );
            }
            break;
    }
    
    exit;
}

// Handle private chat messages
if ($chat_type == "private") {
    // Check if user is blocked
    if (isUserBlocked($user_id)) {
        sendMessage($chat_id, "❌ شما بلاک شده‌اید و نمی‌توانید از ربات استفاده کنید.");
        exit;
    }
    
    // Handle commands
    if ($text && strpos($text, '/') === 0) {
        $command = explode(' ', $text)[0];
        $command = strtolower($command);
        
        // /skip is not a general command - it's only for state handling (e.g., skipping caption in admin workflows)
        // Check if user is in a state that needs /skip handling
        // If so, don't treat /skip as a command, let state handler process it
        if ($command === '/skip') {
            $user_state = dbQueryOne("SELECT state FROM user_states WHERE user_id = ?", [$user_id]);
            if ($user_state === false) {
                // No state, /skip is not valid - show message
                sendMessage($chat_id, "❌ دستور /skip فقط در مراحل خاصی قابل استفاده است.");
                exit;
            }
            // User is in a state, let state handler process /skip
            // Don't treat as command, continue to state handling below
        } else {
            switch ($command) {
            case '/start':
        $start_param = getStartParameter($text);

        if ($start_param) {
                    // Check if it's a file code (all links now use file_code, not file_id)
                    require_once "inc/handlers/upload-handler.php";
                    $file = getFileByCode($start_param);
                    
                    if ($file !== false) {
                        // User wants to download a file
                        $file_id = (int)$file['id'];
                        
                        // Check forced join channels first (file-specific or global)
                        require_once "inc/handlers/channel-handler.php";
                        $required_channels = getFileForcedJoinChannels($file_id);
                        
                        if (!empty($required_channels)) {
                            $membership = checkChannelMembership($user_id, $required_channels);
                            
                            if (!$membership['is_member']) {
                                // Save file_id to state for later download after verification
                                $state_data = json_encode(['file_id' => $file_id]);
                                dbExecute(
                                    "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'pending_file_download', ?) 
                                     ON DUPLICATE KEY UPDATE state = 'pending_file_download', state_data = ?",
                                    [$user_id, $state_data, $state_data]
                                );
                                
                                // Use checkForcedJoinAndNotify for consistent glass keyboard
                                if (!checkForcedJoinAndNotify($user_id, $chat_id, $required_channels)) {
                                    exit;
                                }
                                // After showing forced join message, don't continue to download
                                exit;
                            }
                        }
                        
                        // Handle download - file will be sent directly
                        $result = handleFileDownload($user_id, $file_id, $chat_id);
                        
                        // Don't send additional message - file is already sent by handleFileDownload
                        // Only send message if there was an error
                        if (!$result['success'] && !empty($result['message'])) {
                            sendMessage($chat_id, $result['message']);
                        }
                        exit;
                    }
                    
                    // Check if it's a referral code
                    $referral_result = handleReferralUsage($user_id, $start_param);
                    
                    if ($referral_result['success']) {
                        $points = $referral_result['points_given'];
                        sendMessage($chat_id, "✅ خوش آمدید!\n🎁 شما و معرف شما امتیاز دریافت کردید!");
                    }
                }
                
                // Check forced join channels for all users (except main admin)
                require_once "inc/handlers/channel-handler.php";
                $required_channels = getFileForcedJoinChannels(0); // Get global channels
                
                if (!empty($required_channels)) {
                    // Use checkForcedJoinAndNotify for consistent glass keyboard
                    if (!checkForcedJoinAndNotify($user_id, $chat_id, $required_channels)) {
                        exit;
                    }
                }
                
                // Check if user is new (first time starting)
                $user_exists = dbQueryOne("SELECT id FROM users WHERE id = ?", [$user_id]);
                $is_new_user = ($user_exists === false);
                
                // Give welcome bonus to new users
                if ($is_new_user) {
                    require_once "inc/handlers/point-handler.php";
                    addUserPoints($user_id, 10);
                    $welcome_bonus_msg = "\n\n🎁 هدیه استارت: 10 سکه رایگان دریافت کردید!";
                } else {
                    $welcome_bonus_msg = "";
                }
                
                // Show welcome message
                if (isAdmin($user_id)) {
                    // Check if admin is in user mode
                    if (isRealAdmin($user_id) && isInUserMode($user_id)) {
                        // Admin in user mode - show user menu with exit button
                        $keyboard = getInlineKeyboard('user_menu', 1, ['user_id' => $user_id]);
                        sendMessage($chat_id, "👋 سلام! به ربات آپلودر فایل خوش آمدید.{$welcome_bonus_msg}\n\n💎 امتیاز شما: " . getUserPoints($user_id), $keyboard);
                    } else {
                        // Normal admin menu
                        sendMessage($chat_id, "👋 سلام! به پنل ادمین خوش آمدید.", getInlineKeyboard('main', 1));
                    }
                } else {
                    sendMessage($chat_id, "👋 سلام! به ربات آپلودر فایل خوش آمدید.{$welcome_bonus_msg}\n\n💎 امتیاز شما: " . getUserPoints($user_id) . "\n\n🎁 برای استفاده از کد جایزه/تخفیف/سکه، کد را ارسال کنید.", getInlineKeyboard('user_menu'));
                }
                break;
                
            case '/points':
            case '/امتیاز':
            case '/referral':
            case '/معرف':
                // Redirect to user menu - no text commands for regular users
                sendMessage($chat_id, "⚠️ لطفاً از منوی کاربر استفاده کنید.\n\nبرای دسترسی به منوی اصلی، دستور /start را ارسال کنید.", getInlineKeyboard('user_menu'));
                break;
                
            // Admin commands - redirect to panel menu
            case '/addadmin':
            case '/removeadmin':
            case '/listadmins':
            case '/addchannel':
            case '/removechannel':
            case '/listchannels':
            case '/stats':
            case '/آمار':
            case '/searchuser':
            case '/setvip':
            case '/vip':
            case '/removevip':
            case '/unvip':
            case '/settings':
            case '/تنظیمات':
            case '/broadcast':
            case '/broadcastforward':
            case '/autodelete_30s':
            case '/autodelete_1m':
            case '/autodelete_5m':
            case '/autodelete_10m':
            case '/autodelete_30m':
            case '/autodelete_1h':
            case '/autodelete_6h':
            case '/autodelete_24h':
            case '/autodelete':
            case '/autodelete_off':
            case '/download':
            case '/دانلود':
                if (isAdmin($user_id)) {
                    sendMessage($chat_id, "⚠️ لطفاً از منوی پنل مدیریت استفاده کنید.\n\nبرای دسترسی به منوی اصلی، دستور /start را ارسال کنید.", getInlineKeyboard('main', 1));
                } else {
                    $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
                        ? getInlineKeyboard('user_menu', 1, ['user_id' => $user_id])
                        : getInlineKeyboard('user_menu');
                    sendMessage($chat_id, "⚠️ لطفاً از منوی کاربر استفاده کنید.\n\nبرای دسترسی به منوی اصلی، دستور /start را ارسال کنید.", $keyboard);
                }
                break;
                
            default:
                // Unknown command
                break;
            }
            
            exit;
        }
    }
    
    // Handle file uploads (admin only) - but check state first
    if (isAdmin($user_id)) {
        // Check if user is in a state that requires message handling
        $user_state = dbQueryOne("SELECT state FROM user_states WHERE user_id = ?", [$user_id]);
        
        // If user is in a state that requires message handling, don't treat as file upload
        // States that need to process forwarded messages or media:
        $states_that_need_message_handling = [
            'adding_advertisement',      // Needs media or text for ad content
            'managing_reaction_channel', // Needs forwarded channel message (may have media)
            'managing_channel',          // Needs forwarded channel message (may have media)
            'managing_admin',            // Needs forwarded user message (may have media)
            'adding_ad_channel',         // Needs forwarded channel message (may have media)
            'managing_points',           // Needs forwarded user message (may have media)
            'managing_block',            // Needs forwarded user message (may have media)
            'managing_vip',              // Needs forwarded user message (may have media)
            'broadcasting',               // Needs message (text or media) for broadcast
            'creating_backup'             // Needs bot link and text for backup
        ];
        
        if ($user_state !== false && in_array($user_state['state'], $states_that_need_message_handling)) {
            // Let state handler process it
        } else {
            // Check if message contains a file
            if (isset($message['photo']) || isset($message['video']) || isset($message['document']) ||
                isset($message['audio']) || isset($message['voice']) || isset($message['video_note']) ||
                isset($message['sticker']) || isset($message['animation'])) {
                
                // Check if it's part of an album
                if (isset($message['media_group_id'])) {
                    handleAlbumUpload($message);
                } else {
                    handleFileUpload($message);
                }
                
                exit;
            }
        }
    }
    
    // Handle admin states (broadcasting, adding promo code, etc.)
    // Check for both text and media messages
    if (isAdmin($user_id) && ($text || isset($message['photo']) || isset($message['video']) || 
        isset($message['document']) || isset($message['audio']) || isset($message['voice']) ||
        isset($message['video_note']) || isset($message['sticker']) || isset($message['animation']))) {
        $user_state = dbQueryOne("SELECT state, state_data FROM user_states WHERE user_id = ?", [$user_id]);
        
        if ($user_state) {
            // Check if state_data is not null before decoding
            $state_data = null;
            if (!empty($user_state['state_data'])) {
                $state_data = json_decode($user_state['state_data'], true);
            }
            
            if ($user_state['state'] === 'setting_autodelete') {
                // Handle custom auto-delete seconds
                require_once "inc/handlers/auto-delete-handler.php";
                $type = $state_data['type'] ?? '';
                
                if ($type === 'global_autodelete_custom') {
                    // Global auto-delete custom
                    $seconds = (int)trim($text);
                    
                    if ($seconds > 0) {
                        $msg = handleSetGlobalAutoDelete($seconds);
                        sendMessage($chat_id, $msg);
                    } else {
                        sendMessage($chat_id, "❌ لطفاً عددی بزرگ‌تر از صفر وارد کنید.");
                    }
                } elseif ($type === 'file_autodelete_custom') {
                    // File-specific auto-delete custom (with file_id from state)
                    $file_id = isset($state_data['file_id']) ? (int)$state_data['file_id'] : 0;
                    
                    if ($file_id > 0) {
                        $seconds = (int)trim($text);
                        
                        if ($seconds > 0) {
                            $result = setFileAutoDelete($file_id, $seconds);
                            $time_str = $seconds . " ثانیه";
                            $msg = $result ? "✅ تایم حذف خودکار برای فایل #{$file_id} روی {$time_str} تنظیم شد." : "❌ خطا در تنظیم تایم حذف.";
                            sendMessage($chat_id, $msg);
                        } else {
                            sendMessage($chat_id, "❌ لطفاً عددی بزرگ‌تر از صفر وارد کنید.");
                        }
                    } else {
                        // Fallback: needs reply to file
                        if (!isset($message['reply_to_message'])) {
                            sendMessage($chat_id, "❌ لطفاً به پیام فایلی که می‌خواهید timer برای آن تنظیم کنید reply کنید.");
                            dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                            exit;
                        }
                        
                        $seconds = (int)trim($text);
                        
                        if ($seconds > 0) {
                            handleSetAutoDelete($message, $seconds);
                        } else {
                            sendMessage($chat_id, "❌ لطفاً عددی بزرگ‌تر از صفر وارد کنید.");
                        }
                    }
                }
                
                // Clear state
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'broadcasting') {
                $broadcast_type = $state_data['broadcast_type'] ?? 'normal';
                
                // Clear state
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                
                // Handle broadcast
                require_once "inc/handlers/broadcast-handler.php";
                handleBroadcast($message, $broadcast_type);
                exit;
            } elseif ($user_state['state'] === 'adding_promo') {
                // Handle adding promo code
                require_once "inc/handlers/promo-code-handler.php";
                $promo_type = $state_data['promo_type'] ?? 'gift';
                
                // Parse input: code|value|usage_limit|expires_at
                $parts = explode('|', $text);
                if (count($parts) < 2) {
                    sendMessage($chat_id, "❌ فرمت اشتباه است.\n\nفرمت صحیح:\nکد|مقدار|تعداد استفاده|تاریخ انقضا");
                    exit;
                }
                
                $code = strtoupper(trim($parts[0]));
                $value = (int)trim($parts[1]);
                $usage_limit = isset($parts[2]) ? (int)trim($parts[2]) : 0;
                $expires_at = 0;
                
                if (isset($parts[3]) && !empty(trim($parts[3]))) {
                    $expires_at = strtotime(trim($parts[3]));
                    if ($expires_at === false) {
                        sendMessage($chat_id, "❌ تاریخ انقضا نامعتبر است.");
                        exit;
                    }
                }
                
                if (empty($code) || $value <= 0) {
                    sendMessage($chat_id, "❌ کد و مقدار باید معتبر باشند.");
                    exit;
                }
                
                // Create promo code
                require_once "inc/handlers/promo-code-handler.php";
                if (createPromoCode($code, $promo_type, $value, $usage_limit, $expires_at, $user_id)) {
                    // Clear state
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    
                    $type_names = ['gift' => 'جایزه', 'discount' => 'تخفیف', 'coin' => 'سکه'];
                    $type_name = $type_names[$promo_type] ?? 'جایزه';
                    sendMessage($chat_id, "✅ کد {$type_name} با موفقیت ایجاد شد!\n\nکد: {$code}\nمقدار: {$value}");
                } else {
                    sendMessage($chat_id, "❌ خطا در ایجاد کد. ممکن است کد تکراری باشد.");
                }
                exit;
            } elseif ($user_state['state'] === 'deleting_promo') {
                // Handle deleting promo code
                require_once "inc/handlers/promo-code-handler.php";
                $code_to_delete = trim($text);
                
                if (!empty($code_to_delete)) {
                    $result = deletePromoCode($code_to_delete);
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    sendMessage($chat_id, $result['message']);
                } else {
                    sendMessage($chat_id, "❌ کد نامعتبر است.");
                }
                exit;
            } elseif ($user_state['state'] === 'adding_ad_channel') {
                // Handle adding advertisement channel
                require_once "inc/handlers/channel-handler.php";
                require_once "inc/handlers/advertisement-handler.php";
                
                $step = $state_data['step'] ?? 'channel_info';
                
                if ($step === 'channel_info') {
                    // Extract channel info
                    $channel_info = extractChannelInfo($message);
                    
                    if ($channel_info === false) {
                        sendMessage($chat_id, "❌ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.");
                        exit;
                    }
                    
                    // Save channel info to state
                    $state_data['channel_info'] = $channel_info;
                    $state_data['step'] = 'button_text';
                    dbExecute(
                        "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                        [json_encode($state_data), $user_id]
                    );
                    
                    sendMessage($chat_id, "✅ اطلاعات کانال دریافت شد.\n\n🔘 لطفاً متن دکمه را وارد کنید (یا Enter بزنید برای استفاده از پیش‌فرض: 📢 کانال ما):");
                    exit;
                } elseif ($step === 'button_text') {
                    // Get button text
                    $button_text = trim($text);
                    if (empty($button_text)) {
                        $button_text = '📢 کانال ما';
                    }
                    
                    $channel_info = $state_data['channel_info'];
                    
                    // Add channel
                    $channel_id = $channel_info['channel_id'];
                    if ($channel_id === 0 && !empty($channel_info['channel_username'])) {
                        // For username-only channels, we'll use 0 as ID (will need to resolve later)
                        $channel_id = 0;
                    }
                    
                    $result = addAdvertisementChannel(
                        $channel_id,
                        $channel_info['channel_title'],
                        $channel_info['channel_username'],
                        $channel_info['channel_link'],
                        $button_text
                    );
                    
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    
                    if ($result !== false) {
                        sendMessage($chat_id, "✅ کانال تبلیغاتی با موفقیت اضافه شد!");
                    } else {
                        sendMessage($chat_id, "❌ خطا در افزودن کانال. ممکن است کانال قبلاً اضافه شده باشد.");
                    }
                    exit;
                }
            } elseif ($user_state['state'] === 'removing_ad_channel') {
                // Handle removing advertisement channel
                require_once "inc/handlers/advertisement-handler.php";
                
                $channel_ad_id = (int)trim($text);
                
                if ($channel_ad_id <= 0) {
                    sendMessage($chat_id, "❌ شناسه نامعتبر است.");
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    exit;
                }
                
                if (removeAdvertisementChannel($channel_ad_id)) {
                    sendMessage($chat_id, "✅ کانال تبلیغاتی با موفقیت حذف شد!");
                } else {
                    sendMessage($chat_id, "❌ خطا در حذف کانال.");
                }
                
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'adding_advertisement') {
                // Handle adding advertisement
                require_once "inc/handlers/advertisement-handler.php";
                
                $step = $state_data['step'] ?? 'content';
                
                if ($step === 'content') {
                    // Check if message has media
                    $has_media = isset($message['photo']) || isset($message['video']) || 
                                 isset($message['document']) || isset($message['audio']) ||
                                 isset($message['voice']) || isset($message['video_note']) ||
                                 isset($message['sticker']) || isset($message['animation']);
                    
                    if ($has_media || !empty($text)) {
                        // Save content to state
                        $state_data['content'] = $message;
                        $state_data['step'] = 'caption';
                        dbExecute(
                            "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                            [json_encode($state_data), $user_id]
                        );
                        
                        sendMessage($chat_id, "✅ محتوای تبلیغ دریافت شد.\n\n📝 لطفاً کپشن تبلیغ را ارسال کنید (یا /skip برای رد کردن):");
                        exit;
                    } else {
                        sendMessage($chat_id, "❌ لطفاً محتوای تبلیغ (متن، عکس، ویدیو یا فایل) را ارسال کنید.");
                        exit;
                    }
                } elseif ($step === 'caption') {
                    // Get caption
                    $caption = null;
                    if (strtolower(trim($text)) !== '/skip' && !empty(trim($text))) {
                        $caption = trim($text);
                    }
                    
                    $state_data['caption'] = $caption;
                    $state_data['step'] = 'url';
                    dbExecute(
                        "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                        [json_encode($state_data), $user_id]
                    );
                    
                    sendMessage($chat_id, "✅ کپشن دریافت شد.\n\n🔗 لطفاً لینک تبلیغ را ارسال کنید (یا /skip برای رد کردن):\n\n💡 مثال: https://t.me/channel یا https://example.com");
                    exit;
                } elseif ($step === 'url') {
                    // Get URL
                    $ad_url = null;
                    if (strtolower(trim($text)) !== '/skip' && !empty(trim($text))) {
                        $ad_url = trim($text);
                        // Validate URL
                        if (!filter_var($ad_url, FILTER_VALIDATE_URL) && !preg_match('/^https?:\/\//', $ad_url)) {
                            sendMessage($chat_id, "❌ لینک نامعتبر است. لطفاً یک URL معتبر ارسال کنید (مثال: https://t.me/channel)");
                            exit;
                        }
                    }
                    
                    $state_data['ad_url'] = $ad_url;
                    
                    if ($ad_url !== null) {
                        $state_data['step'] = 'button_text';
                        dbExecute(
                            "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                            [json_encode($state_data), $user_id]
                        );
                        
                        sendMessage($chat_id, "✅ لینک دریافت شد.\n\n🔘 لطفاً متن دکمه را ارسال کنید (یا Enter بزنید برای استفاده از پیش‌فرض: خرید تبلیغات):");
                        exit;
                    } else {
                        // No URL, skip button_text and go to title
                        $state_data['step'] = 'title';
                        dbExecute(
                            "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                            [json_encode($state_data), $user_id]
                        );
                        
                        sendMessage($chat_id, "✅ لینک رد شد.\n\n📝 لطفاً عنوان تبلیغ را ارسال کنید:");
                        exit;
                    }
                } elseif ($step === 'button_text') {
                    // Get button text
                    $button_text = trim($text);
                    if (empty($button_text)) {
                        $button_text = 'خرید تبلیغات';
                    }
                    
                    $state_data['button_text'] = $button_text;
                    $state_data['step'] = 'title';
                    dbExecute(
                        "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                        [json_encode($state_data), $user_id]
                    );
                    
                    sendMessage($chat_id, "✅ متن دکمه دریافت شد.\n\n📝 لطفاً عنوان تبلیغ را ارسال کنید:");
                    exit;
                } elseif ($step === 'title') {
                    // Get title and save advertisement
                    $title = trim($text);
                    
                    if (empty($title)) {
                        sendMessage($chat_id, "❌ عنوان نمی‌تواند خالی باشد.");
                        exit;
                    }
                    
                    $content_message = $state_data['content'] ?? null;
                    if ($content_message === null) {
                        sendMessage($chat_id, "❌ خطا: محتوای تبلیغ یافت نشد. لطفاً دوباره شروع کنید.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    $caption = $state_data['caption'] ?? null;
                    $ad_url = $state_data['ad_url'] ?? null;
                    $button_text = $state_data['button_text'] ?? 'خرید تبلیغات';
                    
                    $ad_id = addAdvertisement($content_message, $title, $caption, $ad_url, $button_text);
                    
                    if ($ad_id !== false) {
                        sendMessage($chat_id, "✅ تبلیغ با موفقیت اضافه شد!\n\n🆔 شناسه تبلیغ: {$ad_id}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در افزودن تبلیغ.");
                    }
                    
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    exit;
                }
            } elseif ($user_state['state'] === 'removing_advertisement') {
                // Handle removing advertisement
                require_once "inc/handlers/advertisement-handler.php";
                
                $ad_id = (int)trim($text);
                
                if ($ad_id <= 0) {
                    sendMessage($chat_id, "❌ شناسه نامعتبر است.");
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    exit;
                }
                
                if (deleteAdvertisement($ad_id)) {
                    sendMessage($chat_id, "✅ تبلیغ با موفقیت حذف شد!");
                } else {
                    sendMessage($chat_id, "❌ خطا در حذف تبلیغ.");
                }
                
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'managing_points') {
                // Handle point management
                require_once "inc/handlers/point-handler.php";
                require_once "inc/handlers/user-handler.php";
                
                $action = $state_data['action'] ?? '';
                $target_user_id = null;
                
                // Check if we have user_id in state (from previous step)
                if (isset($state_data['target_user_id'])) {
                    $target_user_id = (int)$state_data['target_user_id'];
                    // Now get points value
                    $points = (int)trim($text);
                    
                    if ($points <= 0) {
                        sendMessage($chat_id, "❌ مقدار امتیاز باید بزرگ‌تر از صفر باشد.");
                        exit;
                    }
                    
                    switch ($action) {
                        case 'add':
                            handleAddPointsFromPanel($chat_id, $user_id, $target_user_id, $points);
                            break;
                        case 'deduct':
                            handleDeductPointsFromPanel($chat_id, $user_id, $target_user_id, $points);
                            break;
                        case 'set':
                            handleSetPointsFromPanel($chat_id, $user_id, $target_user_id, $points);
                            break;
                    }
                    
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    exit;
                } else {
                    // First step: get user ID
                    $target_user_id = extractUserIdFromMessage($message);
                    
                    if ($target_user_id === false) {
                        sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
                        exit;
                    }
                    
                    // Save user_id to state and ask for points
                    $action_names = ['add' => 'افزودن', 'deduct' => 'کسر', 'set' => 'تنظیم'];
                    $action_name = $action_names[$action] ?? 'افزودن';
                    $state_data['target_user_id'] = $target_user_id;
                    dbExecute(
                        "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                        [json_encode($state_data), $user_id]
                    );
                    sendMessage($chat_id, "✅ کاربر انتخاب شد.\n\n💎 لطفاً مقدار امتیاز را برای {$action_name} وارد کنید:");
                    exit;
                }
            } elseif ($user_state['state'] === 'managing_block') {
                // Handle block/unblock
                require_once "inc/handlers/user-handler.php";
                
                $action = $state_data['action'] ?? '';
                $target_user_id = extractUserIdFromMessage($message);
                
                if ($target_user_id === false) {
                    sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
                    exit;
                }
                
                if ($action === 'block') {
                    handleBlockUserFromPanel($chat_id, $user_id, $target_user_id);
                } else {
                    handleUnblockUserFromPanel($chat_id, $user_id, $target_user_id);
                }
                
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'managing_vip') {
                // Handle VIP management
                require_once "inc/handlers/user-handler.php";
                
                $action = $state_data['action'] ?? '';
                $target_user_id = extractUserIdFromMessage($message);
                
                if ($target_user_id === false) {
                    sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
                    exit;
                }
                
                if ($action === 'set') {
                    if (setUserVIP($target_user_id)) {
                        sendMessage($chat_id, "✅ کاربر با موفقیت VIP شد!");
                    } else {
                        sendMessage($chat_id, "❌ خطا در VIP کردن کاربر.");
                    }
                } else {
                    if (removeUserVIP($target_user_id)) {
                        sendMessage($chat_id, "✅ VIP کاربر با موفقیت حذف شد!");
                    } else {
                        sendMessage($chat_id, "❌ خطا در حذف VIP کاربر.");
                    }
                }
                
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'managing_admin') {
                // Handle admin management (add/remove)
                require_once "inc/handlers/admin-handler.php";
                
                $action = $state_data['action'] ?? '';
                $target_user_id = extractUserIdFromMessage($message);
                
                if ($target_user_id === false) {
                    sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
                    exit;
                }
                
                if ($action === 'add') {
                    // Check if user is main admin
                    if (!isMainAdmin($user_id)) {
                        sendMessage($chat_id, "❌ فقط ادمین اصلی می‌تواند ادمین اضافه کند.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    // Add admin
                    $result = addAdmin($target_user_id, $user_id);
                    
                    if ($result) {
                        sendMessage($chat_id, "✅ ادمین با موفقیت اضافه شد!");
                        logInfo("Admin added: {$target_user_id} by {$user_id}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در افزودن ادمین. ممکن است این کاربر قبلاً ادمین باشد.");
                    }
                } elseif ($action === 'remove') {
                    // Check if user is main admin
                    if (!isMainAdmin($user_id)) {
                        sendMessage($chat_id, "❌ فقط ادمین اصلی می‌تواند ادمین حذف کند.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    // Cannot remove main admin
                    if (isMainAdmin($target_user_id)) {
                        sendMessage($chat_id, "❌ نمی‌توانید ادمین اصلی را حذف کنید.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    // Remove admin
                    $result = removeAdmin($target_user_id);
                    
                    if ($result) {
                        sendMessage($chat_id, "✅ ادمین با موفقیت حذف شد!");
                        logInfo("Admin removed: {$target_user_id} by {$user_id}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در حذف ادمین.");
                    }
                }
                
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'managing_channel') {
                // Handle channel management (add/remove)
                require_once "inc/handlers/channel-handler.php";
                
                $action = $state_data['action'] ?? '';
                $channel_info = extractChannelInfo($message);
                
                if ($channel_info === false) {
                    sendMessage($chat_id, "❌ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.");
                    exit;
                }
                
                if ($action === 'add') {
                    // Check if channel already exists (by channel_id or username if channel_id is 0)
                    if ($channel_info['channel_id'] != 0) {
                        $existing = dbQueryOne(
                            "SELECT id FROM channels WHERE channel_id = ?",
                            [$channel_info['channel_id']]
                        );
                    } else {
                        // If channel_id is 0, check by username
                        $existing = dbQueryOne(
                            "SELECT id FROM channels WHERE channel_username = ?",
                            [$channel_info['channel_username']]
                        );
                    }
                    
                    if ($existing !== false) {
                        sendMessage($chat_id, "❌ این کانال قبلاً اضافه شده است.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    // Add channel
                    $result = dbExecute(
                        "INSERT INTO channels (channel_id, channel_username, channel_title, channel_link) 
                         VALUES (?, ?, ?, ?)",
                        [
                            $channel_info['channel_id'],
                            $channel_info['channel_username'],
                            $channel_info['channel_title'],
                            $channel_info['channel_link']
                        ]
                    );
                    
                    if ($result !== false) {
                        sendMessage($chat_id, "✅ کانال با موفقیت اضافه شد!");
                        logInfo("Channel added: {$channel_info['channel_id']} ({$channel_info['channel_title']}) by {$user_id}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در افزودن کانال.");
                    }
                } elseif ($action === 'remove') {
                    // Remove channel (by channel_id or username)
                    if ($channel_info['channel_id'] != 0) {
                        $result = dbExecute(
                            "DELETE FROM channels WHERE channel_id = ?",
                            [$channel_info['channel_id']]
                        );
                    } else {
                        $result = dbExecute(
                            "DELETE FROM channels WHERE channel_username = ?",
                            [$channel_info['channel_username']]
                        );
                    }
                    
                    if ($result !== false) {
                        sendMessage($chat_id, "✅ کانال با موفقیت حذف شد!");
                        logInfo("Channel removed: {$channel_info['channel_id']} ({$channel_info['channel_title']}) by {$user_id}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در حذف کانال.");
                    }
                }
                
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'managing_reaction_channel') {
                // Handle reaction channel management (add/remove)
                require_once "inc/handlers/channel-handler.php";
                
                $action = $state_data['action'] ?? '';
                $channel_info = extractChannelInfo($message);
                
                if ($channel_info === false) {
                    sendMessage($chat_id, "❌ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.");
                    exit;
                }
                
                if ($action === 'add') {
                    // Add reaction channel
                    $result = addReactionChannel(
                        $channel_info['channel_id'],
                        $channel_info['channel_title'],
                        $channel_info['channel_username'],
                        $channel_info['channel_link']
                    );
                    
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    
                    if ($result) {
                        sendMessage($chat_id, "✅ کانال ری‌اکشن با موفقیت اضافه شد!");
                    } else {
                        sendMessage($chat_id, "❌ خطا در افزودن کانال ری‌اکشن.");
                    }
                    exit;
                } elseif ($action === 'remove') {
                    // Remove reaction channel
                    $channel_id = $channel_info['channel_id'];
                    if ($channel_id == 0 && !empty($channel_info['channel_username'])) {
                        // Try to find by username
                        $found = dbQueryOne(
                            "SELECT channel_id FROM reaction_channels WHERE channel_username = ?",
                            [$channel_info['channel_username']]
                        );
                        if ($found !== false) {
                            $channel_id = $found['channel_id'];
                        }
                    }
                    
                    if ($channel_id == 0) {
                        sendMessage($chat_id, "❌ کانال یافت نشد.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    $result = removeReactionChannel($channel_id);
                    
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    
                    if ($result) {
                        sendMessage($chat_id, "✅ کانال ری‌اکشن با موفقیت حذف شد!");
                    } else {
                        sendMessage($chat_id, "❌ خطا در حذف کانال ری‌اکشن.");
                    }
                    exit;
                }
            } elseif ($user_state['state'] === 'creating_backup') {
                // Handle backup creation
                require_once "inc/handlers/backup-handler.php";
                
                $step = $state_data['step'] ?? 'bot_link';
                
                if ($step === 'bot_link') {
                    // Get bot link
                    $bot_link = trim($text);
                    
                    // Validate URL
                    if (empty($bot_link) || !filter_var($bot_link, FILTER_VALIDATE_URL)) {
                        sendMessage($chat_id, "❌ لینک نامعتبر است. لطفاً یک URL معتبر ارسال کنید.\n\n💡 مثال: https://t.me/example_bot");
                        exit;
                    }
                    
                    // Save bot link to state
                    $state_data['bot_link'] = $bot_link;
                    $state_data['step'] = 'text';
                    dbExecute(
                        "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                        [json_encode($state_data), $user_id]
                    );
                    
                    sendMessage($chat_id, "✅ لینک ربات دریافت شد.\n\n📝 لطفاً متن را ارسال کنید (یا /skip برای رد کردن):");
                    exit;
                } elseif ($step === 'text') {
                    // Get text (optional)
                    $backup_text = '';
                    if (strtolower(trim($text)) !== '/skip' && !empty(trim($text))) {
                        $backup_text = trim($text);
                    }
                    
                    $bot_link = $state_data['bot_link'] ?? '';
                    
                    if (empty($bot_link)) {
                        sendMessage($chat_id, "❌ خطا: لینک ربات یافت نشد. لطفاً دوباره شروع کنید.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    // Create backup
                    $file_id = createBackup($bot_link, $backup_text, $user_id);
                    
                    if ($file_id === false) {
                        sendMessage($chat_id, "❌ خطا در ایجاد بک آپ.");
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                        exit;
                    }
                    
                    // Get file code for download link
                    require_once "inc/handlers/upload-handler.php";
                    $file = getFileById($file_id);
                    $file_code = $file['file_code'] ?? '';
                    
                    // Generate download link
                    require_once "inc/handlers/upload-handler.php";
                    $bot_username = BOT_USERNAME ?? '';
                    if (empty($bot_username)) {
                        $bot_info = bot('getMe', []);
                        if ($bot_info !== false && isset($bot_info['ok']) && $bot_info['ok']) {
                            $bot_username = $bot_info['result']['username'] ?? '';
                        }
                    }
                    
                    $download_link = '';
                    if (!empty($bot_username) && !empty($file_code)) {
                        $download_link = "https://t.me/{$bot_username}?start={$file_code}";
                    }
                    
                    $msg = "✅ بک آپ با موفقیت ایجاد شد!\n\n";
                    $msg .= "🆔 شناسه: {$file_id}\n";
                    if (!empty($download_link)) {
                        $msg .= "\n🔗 لینک دانلود:\n{$download_link}";
                    }
                    
                    sendMessage($chat_id, $msg);
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    exit;
                }
            } elseif ($user_state['state'] === 'setting_advanced') {
                // Handle advanced settings
                require_once "inc/handlers/settings-handler.php";
                
                $type = $state_data['type'] ?? '';
                
                if ($type === 'stars_rate') {
                    $rate = (int)trim($text);
                    handleSetStarsRateFromPanel($chat_id, $user_id, $rate);
                } elseif ($type === 'referral_points') {
                    $parts = explode('|', $text);
                    if (count($parts) >= 2) {
                        $referrer_points = (int)trim($parts[0]);
                        $referred_points = (int)trim($parts[1]);
                        handleSetReferralPointsFromPanel($chat_id, $user_id, $referrer_points, $referred_points);
                    } else {
                        sendMessage($chat_id, "❌ فرمت اشتباه است.\n\nفرمت صحیح:\nمعرف|معرف‌شده");
                        exit;
                    }
                } elseif ($type === 'referral_multilevel_points') {
                    // Format: level1|level2|level3
                    $parts = explode('|', trim($text));
                    if (count($parts) !== 3) {
                        sendMessage($chat_id, "❌ فرمت نامعتبر. لطفاً به فرمت زیر وارد کنید:\n\nسطح1|سطح2|سطح3\n\nمثال:\n5|3|1");
                        exit;
                    }
                    
                    $level1_points = (int)trim($parts[0]);
                    $level2_points = (int)trim($parts[1]);
                    $level3_points = (int)trim($parts[2]);
                    
                    if ($level1_points < 0 || $level2_points < 0 || $level3_points < 0) {
                        sendMessage($chat_id, "❌ امتیازها نمی‌توانند منفی باشند.");
                        exit;
                    }
                    
                    if (setSettingValue('referral_points_level_1', $level1_points) &&
                        setSettingValue('referral_points_level_2', $level2_points) &&
                        setSettingValue('referral_points_level_3', $level3_points)) {
                        sendMessage($chat_id, "✅ امتیازهای رفرال چند سطحی به‌روزرسانی شد!\n\n🔹 سطح 1: {$level1_points}\n🔹 سطح 2: {$level2_points}\n🔹 سطح 3: {$level3_points}");
                        logInfo("Admin {$user_id} set referral multilevel points: level1={$level1_points}, level2={$level2_points}, level3={$level3_points}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در تنظیم امتیازها.");
                    }
                } elseif ($type === 'forced_join_text') {
                    handleSetForcedJoinTextFromPanel($chat_id, $user_id, $text);
                } elseif ($type === 'gambling_min_bet') {
                    $min_bet = (int)trim($text);
                    if ($min_bet < 1) {
                        sendMessage($chat_id, "❌ حداقل شرط باید بزرگ‌تر از صفر باشد.");
                        exit;
                    }
                    
                    if (setSettingValue('gambling_min_bet', $min_bet)) {
                        sendMessage($chat_id, "✅ حداقل شرط به {$min_bet} امتیاز تنظیم شد!");
                        logInfo("Admin {$user_id} set gambling min bet to {$min_bet}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در تنظیم حداقل شرط.");
                    }
                } elseif ($type === 'gambling_max_bet') {
                    $max_bet = (int)trim($text);
                    if ($max_bet < 1) {
                        sendMessage($chat_id, "❌ حداکثر شرط باید بزرگ‌تر از صفر باشد.");
                        exit;
                    }
                    
                    $min_bet = (int)getSettingValue('gambling_min_bet', 1);
                    if ($max_bet < $min_bet) {
                        sendMessage($chat_id, "❌ حداکثر شرط باید بزرگ‌تر یا مساوی حداقل شرط ({$min_bet}) باشد.");
                        exit;
                    }
                    
                    if (setSettingValue('gambling_max_bet', $max_bet)) {
                        sendMessage($chat_id, "✅ حداکثر شرط به {$max_bet} امتیاز تنظیم شد!");
                        logInfo("Admin {$user_id} set gambling max bet to {$max_bet}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در تنظیم حداکثر شرط.");
                    }
                } elseif ($type === 'gambling_multiplier') {
                    $multiplier = (float)trim($text);
                    if ($multiplier <= 0) {
                        sendMessage($chat_id, "❌ نرخ برد باید بزرگ‌تر از صفر باشد.");
                        exit;
                    }
                    
                    if (setSettingValue('gambling_win_multiplier', $multiplier)) {
                        sendMessage($chat_id, "✅ نرخ برد به {$multiplier}x تنظیم شد!");
                        logInfo("Admin {$user_id} set gambling multiplier to {$multiplier}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در تنظیم نرخ برد.");
                    }
                } elseif ($type === 'gambling_daily_limit') {
                    $daily_limit = (int)trim($text);
                    if ($daily_limit < 0) {
                        sendMessage($chat_id, "❌ محدودیت روزانه نمی‌تواند منفی باشد.");
                        exit;
                    }
                    
                    if (setSettingValue('gambling_daily_limit', $daily_limit)) {
                        $msg = $daily_limit > 0 
                            ? "✅ محدودیت روزانه به {$daily_limit} بازی تنظیم شد!"
                            : "✅ محدودیت روزانه حذف شد (بدون محدودیت)";
                        sendMessage($chat_id, $msg);
                        logInfo("Admin {$user_id} set gambling daily limit to {$daily_limit}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در تنظیم محدودیت روزانه.");
                    }
                } elseif ($type === 'gambling_cooldown') {
                    $cooldown = (int)trim($text);
                    if ($cooldown < 0) {
                        sendMessage($chat_id, "❌ زمان انتظار نمی‌تواند منفی باشد.");
                        exit;
                    }
                    
                    if (setSettingValue('gambling_cooldown_seconds', $cooldown)) {
                        $msg = $cooldown > 0 
                            ? "✅ زمان انتظار به {$cooldown} ثانیه تنظیم شد!"
                            : "✅ زمان انتظار حذف شد (بدون انتظار)";
                        sendMessage($chat_id, $msg);
                        logInfo("Admin {$user_id} set gambling cooldown to {$cooldown} seconds");
                    } else {
                        sendMessage($chat_id, "❌ خطا در تنظیم زمان انتظار.");
                    }
                }
                
                dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                exit;
            } elseif ($user_state['state'] === 'searching_user') {
                // Handle user search
                require_once "inc/handlers/user-handler.php";
                
                $users = searchUsers($text);
                
                if ($users === false || empty($users)) {
                    sendMessage($chat_id, "❌ کاربری یافت نشد.");
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    exit;
                }
                
                if (count($users) === 1) {
                    // Show detailed info for single result
                    $target_user_id = (int)$users[0]['id'];
                    // Save target user_id to state for management buttons
                    $state_data = json_encode(['target_user_id' => $target_user_id]);
                    dbExecute(
                        "UPDATE user_states SET state = 'viewing_user', state_data = ? WHERE user_id = ?",
                        [$state_data, $user_id]
                    );
                    displayUserInfo($chat_id, $target_user_id, null, true);
                } else {
                    // Show list of results
                    $message = "🔍 نتایج جستجو:\n\n";
                    
                    foreach ($users as $index => $user) {
                        $name = $user['first_name'] . ($user['last_name'] ? ' ' . $user['last_name'] : '');
                        $username = $user['username'] ? '@' . $user['username'] : 'بدون یوزرنیم';
                        
                        $message .= ($index + 1) . ". {$name}\n";
                        $message .= "   🆔 {$user['id']}\n";
                        $message .= "   📱 {$username}\n\n";
                    }
                    
                    sendMessage($chat_id, $message);
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                }
                exit;
            } elseif ($user_state['state'] === 'viewing_user') {
                // User is viewing a user's info, handle management actions
                $target_user_id = isset($state_data['target_user_id']) ? (int)$state_data['target_user_id'] : null;
                
                if ($target_user_id === null) {
                    // If no target user, treat as new search
                    dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    // Fall through to search
                    require_once "inc/handlers/user-handler.php";
                    $users = searchUsers($text);
                    
                    if ($users === false || empty($users)) {
                        sendMessage($chat_id, "❌ کاربری یافت نشد.");
                        exit;
                    }
                    
                    if (count($users) === 1) {
                        $target_user_id = (int)$users[0]['id'];
                        $state_data = json_encode(['target_user_id' => $target_user_id]);
                        dbExecute(
                            "UPDATE user_states SET state_data = ? WHERE user_id = ?",
                            [$state_data, $user_id]
                        );
                        displayUserInfo($chat_id, $target_user_id, null, true);
                    } else {
                        $message = "🔍 نتایج جستجو:\n\n";
                        foreach ($users as $index => $user) {
                            $name = $user['first_name'] . ($user['last_name'] ? ' ' . $user['last_name'] : '');
                            $username = $user['username'] ? '@' . $user['username'] : 'بدون یوزرنیم';
                            $message .= ($index + 1) . ". {$name}\n   🆔 {$user['id']}\n   📱 {$username}\n\n";
                        }
                        sendMessage($chat_id, $message);
                        dbExecute("DELETE FROM user_states WHERE user_id = ?", [$user_id]);
                    }
                    exit;
                }
            }
        }
    }
    
    // Handle user promo code usage (if text looks like a promo code)
    // Allow both regular users and admins in user mode to use promo codes
    if ($text && preg_match('/^[A-Z0-9]{3,20}$/i', trim($text))) {
        // Check if user is admin but not in user mode - admins can't use promo codes unless in user mode
        if (isAdmin($user_id) && !isInUserMode($user_id)) {
            // Admin not in user mode - don't process as promo code
        } else {
            // Regular user or admin in user mode - process promo code
            require_once "inc/handlers/promo-code-handler.php";
            $result = usePromoCode($user_id, trim($text));
            sendMessage($chat_id, $result['message']);
            exit;
        }
    }
    
    // Handle regular messages for non-admin users (or admins in user mode)
    // Only show message if it wasn't processed as promo code
    if ($text && !isAdmin($user_id)) {
        // Check forced join first
        require_once "inc/handlers/channel-handler.php";
        if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
            exit; // Message already sent
        }
        
        // Check if it's a real admin in user mode
        if (isRealAdmin($user_id) && isInUserMode($user_id)) {
            // Show user menu with exit option
            $keyboard = getInlineKeyboard('user_menu', 1, ['user_id' => $user_id]);
            sendMessage($chat_id, "👋 برای استفاده از ربات، از دستورات استفاده کنید.\n\n💎 امتیاز شما: " . getUserPoints($user_id), $keyboard);
        } else {
            sendMessage($chat_id, "👋 برای استفاده از ربات، از دستورات استفاده کنید.\n\n💎 امتیاز شما: " . getUserPoints($user_id) . "\n\n🎁 برای استفاده از کد جایزه/تخفیف/سکه، کد را ارسال کنید.", getInlineKeyboard('user_menu'));
        }
    } elseif ($text && isAdmin($user_id) && isInUserMode($user_id)) {
        // Check forced join first
        require_once "inc/handlers/channel-handler.php";
        if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
            exit; // Message already sent
        }
        
        // Admin in user mode - show user menu with exit option
        $keyboard = getInlineKeyboard('user_menu', 1, ['user_id' => $user_id]);
        sendMessage($chat_id, "👋 برای استفاده از ربات، از دستورات استفاده کنید.\n\n💎 امتیاز شما: " . getUserPoints($user_id) . "\n\n🎁 برای استفاده از کد جایزه/تخفیف/سکه، کد را ارسال کنید.", $keyboard);
    }
}
