<?php
declare(strict_types=1);

/**
 * Authentication and Authorization Functions
 * Manages admin authentication and authorization checks
 */

require_once __DIR__ . '/database.php';

/**
 * Check if user is in user mode (admin testing user interface)
 * @param int $user_id
 * @return bool
 */
function isInUserMode(int $user_id): bool {
    $user_state = dbQueryOne(
        "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'user_mode'",
        [$user_id]
    );
    
    if ($user_state === false || empty($user_state['state_data'])) {
        return false;
    }
    
    $state_data = json_decode($user_state['state_data'], true);
    return isset($state_data['enabled']) && $state_data['enabled'] === true;
}

/**
 * Check if user is admin (excluding user mode)
 * @param int $user_id
 * @return bool
 */
function isRealAdmin(int $user_id): bool {
    // Check if user is main admin
    if (defined('MAIN_ADMIN_ID') && $user_id == MAIN_ADMIN_ID) {
        return true;
    }
    
    // Check if user is in admins table
    $result = dbQueryOne(
        "SELECT id FROM admins WHERE id = ?",
        [$user_id]
    );
    
    return $result !== false;
}

/**
 * Check if user is admin (respects user mode)
 * @param int $user_id
 * @return bool
 */
function isAdmin(int $user_id): bool {
    // If user is in user mode, treat as regular user
    if (isInUserMode($user_id)) {
        return false;
    }
    
    return isRealAdmin($user_id);
}

/**
 * Check if user is main admin
 * @param int $user_id
 * @return bool
 */
function isMainAdmin(int $user_id): bool {
    return defined('MAIN_ADMIN_ID') && $user_id == MAIN_ADMIN_ID;
}

/**
 * Add admin
 * @param int $admin_id User ID to add as admin
 * @param int $added_by User ID who is adding the admin
 * @return bool
 */
function addAdmin(int $admin_id, int $added_by): bool {
    // Check if already admin
    if (isAdmin($admin_id)) {
        return false;
    }
    
    $result = dbExecute(
        "INSERT INTO admins (id, added_by) VALUES (?, ?)",
        [$admin_id, $added_by]
    );
    
    return $result !== false;
}

/**
 * Remove admin
 * @param int $admin_id User ID to remove from admins
 * @return bool
 */
function removeAdmin(int $admin_id): bool {
    // Cannot remove main admin
    if (defined('MAIN_ADMIN_ID') && $admin_id == MAIN_ADMIN_ID) {
        return false;
    }
    
    $result = dbExecute(
        "DELETE FROM admins WHERE id = ?",
        [$admin_id]
    );
    
    return $result !== false;
}

/**
 * Get list of all admins
 * @return array
 */
function getAllAdmins(): array {
    $admins = [];
    
    // Add main admin
    if (defined('MAIN_ADMIN_ID')) {
        $admins[] = [
            'id' => MAIN_ADMIN_ID,
            'is_main' => true
        ];
    }
    
    // Get other admins from database
    $dbAdmins = dbQuery("SELECT id, added_by, created_at FROM admins ORDER BY created_at DESC");
    
    if ($dbAdmins !== false) {
        foreach ($dbAdmins as $dbAdmin) {
            $admins[] = [
                'id' => (int)$dbAdmin['id'],
                'is_main' => false,
                'added_by' => (int)$dbAdmin['added_by'],
                'created_at' => $dbAdmin['created_at']
            ];
        }
    }
    
    return $admins;
}

/**
 * Check if user is blocked
 * @param int $user_id
 * @return bool
 */
function isUserBlocked(int $user_id): bool {
    $user = dbQueryOne(
        "SELECT is_blocked FROM users WHERE id = ?",
        [$user_id]
    );
    
    if ($user === false) {
        return false;
    }
    
    return (bool)$user['is_blocked'];
}

/**
 * Block user
 * @param int $user_id
 * @return bool
 */
function blockUser(int $user_id): bool {
    // Ensure user exists
    ensureUserExists($user_id);
    
    $result = dbExecute(
        "UPDATE users SET is_blocked = 1 WHERE id = ?",
        [$user_id]
    );
    
    return $result !== false;
}

/**
 * Unblock user
 * @param int $user_id
 * @return bool
 */
function unblockUser(int $user_id): bool {
    $result = dbExecute(
        "UPDATE users SET is_blocked = 0 WHERE id = ?",
        [$user_id]
    );
    
    return $result !== false;
}

/**
 * Ensure user exists in database
 * @param int $user_id
 * @param array $user_data Optional user data from Telegram
 * @return bool
 */
function ensureUserExists(int $user_id, ?array $user_data = null): bool {
    $existing = dbQueryOne("SELECT id FROM users WHERE id = ?", [$user_id]);
    
    if ($existing !== false) {
        // Update user data if provided
        if ($user_data !== null) {
            $username = $user_data['username'] ?? null;
            $first_name = $user_data['first_name'] ?? '';
            $last_name = $user_data['last_name'] ?? null;
            
            dbExecute(
                "UPDATE users SET username = ?, first_name = ?, last_name = ? WHERE id = ?",
                [$username, $first_name, $last_name, $user_id]
            );
        }
        return true;
    }
    
    // Create new user
    $username = $user_data['username'] ?? null;
    $first_name = $user_data['first_name'] ?? 'Unknown';
    $last_name = $user_data['last_name'] ?? null;
    
    $result = dbExecute(
        "INSERT INTO users (id, username, first_name, last_name) VALUES (?, ?, ?, ?)",
        [$user_id, $username, $first_name, $last_name]
    );
    
    // Initialize points for new user
    if ($result !== false) {
        dbExecute(
            "INSERT INTO points (user_id, points) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id",
            [$user_id]
        );
    }
    
    return $result !== false;
}

