<?php
declare(strict_types=1);

/**
 * Database Connection and Helper Functions
 * Manages MySQL database connection and provides helper functions for queries
 */

// Database connection variable
$db = null;

/**
 * Initialize database connection
 * @return PDO|null
 */
function initDatabase(): ?PDO {
    global $db;
    
    if ($db !== null) {
        return $db;
    }
    
    try {
        $host = DB_HOST ?? 'localhost';
        $dbname = DB_NAME ?? 'telegram_bot';
        $username = DB_USER ?? 'root';
        $password = DB_PASS ?? '';
        $charset = DB_CHARSET ?? 'utf8mb4';
        
        $dsn = "mysql:host={$host};dbname={$dbname};charset={$charset}";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, $username, $password, $options);
        return $db;
    } catch (PDOException $e) {
        logError("Database connection failed: " . $e->getMessage());
        return null;
    }
}

/**
 * Execute a query and return results
 * @param string $query SQL query
 * @param array $params Parameters for prepared statement
 * @return array|false
 */
function dbQuery(string $query, array $params = []): array|false {
    $db = initDatabase();
    if (!$db) {
        return false;
    }
    
    try {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        logError("Query failed: " . $e->getMessage() . " | Query: " . $query);
        return false;
    }
}

/**
 * Execute a query and return single row
 * @param string $query SQL query
 * @param array $params Parameters for prepared statement
 * @return array|false
 */
function dbQueryOne(string $query, array $params = []): array|false {
    $db = initDatabase();
    if (!$db) {
        return false;
    }
    
    try {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        $result = $stmt->fetch();
        return $result ?: false;
    } catch (PDOException $e) {
        logError("Query failed: " . $e->getMessage() . " | Query: " . $query);
        return false;
    }
}

/**
 * Execute a query and return affected rows count
 * @param string $query SQL query
 * @param array $params Parameters for prepared statement
 * @return int|false
 */
function dbExecute(string $query, array $params = []): int|false {
    $db = initDatabase();
    if (!$db) {
        return false;
    }
    
    try {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        return $stmt->rowCount();
    } catch (PDOException $e) {
        logError("Execute failed: " . $e->getMessage() . " | Query: " . $query);
        return false;
    }
}

/**
 * Get last insert ID
 * @return string|false
 */
function dbLastInsertId(): string|false {
    $db = initDatabase();
    if (!$db) {
        return false;
    }
    
    return $db->lastInsertId();
}

/**
 * Begin transaction
 * @return bool
 */
function dbBeginTransaction(): bool {
    $db = initDatabase();
    if (!$db) {
        return false;
    }
    
    return $db->beginTransaction();
}

/**
 * Commit transaction
 * @return bool
 */
function dbCommit(): bool {
    $db = initDatabase();
    if (!$db) {
        return false;
    }
    
    return $db->commit();
}

/**
 * Rollback transaction
 * @return bool
 */
function dbRollback(): bool {
    $db = initDatabase();
    if (!$db) {
        return false;
    }
    
    // Check if there's an active transaction before rolling back
    if ($db->inTransaction()) {
        return $db->rollBack();
    }
    
    return true; // No transaction to rollback
}

