<?php
declare(strict_types=1);

// bot function
function bot($method, $data) {
    $url = "https://api.telegram.org/bot" . API_KEY . "/" . $method;

    // Check if reply_markup is a JSON string - if so, handle it specially
    $post_data = [];
    foreach ($data as $key => $value) {
        if ($key === 'reply_markup' && is_string($value)) {
            // reply_markup is already a JSON string, use it as-is
            $post_data[$key] = $value;
        } else {
            $post_data[$key] = $value;
        }
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post_data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

    $result = curl_exec($ch);

    // Check for cURL errors
    if ($result === false) {
        $error = curl_error($ch);
        error_log("cURL Error: " . $error);
        curl_close($ch);
        return false;
    }

    curl_close($ch);

    $decoded = json_decode($result, true);
    
    // Log errors for debugging (but ignore "message is not modified" as it's not a real error)
    if ($decoded !== null && isset($decoded['ok']) && !$decoded['ok']) {
        $error_description = $decoded['description'] ?? '';
        // Don't log "message is not modified" errors as they're not real errors
        if (strpos($error_description, 'message is not modified') === false) {
            error_log("Telegram API Error for method {$method}: " . $error_description);
        }
    }

    return $decoded;
}

// Send or edit message function (smart - tries to edit if message_id exists in state)
function sendOrEditMessage($chat_id, $text, $keyboard = null, $reply_to = null, $user_id = null) {
    // Try to get message_id from user state if user_id provided
    if ($user_id !== null) {
        $user_state = dbQueryOne(
            "SELECT state_data FROM user_states WHERE user_id = ? AND state IS NOT NULL",
            [$user_id]
        );
        
        if ($user_state !== false && !empty($user_state['state_data'])) {
            $state_data = json_decode($user_state['state_data'], true);
            if ($state_data !== null && isset($state_data['last_message_id'])) {
                $message_id = (int)$state_data['last_message_id'];
                $edit_result = editMessage($chat_id, $message_id, $text, $keyboard);
                if ($edit_result !== false && isset($edit_result['ok']) && $edit_result['ok']) {
                    return $edit_result; // Successfully edited
                }
            }
        }
    }
    
    // If edit failed or no state, send new message
    $result = sendMessage($chat_id, $text, $keyboard, $reply_to);
    
    // Save message_id to state if successful and user_id provided
    if ($result !== false && isset($result['ok']) && $result['ok'] && $user_id !== null) {
        $sent_message_id = (int)$result['result']['message_id'];
        $state_data = json_encode(['last_message_id' => $sent_message_id]);
        dbExecute(
            "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'active', ?) 
             ON DUPLICATE KEY UPDATE state_data = ?",
            [$user_id, $state_data, $state_data]
        );
    }
    
    return $result;
}

// Send message function
function sendMessage($chat_id, $text, $keyboard = null, $reply_to = null, $disable_web_page_preview = false) {
    $data = [
        'chat_id' => $chat_id,
        'text' => $text
    ];

    // Always remove reply keyboard first
    $data['reply_markup'] = json_encode(['remove_keyboard' => true, 'selective' => false]);

    // Add inline keyboard if provided
    if ($keyboard) {
        $data['reply_markup'] = json_encode($keyboard);
    }

    // Add reply to message if provided
    if ($reply_to) {
        $data['reply_to_message_id'] = $reply_to;
    }

    // Disable web page preview if requested
    if ($disable_web_page_preview) {
        $data['disable_web_page_preview'] = true;
    }

    return bot('sendMessage', $data);
}

// Edit message text and keyboard
function editMessage($chat_id, $message_id, $text, $keyboard = null) {
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text
    ];

    // Add keyboard if provided
    if ($keyboard) {
        $data['reply_markup'] = json_encode($keyboard, JSON_UNESCAPED_UNICODE);
    } else {
        // If no keyboard provided, remove keyboard
        $data['reply_markup'] = json_encode(['inline_keyboard' => []], JSON_UNESCAPED_UNICODE);
    }

    $result = bot('editMessageText', $data);
    
    // Ignore "message is not modified" error - it's not a real error
    if ($result !== false && isset($result['ok']) && !$result['ok']) {
        $error_description = $result['description'] ?? '';
        if (strpos($error_description, 'message is not modified') !== false) {
            // This is not a real error - message content is the same, which is fine
            // Return success-like response
            return ['ok' => true, 'result' => ['message_id' => $message_id]];
        }
    }
    
    return $result;
}

// Get start parameter from deep link
function getStartParameter($text) {
    if (strpos($text, '/start') === 0) {
        $parts = explode(' ', $text, 2);
        if (isset($parts[1]) && !empty($parts[1])) {
            return trim($parts[1]);
        }
    }
    return null;
}

// Send user info to admin when user starts with deep link
function sendUserInfoToAdmin($admin_id, $user_data, $start_param) {
    $username = $user_data['username'] ?? 'بدون یوزرنیم';
    $first_name = $user_data['first_name'] ?? 'بدون نام';
    $last_name = $user_data['last_name'] ?? '';
    $user_id = $user_data['id'];
    $full_name = $first_name . ($last_name ? ' ' . $last_name : '');

    $message = "🔔 کاربر جدید با Deep Link\n\n";
    $message .= "👤 نام: {$full_name}\n";
    $message .= "🆔 شناسه: {$user_id}\n";
    $message .= "📱 یوزرنیم: @{$username}\n";
    $message .= "🔗 پارامتر: {$start_param}";

    return sendMessage($admin_id, $message);
}