<?php
declare(strict_types=1);

/**
 * Advertisement Handler
 * Handles advertisement management
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../logger.php';

/**
 * Add advertisement
 * @param array $message Admin's message containing ad
 * @param string $title Ad title
 * @param string|null $caption Ad caption
 * @param string|null $ad_url Advertisement URL
 * @param string $button_text Button text (default: 'خرید تبلیغات')
 * @return int|false Advertisement ID
 */
function addAdvertisement(array $message, string $title, ?string $caption = null, ?string $ad_url = null, string $button_text = 'خرید تبلیغات'): int|false {
    $content = $message['text'] ?? '';
    $media_type = null;
    $media_file_id = null;
    
    // Extract media if exists
    if (isset($message['photo'])) {
        $photos = is_array($message['photo']) ? $message['photo'] : [$message['photo']];
        $largest_photo = end($photos);
        $media_type = 'photo';
        $media_file_id = $largest_photo['file_id'];
        $content = $message['caption'] ?? $content;
    } elseif (isset($message['video'])) {
        $media_type = 'video';
        $media_file_id = $message['video']['file_id'];
        $content = $message['caption'] ?? $content;
    } elseif (isset($message['document'])) {
        $media_type = 'document';
        $media_file_id = $message['document']['file_id'];
        $content = $message['caption'] ?? $content;
    } elseif (isset($message['animation'])) {
        $media_type = 'animation';
        $media_file_id = $message['animation']['file_id'];
        $content = $message['caption'] ?? $content;
    }
    
    $result = dbExecute(
        "INSERT INTO advertisements (title, content, caption, media_type, media_file_id, ad_url, button_text) 
         VALUES (?, ?, ?, ?, ?, ?, ?)",
        [$title, $content, $caption, $media_type, $media_file_id, $ad_url, $button_text]
    );
    
    if ($result === false) {
        return false;
    }
    
    $ad_id = dbLastInsertId();
    return $ad_id ? (int)$ad_id : false;
}

/**
 * Delete advertisement
 * @param int $ad_id
 * @return bool
 */
function deleteAdvertisement(int $ad_id): bool {
    $result = dbExecute(
        "DELETE FROM advertisements WHERE id = ?",
        [$ad_id]
    );
    
    return $result !== false;
}

/**
 * Get active advertisements
 * @return array|false
 */
function getActiveAdvertisements(): array|false {
    return dbQuery(
        "SELECT * FROM advertisements WHERE is_active = 1 ORDER BY created_at DESC"
    );
}

/**
 * Get advertisement by ID
 * @param int $ad_id
 * @return array|false
 */
function getAdvertisementById(int $ad_id): array|false {
    return dbQueryOne(
        "SELECT * FROM advertisements WHERE id = ?",
        [$ad_id]
    );
}

/**
 * Toggle advertisement active status
 * @param int $ad_id
 * @return bool
 */
function toggleAdvertisement(int $ad_id): bool {
    $ad = getAdvertisementById($ad_id);
    
    if ($ad === false) {
        return false;
    }
    
    $new_status = $ad['is_active'] ? 0 : 1;
    
    $result = dbExecute(
        "UPDATE advertisements SET is_active = ? WHERE id = ?",
        [$new_status, $ad_id]
    );
    
    return $result !== false;
}

/**
 * Show advertisement to user
 * @param int $user_id
 * @return bool
 */
function showAdvertisementToUser(int $user_id): bool {
    require_once __DIR__ . '/settings-handler.php';
    
    $ads = getActiveAdvertisements();
    
    if ($ads === false || empty($ads)) {
        return false;
    }
    
    // Get random ad or first ad
    $ad = $ads[array_rand($ads)];
    
    // Check if mandatory seen is enabled
    $mandatory_seen_enabled = isMandatorySeenEnabled();
    
    // Build caption with button
    $caption = $ad['caption'] ?? $ad['content'] ?? '';
    
    // Build inline keyboard with button if URL exists
    $reply_markup = null;
    if (!empty($ad['ad_url'])) {
        $button_text = $ad['button_text'] ?? 'خرید تبلیغات';
        $reply_markup = json_encode([
            'inline_keyboard' => [
                [
                    [
                        'text' => $button_text,
                        'url' => $ad['ad_url']
                    ]
                ]
            ]
        ], JSON_UNESCAPED_UNICODE);
    }
    
    // Send ad to user
    $result = false;
    
    if ($ad['media_type'] === 'photo' && $ad['media_file_id']) {
        $data = [
            'chat_id' => $user_id,
            'photo' => $ad['media_file_id']
        ];
        if (!empty($caption)) {
            $data['caption'] = $caption;
        }
        if ($reply_markup !== null) {
            $data['reply_markup'] = $reply_markup;
        }
        // Add mandatory seen effect if enabled
        if ($mandatory_seen_enabled) {
            $data['message_effect_id'] = '👀';
        }
        $result = bot('sendPhoto', $data);
    } elseif ($ad['media_type'] === 'video' && $ad['media_file_id']) {
        $data = [
            'chat_id' => $user_id,
            'video' => $ad['media_file_id']
        ];
        if (!empty($caption)) {
            $data['caption'] = $caption;
        }
        if ($reply_markup !== null) {
            $data['reply_markup'] = $reply_markup;
        }
        // Add mandatory seen effect if enabled
        if ($mandatory_seen_enabled) {
            $data['message_effect_id'] = '👀';
        }
        $result = bot('sendVideo', $data);
    } elseif ($ad['media_type'] === 'document' && $ad['media_file_id']) {
        $data = [
            'chat_id' => $user_id,
            'document' => $ad['media_file_id']
        ];
        if (!empty($caption)) {
            $data['caption'] = $caption;
        }
        if ($reply_markup !== null) {
            $data['reply_markup'] = $reply_markup;
        }
        // Add mandatory seen effect if enabled
        if ($mandatory_seen_enabled) {
            $data['message_effect_id'] = '👀';
        }
        $result = bot('sendDocument', $data);
    } elseif ($ad['media_type'] === 'animation' && $ad['media_file_id']) {
        $data = [
            'chat_id' => $user_id,
            'animation' => $ad['media_file_id']
        ];
        if (!empty($caption)) {
            $data['caption'] = $caption;
        }
        if ($reply_markup !== null) {
            $data['reply_markup'] = $reply_markup;
        }
        // Add mandatory seen effect if enabled
        if ($mandatory_seen_enabled) {
            $data['message_effect_id'] = '👀';
        }
        $result = bot('sendAnimation', $data);
    } elseif (!empty($ad['content'])) {
        // Text-only advertisement
        $keyboard = null;
        if ($reply_markup !== null) {
            $keyboard = json_decode($reply_markup, true);
        }
        // Use sendMessageWithMandatorySeen if enabled
        if ($mandatory_seen_enabled) {
            $result = sendMessageWithMandatorySeen($user_id, $caption, $keyboard);
        } else {
            $result = sendMessage($user_id, $caption, $keyboard);
        }
    }
    
    if ($result !== false && isset($result['ok']) && $result['ok']) {
        // Update show count
        dbExecute(
            "UPDATE advertisements SET show_count = show_count + 1 WHERE id = ?",
            [$ad['id']]
        );
        
        // Log analytics
        logAnalytics('advertisement_shown', $user_id, null, [
            'ad_id' => $ad['id']
        ]);
        
        return true;
    }
    
    return false;
}

/**
 * List all advertisements
 * @param int $chat_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function listAdvertisements(int $chat_id, ?int $message_id = null): void {
    $ads = dbQuery("SELECT * FROM advertisements ORDER BY created_at DESC");
    
    $keyboard = getInlineKeyboard('advertisement');
    
    if ($ads === false || empty($ads)) {
        $msg = "📋 لیست تبلیغات خالی است.";
        if ($message_id !== null) {
            editMessage($chat_id, $message_id, $msg, $keyboard);
        } else {
            sendMessage($chat_id, $msg, $keyboard);
        }
        return;
    }
    
    $message = "📋 لیست تبلیغات:\n\n";
    
    foreach ($ads as $index => $ad) {
        $status = $ad['is_active'] ? '✅ فعال' : '❌ غیرفعال';
        $message .= ($index + 1) . ". {$ad['title']} ({$status})\n";
        $message .= "   👁 نمایش: {$ad['show_count']}\n";
        $message .= "   🆔 شناسه: {$ad['id']}\n\n";
    }
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Get active advertisement channels
 * @return array|false
 */
function getActiveAdvertisementChannels(): array|false {
    return dbQuery(
        "SELECT * FROM advertisement_channels WHERE is_active = 1 ORDER BY display_order ASC, created_at ASC"
    );
}

/**
 * Get random advertisement channel
 * @return array|false
 */
function getRandomAdvertisementChannel(): array|false {
    $channels = getActiveAdvertisementChannels();
    
    if ($channels === false || empty($channels)) {
        return false;
    }
    
    // Return random channel
    return $channels[array_rand($channels)];
}

/**
 * Add advertisement channel
 * @param int $channel_id
 * @param string $channel_title
 * @param string|null $channel_username
 * @param string|null $channel_link
 * @param string $button_text
 * @return int|false Channel ID
 */
function addAdvertisementChannel(int $channel_id, string $channel_title, ?string $channel_username = null, ?string $channel_link = null, string $button_text = '📢 کانال ما'): int|false {
    // Check if channel already exists
    $existing = dbQueryOne(
        "SELECT id FROM advertisement_channels WHERE channel_id = ?",
        [$channel_id]
    );
    
    if ($existing !== false) {
        return false; // Channel already exists
    }
    
    // Get max display_order
    $max_order = dbQueryOne("SELECT MAX(display_order) as max_order FROM advertisement_channels");
    $display_order = ($max_order !== false && isset($max_order['max_order'])) ? (int)$max_order['max_order'] + 1 : 0;
    
    $result = dbExecute(
        "INSERT INTO advertisement_channels (channel_id, channel_username, channel_title, channel_link, button_text, display_order) 
         VALUES (?, ?, ?, ?, ?, ?)",
        [$channel_id, $channel_username, $channel_title, $channel_link, $button_text, $display_order]
    );
    
    if ($result === false) {
        return false;
    }
    
    $channel_ad_id = dbLastInsertId();
    return $channel_ad_id ? (int)$channel_ad_id : false;
}

/**
 * Remove advertisement channel
 * @param int $channel_ad_id Channel advertisement ID
 * @return bool
 */
function removeAdvertisementChannel(int $channel_ad_id): bool {
    $result = dbExecute(
        "DELETE FROM advertisement_channels WHERE id = ?",
        [$channel_ad_id]
    );
    
    return $result !== false;
}

/**
 * Toggle advertisement channel active status
 * @param int $channel_ad_id
 * @return bool
 */
function toggleAdvertisementChannel(int $channel_ad_id): bool {
    $channel = dbQueryOne(
        "SELECT is_active FROM advertisement_channels WHERE id = ?",
        [$channel_ad_id]
    );
    
    if ($channel === false) {
        return false;
    }
    
    $new_status = $channel['is_active'] ? 0 : 1;
    
    $result = dbExecute(
        "UPDATE advertisement_channels SET is_active = ? WHERE id = ?",
        [$new_status, $channel_ad_id]
    );
    
    return $result !== false;
}

/**
 * Get advertisement button (glass style)
 * @param array|null $channel Channel data (optional, if null gets random)
 * @return array|null Inline keyboard with advertisement button
 */
function getAdvertisementButton(?array $channel = null): ?array {
    // Check if advertisements are enabled
    require_once __DIR__ . '/settings-handler.php';
    $enabled = (bool)getSettingValue('advertisement_enabled', true);
    
    if (!$enabled) {
        return null;
    }
    
    // Get channel if not provided
    if ($channel === null) {
        $channel = getRandomAdvertisementChannel();
    }
    
    if ($channel === false) {
        return null; // No active channels
    }
    
    // Build channel URL
    $channel_url = null;
    if (!empty($channel['channel_link'])) {
        $channel_url = $channel['channel_link'];
    } elseif (!empty($channel['channel_username'])) {
        $username = ltrim($channel['channel_username'], '@');
        $channel_url = "https://t.me/{$username}";
    } elseif ($channel['channel_id']) {
        // For private channels, use channel_id format
        $channel_url = "https://t.me/c/" . str_replace('-100', '', (string)$channel['channel_id']);
    }
    
    if ($channel_url === null) {
        return null;
    }
    
    $button_text = $channel['button_text'] ?? '📢 کانال ما';
    
    // Create glass style button (inline keyboard with URL)
    return [
        'inline_keyboard' => [
            [
                [
                    'text' => $button_text,
                    'url' => $channel_url
                ]
            ]
        ]
    ];
}

/**
 * List advertisement channels
 * @param int $chat_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function listAdvertisementChannels(int $chat_id, ?int $message_id = null): void {
    $channels = dbQuery("SELECT * FROM advertisement_channels ORDER BY display_order ASC, created_at DESC");
    
    $keyboard = getInlineKeyboard('advertisement_channels');
    
    if ($channels === false || empty($channels)) {
        $msg = "📋 لیست کانال‌های تبلیغاتی خالی است.";
        if ($message_id !== null) {
            editMessage($chat_id, $message_id, $msg, $keyboard);
        } else {
            sendMessage($chat_id, $msg, $keyboard);
        }
        return;
    }
    
    $message = "📋 لیست کانال‌های تبلیغاتی:\n\n";
    
    foreach ($channels as $index => $channel) {
        $status = $channel['is_active'] ? '✅ فعال' : '❌ غیرفعال';
        $username = $channel['channel_username'] ? '@' . $channel['channel_username'] : 'بدون یوزرنیم';
        $message .= ($index + 1) . ". {$channel['channel_title']} ({$status})\n";
        $message .= "   📱 {$username}\n";
        $message .= "   🔘 متن دکمه: {$channel['button_text']}\n";
        $message .= "   👆 کلیک: {$channel['click_count']}\n";
        $message .= "   🆔 شناسه: {$channel['id']}\n\n";
    }
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

