<?php
declare(strict_types=1);

/**
 * Auto-Delete System Handler
 * Handles automatic message deletion using Telegram deleteMessage API
 * 
 * IMPORTANT: This only deletes messages from user's chat, NOT from database!
 * Files remain in database and can be downloaded again by other users.
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/upload-handler.php';

/**
 * Set auto-delete timer for file (in seconds)
 * @param int $file_id
 * @param int $seconds Seconds until deletion
 * @return bool
 */
function setFileAutoDelete(int $file_id, int $seconds): bool {
    if ($seconds <= 0) {
        // If seconds is 0, disable auto-delete
        $result = dbExecute(
            "UPDATE files SET auto_delete_at = NULL WHERE id = ?",
            [$file_id]
        );
        return $result !== false;
    }
    
    // Store seconds as integer
    $result = dbExecute(
        "UPDATE files SET auto_delete_at = ? WHERE id = ?",
        [$seconds, $file_id]
    );
    
    return $result !== false;
}

/**
 * Set auto-delete timer for file (in hours)
 * @param int $file_id
 * @param int $hours Hours until deletion
 * @return bool
 */
function setFileAutoDeleteHours(int $file_id, int $hours): bool {
    return setFileAutoDelete($file_id, $hours * 3600);
}

/**
 * Remove auto-delete timer for file
 * @param int $file_id
 * @return bool
 */
function removeFileAutoDelete(int $file_id): bool {
    $result = dbExecute(
        "UPDATE files SET auto_delete_at = NULL WHERE id = ?",
        [$file_id]
    );
    
    return $result !== false;
}

/**
 * Get auto-delete timer for file (in seconds)
 * @param int $file_id
 * @return int|false Seconds until deletion, or false if not set
 */
function getFileAutoDeleteTimer(int $file_id): int|false {
    $file = getFileById($file_id);
    
    if ($file === false || empty($file['auto_delete_at'])) {
        return false;
    }
    
    // auto_delete_at stores seconds as integer
    $seconds = (int)$file['auto_delete_at'];
    
    if ($seconds <= 0) {
        return false;
    }
    
    return $seconds;
}

/**
 * Handle set auto-delete command
 * @param array $message Telegram message object
 * @param int $seconds Seconds until deletion
 * @return void
 */
function handleSetAutoDelete(array $message, int $seconds): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند timer حذف خودکار تنظیم کنند.");
        return;
    }
    
    // Check if message is reply to a file
    if (!isset($message['reply_to_message'])) {
        sendMessage($chat_id, "❌ لطفاً به پیام فایلی که می‌خواهید timer برای آن تنظیم کنید reply کنید.");
        return;
    }
    
    $replied_message = $message['reply_to_message'];
    $file_info = extractFileInfo($replied_message);
    
    if ($file_info === false) {
        sendMessage($chat_id, "❌ خطا در دریافت اطلاعات فایل.");
        return;
    }
    
    // Find file in database
    $file = dbQueryOne(
        "SELECT id FROM files WHERE file_id = ? AND is_active = 1 ORDER BY created_at DESC LIMIT 1",
        [$file_info['file_id']]
    );
    
    if ($file === false) {
        sendMessage($chat_id, "❌ فایل در دیتابیس یافت نشد.");
        return;
    }
    
    $file_id = (int)$file['id'];
    
    // Set auto-delete
    if (setFileAutoDelete($file_id, $seconds)) {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;
        
        $time_str = '';
        if ($hours > 0) {
            $time_str .= "{$hours} ساعت ";
        }
        if ($minutes > 0) {
            $time_str .= "{$minutes} دقیقه ";
        }
        if ($secs > 0 || $time_str === '') {
            $time_str .= "{$secs} ثانیه";
        }
        
        sendMessage($chat_id, "✅ Timer حذف خودکار تنظیم شد!\n⏰ پیام‌های ارسال شده به کاربران با این تنظیمات بعد از {$time_str} از چت کاربران حذف خواهند شد.\n📝 توجه: فایل در دیتابیس باقی می‌ماند و قابل دانلود مجدد است.");
    } else {
        sendMessage($chat_id, "❌ خطا در تنظیم timer حذف خودکار.");
    }
}

/**
 * Get global auto-delete timer (in seconds)
 * @return int|false Seconds until deletion, or false if disabled
 */
function getGlobalAutoDeleteTimer(): int|false {
    require_once __DIR__ . '/settings-handler.php';
    $seconds = getSettingValue('global_auto_delete_seconds', 0);
    
    if ($seconds <= 0) {
        return false;
    }
    
    return (int)$seconds;
}

/**
 * Set global auto-delete timer (in seconds)
 * @param int $seconds Seconds until deletion (0 = disabled)
 * @return bool
 */
function setGlobalAutoDeleteTimer(int $seconds): bool {
    require_once __DIR__ . '/settings-handler.php';
    
    if ($seconds < 0) {
        return false;
    }
    
    return setSettingValue('global_auto_delete_seconds', $seconds);
}

/**
 * Handle set global auto-delete command
 * @param int $seconds Seconds until deletion
 * @return string Success or error message
 */
function handleSetGlobalAutoDelete(int $seconds): string {
    if (setGlobalAutoDeleteTimer($seconds)) {
        if ($seconds > 0) {
            $hours = floor($seconds / 3600);
            $minutes = floor(($seconds % 3600) / 60);
            $secs = $seconds % 60;
            
            $time_str = '';
            if ($hours > 0) {
                $time_str .= "{$hours} ساعت ";
            }
            if ($minutes > 0) {
                $time_str .= "{$minutes} دقیقه ";
            }
            if ($secs > 0 || $time_str === '') {
                $time_str .= "{$secs} ثانیه";
            }
            
            return "✅ Timer حذف خودکار سراسری تنظیم شد!\n⏰ تمام پیام‌های ارسال شده به کاربران بعد از {$time_str} از چت کاربران حذف خواهند شد.\n📝 توجه: فایل‌ها در دیتابیس باقی می‌مانند و قابل دانلود مجدد هستند.";
        } else {
            return "✅ Timer حذف خودکار سراسری غیرفعال شد!";
        }
    } else {
        return "❌ خطا در تنظیم timer حذف خودکار سراسری.";
    }
}
