<?php
declare(strict_types=1);

/**
 * Backup Handler
 * Handles backup creation (bot link + text) - uses files table
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/upload-handler.php';

/**
 * Create a backup entry (saves to files table)
 * @param string $bot_link Link to another bot
 * @param string $text Text content
 * @param int $user_id User ID who created the backup
 * @return int|false File ID on success, false on failure
 */
function createBackup(string $bot_link, string $text, int $user_id): int|false {
    // Validate bot link
    if (empty($bot_link) || !filter_var($bot_link, FILTER_VALIDATE_URL)) {
        return false;
    }
    
    // Create backup data as JSON in caption
    $backup_data = [
        'bot_link' => $bot_link,
        'text' => $text,
        'type' => 'backup'
    ];
    $caption = json_encode($backup_data, JSON_UNESCAPED_UNICODE);
    
    // Generate unique file code
    $file_code = generateFileCode();
    
    // Save to files table
    // For backup: file_id = message_id (we'll use a placeholder, actual message_id will be set when user sends it)
    // file_type = 'document' (or we can check caption for backup type)
    $result = dbExecute(
        "INSERT INTO files (file_id, file_code, file_type, file_size, caption, uploaded_by) 
         VALUES (?, ?, 'document', 0, ?, ?)",
        ['backup_' . time(), $file_code, $caption, $user_id]
    );
    
    if ($result === false) {
        logError("Failed to create backup for user {$user_id}");
        return false;
    }
    
    $file_id = (int)dbLastInsertId();
    logInfo("Backup created: {$file_id} (code: {$file_code}) by user {$user_id}");
    
    return $file_id;
}

/**
 * Update backup with message_id
 * @param int $file_id File ID (backup ID)
 * @param int $message_id Telegram message ID
 * @return bool
 */
function updateBackupMessageId(int $file_id, int $message_id): bool {
    $result = dbExecute(
        "UPDATE files SET file_id = ? WHERE id = ?",
        ['backup_msg_' . $message_id, $file_id]
    );
    
    return $result !== false;
}

/**
 * Get backup data from file
 * @param array $file File data from database
 * @return array|false Backup data (bot_link, text) or false
 */
function getBackupData(array $file): array|false {
    if (empty($file['caption'])) {
        return false;
    }
    
    $backup_data = json_decode($file['caption'], true);
    
    if (!is_array($backup_data) || !isset($backup_data['type']) || $backup_data['type'] !== 'backup') {
        return false;
    }
    
    return [
        'bot_link' => $backup_data['bot_link'] ?? '',
        'text' => $backup_data['text'] ?? ''
    ];
}

/**
 * Check if file is a backup
 * @param array $file File data from database
 * @return bool
 */
function isBackup(array $file): bool {
    if (empty($file['caption'])) {
        return false;
    }
    
    $backup_data = json_decode($file['caption'], true);
    
    return is_array($backup_data) && isset($backup_data['type']) && $backup_data['type'] === 'backup';
}

/**
 * Send backup to user (bot link + text)
 * @param int $user_id User ID
 * @param int $chat_id Chat ID
 * @param array $backup_data Backup data (bot_link, text)
 * @param int $file_id File ID (for auto-delete)
 * @return array|false Returns array with 'success' => bool, 'message_id' => int|null, or false on error
 */
function sendBackupToUser(int $user_id, int $chat_id, array $backup_data, int $file_id = 0): array|false {
    $bot_link = $backup_data['bot_link'] ?? '';
    $text = $backup_data['text'] ?? '';
    
    if (empty($bot_link)) {
        return false;
    }
    
    // Build message text
    $message_text = '';
    if (!empty($text)) {
        $message_text = $text . "\n\n";
    }
    $message_text .= "🔗 لینک ربات:";
    
    // Create inline keyboard with bot link
    $keyboard = [
        'inline_keyboard' => [
            [
                [
                    'text' => '🤖 باز کردن ربات',
                    'url' => $bot_link
                ]
            ]
        ]
    ];
    
    // Send single message with keyboard
    $result = sendMessage($chat_id, $message_text, $keyboard);
    
    if ($result === false || !isset($result['ok']) || !$result['ok']) {
        return false;
    }
    
    $sent_message_id = (int)$result['result']['message_id'];
    
    return [
        'success' => true,
        'message_id' => $sent_message_id
    ];
}

