<?php
declare(strict_types=1);

/**
 * Channel Management Handler
 * Handles forced join channel management
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';

/**
 * Handle add channel command
 * @param array $message Telegram message object
 * @return void
 */
function handleAddChannel(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کانال اضافه کنند.");
        return;
    }
    
    // Extract channel info from message
    $channel_info = extractChannelInfo($message);
    
    if ($channel_info === false) {
        sendMessage($chat_id, "❌ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.");
        return;
    }
    
    // Check if channel already exists (by channel_id or username if channel_id is 0)
    if ($channel_info['channel_id'] != 0) {
        $existing = dbQueryOne(
            "SELECT id FROM channels WHERE channel_id = ?",
            [$channel_info['channel_id']]
        );
    } else {
        // If channel_id is 0, check by username
        $existing = dbQueryOne(
            "SELECT id FROM channels WHERE channel_username = ?",
            [$channel_info['channel_username']]
        );
    }
    
    if ($existing !== false) {
        sendMessage($chat_id, "❌ این کانال قبلاً اضافه شده است.");
        return;
    }
    
    // Add channel
    $result = dbExecute(
        "INSERT INTO channels (channel_id, channel_username, channel_title, channel_link) 
         VALUES (?, ?, ?, ?)",
        [
            $channel_info['channel_id'],
            $channel_info['channel_username'],
            $channel_info['channel_title'],
            $channel_info['channel_link']
        ]
    );
    
    if ($result !== false) {
        sendMessage($chat_id, "✅ کانال با موفقیت اضافه شد!");
        logInfo("Channel added: {$channel_info['channel_id']} ({$channel_info['channel_title']}) by {$user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در افزودن کانال.");
    }
}

/**
 * Handle remove channel command
 * @param array $message Telegram message object
 * @return void
 */
function handleRemoveChannel(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کانال حذف کنند.");
        return;
    }
    
    // Extract channel info
    $channel_info = extractChannelInfo($message);
    
    if ($channel_info === false) {
        sendMessage($chat_id, "❌ لطفاً یک پیام از کانال را forward کنید یا شناسه/یوزرنیم کانال را ارسال کنید.");
        return;
    }
    
    // Remove channel (by channel_id or username)
    if ($channel_info['channel_id'] != 0) {
        $result = dbExecute(
            "DELETE FROM channels WHERE channel_id = ?",
            [$channel_info['channel_id']]
        );
    } else {
        $result = dbExecute(
            "DELETE FROM channels WHERE channel_username = ?",
            [$channel_info['channel_username']]
        );
    }
    
    if ($result !== false) {
        sendMessage($chat_id, "✅ کانال با موفقیت حذف شد!");
        logInfo("Channel removed: {$channel_info['channel_id']} ({$channel_info['channel_title']}) by {$user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در حذف کانال.");
    }
}

/**
 * Handle list channels command
 * @param int $chat_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function handleListChannels(int $chat_id, ?int $message_id = null): void {
    $channels = dbQuery("SELECT * FROM channels WHERE is_active = 1 ORDER BY created_at DESC");
    
    $keyboard = getInlineKeyboard('channel_settings');
    
    if ($channels === false || empty($channels)) {
        $msg = "📋 لیست کانال‌های forced join خالی است.";
        if ($message_id !== null) {
            editMessage($chat_id, $message_id, $msg, $keyboard);
        } else {
            sendMessage($chat_id, $msg, $keyboard);
        }
        return;
    }
    
    $message = "📋 لیست کانال‌های Forced Join:\n\n";
    
    foreach ($channels as $index => $channel) {
        $title = $channel['channel_title'];
        $username = $channel['channel_username'] ? '@' . $channel['channel_username'] : 'بدون یوزرنیم';
        $link = $channel['channel_link'] ?: 'بدون لینک';
        
        $message .= ($index + 1) . ". {$title}\n";
        $message .= "   📱 یوزرنیم: {$username}\n";
        $message .= "   🔗 لینک: {$link}\n";
        $message .= "   🆔 شناسه: {$channel['channel_id']}\n\n";
    }
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Extract channel info from message
 * @param array $message
 * @return array|false
 */
function extractChannelInfo(array $message): array|false {
    // Check if message is forwarded from channel
    if (isset($message['forward_from_chat'])) {
        $chat = $message['forward_from_chat'];
        
        if ($chat['type'] === 'channel') {
            $channel_id = (int)$chat['id'];
            // For private channels, channel_id is negative (e.g., -1001234567890)
            // We need to store it as is, but for public channels we can use username
            $username = $chat['username'] ?? null;
            $title = $chat['title'] ?? 'Unknown';
            
            return [
                'channel_id' => $channel_id,
                'channel_username' => $username,
                'channel_title' => $title,
                'channel_link' => $username ? 'https://t.me/' . $username : null
            ];
        }
    }
    
    // Check if message text contains channel info
    if (isset($message['text'])) {
        $text = trim($message['text']);
        // Remove command part if exists
        $text = preg_replace('/^\/\w+\s*/', '', $text);
        $text = trim($text);
        
        if (empty($text)) {
            return false;
        }
        
        // Check if it's a username (with or without @)
        if (preg_match('/^@?([a-zA-Z0-9_]{5,32})$/', $text, $matches)) {
            $username = $matches[1];
            // Try to get channel info from Telegram
            $chat_info = bot('getChat', ['chat_id' => '@' . $username]);
            
            if ($chat_info !== false && isset($chat_info['ok']) && $chat_info['ok']) {
                $chat = $chat_info['result'];
                return [
                    'channel_id' => (int)$chat['id'],
                    'channel_username' => $username,
                    'channel_title' => $chat['title'] ?? $username,
                    'channel_link' => 'https://t.me/' . $username
                ];
            } else {
                // If we can't get info, still return with username
                return [
                    'channel_id' => 0, // Will need to be resolved later
                    'channel_username' => $username,
                    'channel_title' => $username,
                    'channel_link' => 'https://t.me/' . $username
                ];
            }
        }
        
        // Check if it's a numeric ID (can be negative for private channels)
        if (preg_match('/^-?\d+$/', $text)) {
            $channel_id = (int)$text;
            // Try to get channel info from Telegram
            $chat_info = bot('getChat', ['chat_id' => $channel_id]);
            
            if ($chat_info !== false && isset($chat_info['ok']) && $chat_info['ok']) {
                $chat = $chat_info['result'];
                return [
                    'channel_id' => $channel_id,
                    'channel_username' => $chat['username'] ?? null,
                    'channel_title' => $chat['title'] ?? 'Channel ' . $channel_id,
                    'channel_link' => isset($chat['username']) ? 'https://t.me/' . $chat['username'] : null
                ];
            } else {
                // If we can't get info, still return with ID
                return [
                    'channel_id' => $channel_id,
                    'channel_username' => null,
                    'channel_title' => 'Channel ' . $channel_id,
                    'channel_link' => null
                ];
            }
        }
    }
    
    return false;
}

/**
 * Get all active channels
 * @return array|false
 */
function getAllActiveChannels(): array|false {
    return dbQuery("SELECT * FROM channels WHERE is_active = 1");
}

/**
 * Check if user is member of required channels
 * @param int $user_id
 * @param array|null $required_channels Array of channel IDs (if null, check all active channels)
 * @return array ['is_member' => bool, 'missing_channels' => array]
 */
function checkChannelMembership(int $user_id, ?array $required_channels = null): array {
    // Main admin is exempt from forced join check
    if (isMainAdmin($user_id)) {
        return ['is_member' => true, 'missing_channels' => []];
    }
    
    if ($required_channels === null) {
        $channels = getAllActiveChannels();
        if ($channels === false) {
            return ['is_member' => true, 'missing_channels' => []];
        }
        $required_channels = array_column($channels, 'channel_id');
    }
    
    // If no channels required, user is considered a member
    if (empty($required_channels)) {
        return ['is_member' => true, 'missing_channels' => []];
    }
    
    $missing_channels = [];
    
    foreach ($required_channels as $channel_id) {
        // Skip if channel_id is 0 or invalid
        if (empty($channel_id) || $channel_id == 0) {
            continue;
        }
        
        $result = bot('getChatMember', [
            'chat_id' => $channel_id,
            'user_id' => $user_id
        ]);
        
        if ($result === false || !isset($result['ok']) || !$result['ok']) {
            $error_description = $result['description'] ?? '';
            
            // Check for specific errors
            if (strpos($error_description, 'member list is inaccessible') !== false) {
                // Bot can't access member list - this usually means bot is not a member of the channel
                // In this case, we assume user needs to join (safer approach)
                // Log warning for admin to add bot to channel
                error_log("Warning: Bot cannot check membership for channel {$channel_id}. Bot may need to be added to the channel. Error: {$error_description}");
                
                // Send notification to main admin
                require_once __DIR__ . '/../config.php';
                if (defined('MAIN_ADMIN_ID')) {
                    $channel_info = dbQueryOne(
                        "SELECT channel_title, channel_username, channel_link FROM channels WHERE channel_id = ?",
                        [$channel_id]
                    );
                    $channel_name = $channel_info['channel_title'] ?? "کانال {$channel_id}";
                    $channel_link = $channel_info['channel_link'] ?? ($channel_info['channel_username'] ? 'https://t.me/' . $channel_info['channel_username'] : '');
                    
                    $admin_message = "⚠️ هشدار: ربات نمی‌تواند عضویت کاربران را در کانال زیر بررسی کند:\n\n";
                    $admin_message .= "📢 {$channel_name}\n";
                    if ($channel_link) {
                        $admin_message .= "🔗 {$channel_link}\n\n";
                    }
                    $admin_message .= "❌ خطا: {$error_description}\n\n";
                    $admin_message .= "💡 راه حل: لطفاً ربات را به عنوان ادمین یا عضو به کانال اضافه کنید.";
                    
                    sendMessage(MAIN_ADMIN_ID, $admin_message);
                }
                
                $missing_channels[] = $channel_id;
                continue;
            } elseif (strpos($error_description, 'user not found') !== false) {
                // User not found, assume not a member
                $missing_channels[] = $channel_id;
                continue;
            } elseif (strpos($error_description, 'chat not found') !== false) {
                // Chat not found, skip this channel (don't add to missing)
                continue;
            } else {
                // Other errors, assume user is not member (safer approach)
                // Log error for debugging
                error_log("Error checking membership for channel {$channel_id} and user {$user_id}: {$error_description}");
                $missing_channels[] = $channel_id;
                continue;
            }
        }
        
        $status = $result['result']['status'] ?? 'left';
        
        // User is not a member if status is 'left' or 'kicked'
        if (in_array($status, ['left', 'kicked'], true)) {
            $missing_channels[] = $channel_id;
        }
    }
    
    return [
        'is_member' => empty($missing_channels),
        'missing_channels' => $missing_channels
    ];
}

/**
 * Check forced join and send message if user is not member
 * This is a central function to check forced join in all user interactions
 * @param int $user_id
 * @param int $chat_id
 * @param array|null $required_channels Optional: specific channels to check (if null, uses global channels)
 * @return bool True if user is member (or no channels required), false if not member (message sent)
 */
function checkForcedJoinAndNotify(int $user_id, int $chat_id, ?array $required_channels = null): bool {
    // Check if forced join is enabled
    require_once __DIR__ . '/settings-handler.php';
    $forced_join_enabled = (bool)getSettingValue('forced_join_enabled', true);
    
    if (!$forced_join_enabled) {
        return true; // Forced join is disabled
    }
    
    // All admins are exempt from forced join check
    if (isAdmin($user_id)) {
        return true;
    }
    
    // Get channels to check
    if ($required_channels === null) {
        // Get global forced join channels
        $required_channels = getFileForcedJoinChannels(0);
    }
    
    if (empty($required_channels)) {
        return true; // No channels required
    }
    
    $membership = checkChannelMembership($user_id, $required_channels);
    
    if (!$membership['is_member']) {
        $forced_text = getForcedJoinText();
        $message = "{$forced_text}\n\n";
        
        // Get channel info
        $placeholders = implode(',', array_fill(0, count($membership['missing_channels']), '?'));
        $channels = dbQuery(
            "SELECT * FROM channels WHERE channel_id IN ({$placeholders})",
            $membership['missing_channels']
        );
        
        // List of emojis for channels
        $channel_emojis = ['📢', '🔔', '📣', '📡', '📻', '📺', '📰', '📬', '📭', '📮', '💬', '🗣️', '👥', '👤', '🌐', '🔗', '⭐', '✨', '🎯', '🎪', '🎨', '🎭', '🎬', '🎤', '🎧', '🎵', '🎶', '🎸', '🎹', '🎺', '🎻', '🥁', '🎷', '🎯', '🎲', '🎳', '🎰', '🎮', '🕹️', '🎞️', '📽️', '🎬', '🎥', '📹', '📷', '📸', '📱', '💻', '🖥️', '⌨️', '🖱️', '🖨️', '📠', '📞', '☎️', '📟', '📺', '📻', '🔊', '🔉', '🔈', '🔇', '📢', '📣', '📯', '🔔', '🔕', '📳', '📴', '📶', '📡', '📧', '📨', '📩', '📤', '📥', '📦', '📫', '📪', '📬', '📭', '📮', '📯', '📰', '📓', '📔', '📒', '📕', '📗', '📘', '📙', '📚', '📖', '📝', '📄', '📃', '📑', '📊', '📈', '📉', '📜', '📋', '📌', '📍', '📎', '🖇️', '📏', '📐', '✂️', '🗑️', '🔒', '🔓', '🔐', '🔑', '🗝️', '🔨', '⚒️', '🛠️', '⛏️', '🔩', '⚙️', '🗜️', '⚡', '🔥', '💧', '🌊', '☀️', '🌙', '⭐', '🌟', '💫', '✨', '⚡', '☄️', '💥', '💢', '💦', '💨'];
        
        // Build inline keyboard with buttons for each channel
        $keyboard_buttons = [];
        
        if ($channels !== false && !empty($channels)) {
            foreach ($channels as $channel) {
                $link = $channel['channel_link'] ?: ($channel['channel_username'] ? 'https://t.me/' . $channel['channel_username'] : '');
                
                if ($link) {
                    // Get random emoji for this channel
                    $random_emoji = $channel_emojis[array_rand($channel_emojis)];
                    $channel_title = $channel['channel_title'] ?? 'کانال';
                    
                    // Add button for this channel
                    $keyboard_buttons[] = [
                        [
                            'text' => "{$random_emoji} {$channel_title}",
                            'url' => $link
                        ]
                    ];
                }
            }
        }
        
        // Build keyboard
        $keyboard = null;
        if (!empty($keyboard_buttons)) {
            // Add "بررسی عضویت" button
            $keyboard_buttons[] = [
                [
                    'text' => '✅ بررسی عضویت',
                    'callback_data' => 'verify_join'
                ]
            ];
            
            $keyboard = [
                'inline_keyboard' => $keyboard_buttons
            ];
        }
        
        // Send message with inline keyboard and disable web page preview
        sendMessage($chat_id, $message, $keyboard, null, true);
        return false; // User is not member
    }
    
    return true; // User is member
}

/**
 * Get all active reaction channels
 * @return array|false
 */
function getActiveReactionChannels(): array|false {
    // Check if reaction channels are enabled
    require_once __DIR__ . '/settings-handler.php';
    $reaction_channels_enabled = (bool)getSettingValue('reaction_channels_enabled', true);
    
    if (!$reaction_channels_enabled) {
        return false; // Reaction channels are disabled
    }
    
    return dbQuery("SELECT * FROM reaction_channels WHERE is_active = 1 ORDER BY id ASC");
}

/**
 * Add reaction channel
 * @param int $channel_id
 * @param string $channel_title
 * @param string|null $channel_username
 * @param string|null $channel_link
 * @return bool
 */
function addReactionChannel(int $channel_id, string $channel_title, ?string $channel_username = null, ?string $channel_link = null): bool {
    // Check if channel already exists
    $existing = dbQueryOne(
        "SELECT id FROM reaction_channels WHERE channel_id = ?",
        [$channel_id]
    );
    
    if ($existing !== false) {
        // Update existing
        return dbExecute(
            "UPDATE reaction_channels SET channel_title = ?, channel_username = ?, channel_link = ?, is_active = 1 WHERE channel_id = ?",
            [$channel_title, $channel_username, $channel_link, $channel_id]
        ) !== false;
    }
    
    // Insert new
    return dbExecute(
        "INSERT INTO reaction_channels (channel_id, channel_title, channel_username, channel_link) VALUES (?, ?, ?, ?)",
        [$channel_id, $channel_title, $channel_username, $channel_link]
    ) !== false;
}

/**
 * Remove reaction channel
 * @param int $channel_id
 * @return bool
 */
function removeReactionChannel(int $channel_id): bool {
    return dbExecute(
        "DELETE FROM reaction_channels WHERE channel_id = ?",
        [$channel_id]
    ) !== false;
}

/**
 * Toggle reaction channel status
 * @param int $channel_id
 * @return bool
 */
function toggleReactionChannel(int $channel_id): bool {
    $current = dbQueryOne(
        "SELECT is_active FROM reaction_channels WHERE channel_id = ?",
        [$channel_id]
    );
    
    if ($current === false) {
        return false;
    }
    
    $new_status = $current['is_active'] ? 0 : 1;
    return dbExecute(
        "UPDATE reaction_channels SET is_active = ? WHERE channel_id = ?",
        [$new_status, $channel_id]
    ) !== false;
}

/**
 * Show reaction message to user after successful join verification
 * @param int $user_id
 * @param int $chat_id
 * @param int|null $pending_file_id Optional file_id to download after reaction verification
 * @return bool
 */
function showReactionMessage(int $user_id, int $chat_id, ?int $pending_file_id = null): bool {
    $reaction_channels = getActiveReactionChannels();
    
    if ($reaction_channels === false || empty($reaction_channels)) {
        // No reaction channels set, allow user to continue
        return true;
    }
    
    $message = "✅ شما با موفقیت در تمام کانال‌های اجباری عضو شدید!\n\n";
    $message .= "📌 برای ادامه فرایند، لطفاً وارد کانال زیر شوید و 2 پست آخر را ری‌اکشن بزنید:\n\n";
    
    // List of emojis for channels
    $channel_emojis = ['📢', '🔔', '📣', '📡', '📻', '📺', '📰', '📬', '📭', '📮', '💬', '🗣️', '👥', '👤', '🌐', '🔗', '⭐', '✨', '🎯', '🎪', '🎨', '🎭', '🎬', '🎤', '🎧', '🎵', '🎶', '🎸', '🎹', '🎺', '🎻', '🥁', '🎷'];
    
    // Build inline keyboard with buttons for each reaction channel
    $keyboard_buttons = [];
    
    foreach ($reaction_channels as $channel) {
        $link = $channel['channel_link'] ?: ($channel['channel_username'] ? 'https://t.me/' . $channel['channel_username'] : '');
        
        if ($link) {
            // Get random emoji for this channel
            $random_emoji = $channel_emojis[array_rand($channel_emojis)];
            $channel_title = $channel['channel_title'] ?? 'کانال';
            
            $message .= "{$random_emoji} {$channel_title}\n";
            
            // Add button for this channel
            $keyboard_buttons[] = [
                [
                    'text' => "{$random_emoji} {$channel_title}",
                    'url' => $link
                ]
            ];
        }
    }
    
    $message .= "\n⏱️ بعد از 10 ثانیه می‌توانید ادامه دهید.";
    
    // Build keyboard
    $keyboard = null;
    if (!empty($keyboard_buttons)) {
        // Add "بررسی ری‌اکشن" button
        $keyboard_buttons[] = [
            [
                'text' => '✅ بررسی ری‌اکشن',
                'callback_data' => 'verify_reaction'
            ]
        ];
        
        $keyboard = [
            'inline_keyboard' => $keyboard_buttons
        ];
    }
    
    // Save reaction check time to user state
    $reaction_check_time = time() + 10; // 10 seconds from now
    $state_data = [
        'reaction_check_time' => $reaction_check_time,
        'reaction_channels' => array_column($reaction_channels, 'channel_id'),
        'chat_id' => $chat_id // Store chat_id for auto-processing
    ];
    
    // If there's a pending file_id, save it too
    if ($pending_file_id !== null) {
        $state_data['pending_file_id'] = $pending_file_id;
    }
    
    dbExecute(
        "INSERT INTO user_states (user_id, state, state_data) VALUES (?, 'reaction_check', ?) 
         ON DUPLICATE KEY UPDATE state = 'reaction_check', state_data = ?",
        [$user_id, json_encode($state_data), json_encode($state_data)]
    );
    
    // Send message with inline keyboard and disable web page preview
    sendMessage($chat_id, $message, $keyboard, null, true);
    
    return true;
}

/**
 * Set forced join channels for a file
 * @param int $file_id
 * @param array $channel_ids
 * @return bool
 */
function setFileForcedJoinChannels(int $file_id, array $channel_ids): bool {
    $channels_json = json_encode($channel_ids, JSON_UNESCAPED_UNICODE);
    
    $result = dbExecute(
        "UPDATE files SET forced_join_channels = ? WHERE id = ?",
        [$channels_json, $file_id]
    );
    
    return $result !== false;
}

/**
 * Get forced join text from settings
 * @return string
 */
function getForcedJoinText(): string {
    $setting = dbQueryOne(
        "SELECT setting_value FROM settings WHERE setting_key = 'forced_join_text'"
    );
    
    if ($setting === false) {
        return 'لطفاً ابتدا در کانال‌های زیر عضو شوید:';
    }
    
    return $setting['setting_value'] ?: 'لطفاً ابتدا در کانال‌های زیر عضو شوید:';
}

/**
 * Update forced join text
 * @param string $text
 * @return bool
 */
function updateForcedJoinText(string $text): bool {
    $result = dbExecute(
        "UPDATE settings SET setting_value = ? WHERE setting_key = 'forced_join_text'",
        [$text]
    );
    
    if ($result === false) {
        // Insert if doesn't exist
        return dbExecute(
            "INSERT INTO settings (setting_key, setting_value) VALUES ('forced_join_text', ?)",
            [$text]
        ) !== false;
    }
    
    return true;
}

/**
 * Get global forced join channels
 * @return array Array of channel IDs
 */
function getGlobalForcedJoinChannels(): array {
    require_once __DIR__ . '/settings-handler.php';
    $channels_json = getSettingValue('global_forced_join_channels', '[]');
    
    $channels = json_decode($channels_json, true);
    
    if (!is_array($channels)) {
        return [];
    }
    
    return $channels;
}

/**
 * Set global forced join channels
 * @param array $channel_ids Array of channel IDs
 * @return bool
 */
function setGlobalForcedJoinChannels(array $channel_ids): bool {
    require_once __DIR__ . '/settings-handler.php';
    $channels_json = json_encode($channel_ids, JSON_UNESCAPED_UNICODE);
    
    return setSettingValue('global_forced_join_channels', $channels_json);
}

/**
 * Get forced join channels for file (file-specific or global)
 * @param int $file_id
 * @return array Array of channel IDs
 */
function getFileForcedJoinChannels(int $file_id): array {
    // If file_id is 0, return global channels (for /start check)
    if ($file_id == 0) {
        $channels = getAllActiveChannels();
        if ($channels === false || empty($channels)) {
            return [];
        }
        return array_column($channels, 'channel_id');
    }
    
    $file = getFileById($file_id);
    
    // Check if file has specific forced join channels
    if ($file !== false && !empty($file['forced_join_channels'])) {
        $channels = json_decode($file['forced_join_channels'], true);
        if (is_array($channels) && !empty($channels)) {
            return $channels;
        }
    }
    
    // If file doesn't have specific channels, return all active channels from database
    $channels = getAllActiveChannels();
    if ($channels === false || empty($channels)) {
        return [];
    }
    
    // Return channel IDs from active channels
    return array_column($channels, 'channel_id');
}

/**
 * Handle message reaction update from Telegram
 * @param array $message_reaction Telegram message_reaction update
 * @return void
 */
function handleMessageReaction(array $message_reaction): void {
    // Extract reaction data
    $user = $message_reaction['user'] ?? null;
    $chat = $message_reaction['chat'] ?? null;
    $message_id = $message_reaction['message_id'] ?? null;
    $old_reaction = $message_reaction['old_reaction'] ?? null;
    $new_reaction = $message_reaction['new_reaction'] ?? null;
    
    if (!$user || !$chat || !$message_id) {
        return;
    }
    
    $user_id = (int)$user['id'];
    $channel_id = (int)$chat['id'];
    
    // Check if this is a reaction channel
    $reaction_channels = getActiveReactionChannels();
    if ($reaction_channels === false || empty($reaction_channels)) {
        return; // No reaction channels set
    }
    
    $reaction_channel_ids = array_column($reaction_channels, 'channel_id');
    if (!in_array($channel_id, $reaction_channel_ids)) {
        return; // Not a reaction channel
    }
    
    // If new reaction exists, save it
    if ($new_reaction && !empty($new_reaction)) {
        $reactions = is_array($new_reaction) ? $new_reaction : [$new_reaction];
        
        foreach ($reactions as $reaction) {
            $reaction_type = $reaction['type'] ?? 'unknown';
            $reaction_emoji = $reaction['emoji'] ?? null;
            
            // Save reaction to database
            dbExecute(
                "INSERT INTO user_reactions (user_id, channel_id, message_id, reaction_type) 
                 VALUES (?, ?, ?, ?) 
                 ON DUPLICATE KEY UPDATE reaction_type = ?, created_at = CURRENT_TIMESTAMP",
                [$user_id, $channel_id, $message_id, $reaction_type, $reaction_type]
            );
        }
        
        // After saving reaction, check if user has completed all required reactions
        // and automatically proceed if they have a pending file download
        checkAndProcessPendingFileAfterReaction($user_id);
    }
    
    // If old reaction exists and new reaction is empty, remove it
    if ($old_reaction && empty($new_reaction)) {
        $old_reactions = is_array($old_reaction) ? $old_reaction : [$old_reaction];
        
        foreach ($old_reactions as $reaction) {
            dbExecute(
                "DELETE FROM user_reactions WHERE user_id = ? AND channel_id = ? AND message_id = ?",
                [$user_id, $channel_id, $message_id]
            );
        }
    }
}

/**
 * Check if user has completed all required reactions and automatically process pending file download
 * @param int $user_id User ID
 * @return void
 */
function checkAndProcessPendingFileAfterReaction(int $user_id): void {
    require_once __DIR__ . '/../database.php';
    require_once __DIR__ . '/../functions.php';
    require_once __DIR__ . '/point-handler.php';
    require_once __DIR__ . '/user-handler.php';
    
    // Check if user has pending file download state
    $user_state = dbQueryOne(
        "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'reaction_check'",
        [$user_id]
    );
    
    if ($user_state === false || empty($user_state['state_data'])) {
        return; // No pending file download
    }
    
    $state_data = json_decode($user_state['state_data'], true);
    $reaction_check_time = $state_data['reaction_check_time'] ?? 0;
    $pending_file_id = $state_data['pending_file_id'] ?? null;
    
    // Check if 10 seconds have passed (cooldown)
    if (time() < $reaction_check_time) {
        return; // Still in cooldown
    }
    
    // Check if user is member of reaction channels
    $reaction_channels = getActiveReactionChannels();
    
    if ($reaction_channels === false || empty($reaction_channels)) {
        return; // No reaction channels set
    }
    
    $reaction_channel_ids = array_column($reaction_channels, 'channel_id');
    $membership = checkChannelMembership($user_id, $reaction_channel_ids);
    
    if (!$membership['is_member']) {
        return; // User is not a member of reaction channels
    }
    
    // Check if user has reacted to last 2 posts in each reaction channel
    $has_reactions = checkUserReactions($user_id, $reaction_channels, 2);
    
    if (!$has_reactions) {
        return; // User hasn't completed all reactions yet
    }
    
    // User has completed all reactions! Process the file download
    // Get chat_id from user state or try to get it from user
    // We need chat_id to send messages, but we don't have it in message_reaction context
    // So we'll need to get it from somewhere else or store it in state
    
    // For now, we'll just clear the state and let the user click the button
    // Or we can try to get chat_id from the last message sent to user
    
    // Actually, we can't send message here because we don't have chat_id
    // So we'll just mark that reactions are complete and user can click button
    // Or we can store chat_id in state_data when reaction_check state is created
    
    // Let's check if chat_id is stored in state_data
    $chat_id = $state_data['chat_id'] ?? null;
    
    if ($chat_id === null) {
        // Can't send message without chat_id, just return
        // User will need to click the button manually
        return;
    }
    
    // Clear states
    dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'reaction_check'", [$user_id]);
    dbExecute("DELETE FROM user_states WHERE user_id = ? AND state = 'pending_file_download'", [$user_id]);
    
    if ($pending_file_id !== null && $pending_file_id > 0) {
        // Check points before downloading
        $is_admin = isAdmin($user_id);
        $is_vip = isUserVIP($user_id);
        
        // Check if user has enough points (only for non-admin, non-VIP users)
        if (!$is_admin && !$is_vip) {
            if (!hasEnoughPoints($user_id, 1)) {
                sendMessage($chat_id, "❌ امتیاز شما کافی نیست.\n💡 برای کسب امتیاز از سیستم معرف استفاده کنید یا با Telegram Stars خریداری کنید.");
                return;
            }
            
            // Deduct 1 point
            if (!deductUserPoints($user_id, 1)) {
                sendMessage($chat_id, "❌ خطا در کسر امتیاز.");
                return;
            }
            
            // Show deduction message
            $remaining_points = getUserPoints($user_id);
            sendMessage($chat_id, "💎 1 امتیاز کسر شد!\n\n💎 امتیاز باقی‌مانده: {$remaining_points}");
        }
        
        // Download the file (skip forced join check)
        $result = handleFileDownload($user_id, $pending_file_id, $chat_id, true);
        
        if ($result['success']) {
            // File already sent, show success message
            sendMessage($chat_id, $result['message']);
        } else {
            sendMessage($chat_id, $result['message']);
        }
    } else {
        sendMessage($chat_id, "✅ شما می‌توانید ادامه دهید!");
    }
}

/**
 * Check if user has reacted to posts in reaction channels
 * @param int $user_id
 * @param array $reaction_channels Array of reaction channel data
 * @param int $required_posts Number of posts user must react to (default: 2)
 * @return bool
 */
function checkUserReactions(int $user_id, array $reaction_channels, int $required_posts = 2): bool {
    if (empty($reaction_channels)) {
        return true; // No reaction channels, allow
    }
    
    // For each reaction channel, check if user has reacted to at least required_posts
    foreach ($reaction_channels as $channel) {
        $channel_id = (int)$channel['channel_id'];
        
        // Check if user has reacted to at least required_posts different messages in this channel
        // We check distinct message_ids to ensure user reacted to different posts
        $reactions = dbQueryOne(
            "SELECT COUNT(DISTINCT message_id) as reaction_count 
             FROM user_reactions 
             WHERE user_id = ? AND channel_id = ?",
            [$user_id, $channel_id]
        );
        
        if ($reactions === false) {
            return false; // Error querying
        }
        
        $reaction_count = (int)($reactions['reaction_count'] ?? 0);
        if ($reaction_count < $required_posts) {
            return false; // User hasn't reacted to enough posts
        }
    }
    
    return true; // User has reacted to required posts in all channels
}

