<?php
declare(strict_types=1);

/**
 * Daily Spin Handler
 * Handles daily spin wheel for users to earn random points (0-8)
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/point-handler.php';
require_once __DIR__ . '/upload-handler.php'; // For logAnalytics function

/**
 * Check if user can spin today
 * @param int $user_id
 * @return array ['can_spin' => bool, 'last_spin' => string|null, 'message' => string]
 */
function canUserSpinToday(int $user_id): array {
    $last_spin = dbQueryOne(
        "SELECT last_spin_date FROM daily_spins WHERE user_id = ?",
        [$user_id]
    );
    
    if ($last_spin === false) {
        // User never spun before
        return [
            'can_spin' => true,
            'last_spin' => null,
            'message' => ''
        ];
    }
    
    $last_spin_date = $last_spin['last_spin_date'];
    $today = date('Y-m-d');
    
    if ($last_spin_date === $today) {
        // Already spun today
        $next_spin = date('Y-m-d', strtotime('+1 day'));
        return [
            'can_spin' => false,
            'last_spin' => $last_spin_date,
            'message' => "⏰ شما امروز اسپین کرده‌اید!\n\n🔄 اسپین بعدی: {$next_spin}"
        ];
    }
    
    return [
        'can_spin' => true,
        'last_spin' => $last_spin_date,
        'message' => ''
    ];
}

/**
 * Perform daily spin
 * @param int $user_id
 * @param int $chat_id
 * @return array ['success' => bool, 'points' => int, 'message' => string]
 */
function performDailySpin(int $user_id, int $chat_id): array {
    // Check if daily spin is enabled
    require_once __DIR__ . '/settings-handler.php';
    $daily_spin_enabled = (bool)getSettingValue('daily_spin_enabled', true);
    
    if (!$daily_spin_enabled) {
        return [
            'success' => false,
            'points' => 0,
            'message' => '❌ اسپین روزانه غیرفعال است.'
        ];
    }
    
    // Check forced join first
    require_once __DIR__ . '/channel-handler.php';
    if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
        return [
            'success' => false,
            'points' => 0,
            'message' => '' // Message already sent by checkForcedJoinAndNotify
        ];
    }
    
    // Check if user can spin
    $can_spin = canUserSpinToday($user_id);
    
    if (!$can_spin['can_spin']) {
        return [
            'success' => false,
            'points' => 0,
            'message' => $can_spin['message']
        ];
    }
    
    // Generate random points (0-8)
    $points_earned = rand(0, 8);
    
    // Send spinner sticker or animation
    $spinner_sticker_id = defined('SPINNER_STICKER_ID') ? SPINNER_STICKER_ID : '';
    
    // If no sticker configured, use a simple animation or text
    if (empty($spinner_sticker_id)) {
        // Send a message with spinning emoji instead
        $spin_message = sendMessage($chat_id, "🎰 در حال چرخش...");
        
        if ($spin_message !== false && isset($spin_message['ok']) && $spin_message['ok']) {
            $spin_message_id = (int)$spin_message['result']['message_id'];
            
            // Wait a bit for "spinning" effect (2 seconds)
            sleep(2);
            
            // Delete the spinning message
            bot('deleteMessage', [
                'chat_id' => $chat_id,
                'message_id' => $spin_message_id
            ]);
        }
    } else {
        // Send spinner sticker
        $sticker_result = bot('sendSticker', [
            'chat_id' => $chat_id,
            'sticker' => $spinner_sticker_id
        ]);
        
        if ($sticker_result !== false && isset($sticker_result['ok']) && $sticker_result['ok']) {
            $sticker_message_id = (int)$sticker_result['result']['message_id'];
            
            // Wait a bit for "spinning" effect (3 seconds)
            sleep(3);
            
            // Delete the spinner sticker
            bot('deleteMessage', [
                'chat_id' => $chat_id,
                'message_id' => $sticker_message_id
            ]);
        }
    }
    
    // Add points to user
    if ($points_earned > 0) {
        addUserPoints($user_id, $points_earned);
    }
    
    // Update last spin date
    $today = date('Y-m-d');
    dbExecute(
        "INSERT INTO daily_spins (user_id, last_spin_date, points_earned) 
         VALUES (?, ?, ?) 
         ON DUPLICATE KEY UPDATE last_spin_date = ?, points_earned = ?",
        [$user_id, $today, $points_earned, $today, $points_earned]
    );
    
    // Log analytics
    logAnalytics('daily_spin', $user_id, null, [
        'points_earned' => $points_earned
    ]);
    
    $message = "🎉 اسپین شما انجام شد!\n\n";
    if ($points_earned > 0) {
        $message .= "💰 {$points_earned} سکه برنده شدید!\n\n";
    } else {
        $message .= "😔 این بار برنده نشدید!\n\n";
    }
    $message .= "💎 امتیاز کل شما: " . getUserPoints($user_id) . "\n\n";
    $message .= "🔄 اسپین بعدی: فردا";
    
    return [
        'success' => true,
        'points' => $points_earned,
        'message' => $message
    ];
}

/**
 * Get user spin statistics
 * @param int $user_id
 * @return array|false
 */
function getUserSpinStats(int $user_id): array|false {
    $stats = dbQueryOne(
        "SELECT 
            COUNT(*) as total_spins,
            SUM(points_earned) as total_points_earned,
            AVG(points_earned) as avg_points,
            MAX(points_earned) as max_points,
            last_spin_date
         FROM daily_spins 
         WHERE user_id = ?",
        [$user_id]
    );
    
    return $stats;
}

