<?php
declare(strict_types=1);

/**
 * Gambling System Handler
 * Handles gambling games (number guess, coin flip, dice roll)
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/point-handler.php';
require_once __DIR__ . '/settings-handler.php';

/**
 * Check if gambling is enabled
 * @return bool
 */
function checkGamblingEnabled(): bool {
    return (bool)getSettingValue('gambling_enabled', true);
}

/**
 * Check bet limits
 * @param int $bet_amount
 * @return array ['valid' => bool, 'message' => string]
 */
function checkBetLimits(int $bet_amount): array {
    $min_bet = (int)getSettingValue('gambling_min_bet', 1);
    $max_bet = (int)getSettingValue('gambling_max_bet', 100);
    
    if ($bet_amount < $min_bet) {
        return [
            'valid' => false,
            'message' => "❌ حداقل شرط {$min_bet} امتیاز است."
        ];
    }
    
    if ($bet_amount > $max_bet) {
        return [
            'valid' => false,
            'message' => "❌ حداکثر شرط {$max_bet} امتیاز است."
        ];
    }
    
    return ['valid' => true, 'message' => ''];
}

/**
 * Check daily limit
 * @param int $user_id
 * @return array ['valid' => bool, 'message' => string, 'remaining' => int]
 */
function checkDailyLimit(int $user_id): array {
    $daily_limit = (int)getSettingValue('gambling_daily_limit', 0);
    
    // If limit is 0, no limit
    if ($daily_limit === 0) {
        return ['valid' => true, 'message' => '', 'remaining' => -1];
    }
    
    // Get today's games count
    $today_start = date('Y-m-d 00:00:00');
    $games_today = dbQueryOne(
        "SELECT COUNT(*) as count FROM games WHERE user_id = ? AND created_at >= ?",
        [$user_id, $today_start]
    );
    
    $games_count = (int)($games_today['count'] ?? 0);
    $remaining = $daily_limit - $games_count;
    
    if ($games_count >= $daily_limit) {
        return [
            'valid' => false,
            'message' => "❌ شما به حداکثر تعداد بازی روزانه ({$daily_limit}) رسیده‌اید.",
            'remaining' => 0
        ];
    }
    
    return [
        'valid' => true,
        'message' => '',
        'remaining' => $remaining
    ];
}

/**
 * Check cooldown
 * @param int $user_id
 * @return array ['valid' => bool, 'message' => string, 'wait_seconds' => int]
 */
function checkCooldown(int $user_id): array {
    $cooldown_seconds = (int)getSettingValue('gambling_cooldown_seconds', 0);
    
    // If cooldown is 0, no cooldown
    if ($cooldown_seconds === 0) {
        return ['valid' => true, 'message' => '', 'wait_seconds' => 0];
    }
    
    // Get last game time
    $last_game = dbQueryOne(
        "SELECT last_game_at FROM user_gambling_stats WHERE user_id = ?",
        [$user_id]
    );
    
    if ($last_game === false || empty($last_game['last_game_at'])) {
        return ['valid' => true, 'message' => '', 'wait_seconds' => 0];
    }
    
    $last_game_time = strtotime($last_game['last_game_at']);
    $current_time = time();
    $elapsed = $current_time - $last_game_time;
    
    if ($elapsed < $cooldown_seconds) {
        $wait_seconds = $cooldown_seconds - $elapsed;
        $wait_minutes = (int)ceil($wait_seconds / 60);
        
        return [
            'valid' => false,
            'message' => "❌ لطفاً {$wait_minutes} دقیقه صبر کنید تا بتوانید دوباره بازی کنید.",
            'wait_seconds' => $wait_seconds
        ];
    }
    
    return ['valid' => true, 'message' => '', 'wait_seconds' => 0];
}

/**
 * Calculate win amount
 * @param int $bet_amount
 * @return int
 */
function calculateWinAmount(int $bet_amount): int {
    $multiplier = (float)getSettingValue('gambling_win_multiplier', 2.0);
    return (int)round($bet_amount * $multiplier);
}

/**
 * Process game result
 * @param int $user_id
 * @param string $game_type
 * @param int $bet_amount
 * @param bool $won
 * @param int $win_amount
 * @param array|null $game_data Additional game data
 * @return bool
 */
function processGameResult(int $user_id, string $game_type, int $bet_amount, bool $won, int $win_amount, ?array $game_data = null): bool {
    // Record game
    $game_data_json = $game_data ? json_encode($game_data, JSON_UNESCAPED_UNICODE) : null;
    $result = $won ? 'win' : 'lose';
    
    $game_id = dbExecute(
        "INSERT INTO games (user_id, game_type, bet_amount, win_amount, result, game_data) 
         VALUES (?, ?, ?, ?, ?, ?)",
        [$user_id, $game_type, $bet_amount, $won ? $win_amount : 0, $result, $game_data_json]
    );
    
    if ($game_id === false) {
        return false;
    }
    
    // Update user stats
    $stats = dbQueryOne(
        "SELECT * FROM user_gambling_stats WHERE user_id = ?",
        [$user_id]
    );
    
    if ($stats === false) {
        // Create stats record
        dbExecute(
            "INSERT INTO user_gambling_stats (user_id, total_games, total_bet, total_won, total_lost, last_game_at) 
             VALUES (?, 1, ?, ?, ?, NOW())",
            [$user_id, $bet_amount, $won ? $win_amount : 0, $won ? 0 : $bet_amount]
        );
    } else {
        // Update stats
        dbExecute(
            "UPDATE user_gambling_stats 
             SET total_games = total_games + 1,
                 total_bet = total_bet + ?,
                 total_won = total_won + ?,
                 total_lost = total_lost + ?,
                 last_game_at = NOW()
             WHERE user_id = ?",
            [$bet_amount, $won ? $win_amount : 0, $won ? 0 : $bet_amount, $user_id]
        );
    }
    
    // Log analytics
    logAnalytics('gambling_game', $user_id, null, [
        'game_type' => $game_type,
        'bet_amount' => $bet_amount,
        'won' => $won,
        'win_amount' => $won ? $win_amount : 0
    ]);
    
    return true;
}

/**
 * Play number guess game
 * @param int $user_id
 * @param int $bet_amount
 * @param int $guessed_number Number between 1 and 10
 * @return array ['success' => bool, 'message' => string, 'won' => bool, 'win_amount' => int, 'actual_number' => int]
 */
function playNumberGuess(int $user_id, int $bet_amount, int $guessed_number): array {
    // Validate guessed number
    if ($guessed_number < 1 || $guessed_number > 10) {
        return [
            'success' => false,
            'message' => '❌ عدد باید بین 1 تا 10 باشد.',
            'won' => false,
            'win_amount' => 0,
            'actual_number' => 0
        ];
    }
    
    // Check if user has enough points
    if (!hasEnoughPoints($user_id, $bet_amount)) {
        return [
            'success' => false,
            'message' => "❌ امتیاز شما کافی نیست.\n💎 امتیاز شما: " . getUserPoints($user_id),
            'won' => false,
            'win_amount' => 0,
            'actual_number' => 0
        ];
    }
    
    // Deduct bet amount
    if (!deductUserPoints($user_id, $bet_amount)) {
        return [
            'success' => false,
            'message' => '❌ خطا در کسر امتیاز.',
            'won' => false,
            'win_amount' => 0,
            'actual_number' => 0
        ];
    }
    
    // Generate random number (1-10)
    $actual_number = rand(1, 10);
    $won = ($guessed_number === $actual_number);
    $win_amount = $won ? calculateWinAmount($bet_amount) : 0;
    
    // Add win amount if won
    if ($won) {
        addUserPoints($user_id, $win_amount);
    }
    
    // Process result
    processGameResult($user_id, 'number_guess', $bet_amount, $won, $win_amount, [
        'guessed_number' => $guessed_number,
        'actual_number' => $actual_number
    ]);
    
    $message = $won 
        ? "🎉 تبریک! شما برنده شدید!\n\n🎯 عدد شما: {$guessed_number}\n🎲 عدد ربات: {$actual_number}\n💎 امتیاز برنده: {$win_amount}\n\n💎 امتیاز جدید: " . getUserPoints($user_id)
        : "😔 متأسفانه باختید!\n\n🎯 عدد شما: {$guessed_number}\n🎲 عدد ربات: {$actual_number}\n💎 امتیاز کسر شده: {$bet_amount}\n\n💎 امتیاز باقی‌مانده: " . getUserPoints($user_id);
    
    return [
        'success' => true,
        'message' => $message,
        'won' => $won,
        'win_amount' => $win_amount,
        'actual_number' => $actual_number
    ];
}

/**
 * Play coin flip game
 * @param int $user_id
 * @param int $bet_amount
 * @param string $choice 'heads' or 'tails'
 * @return array ['success' => bool, 'message' => string, 'won' => bool, 'win_amount' => int, 'result' => string]
 */
function playCoinFlip(int $user_id, int $bet_amount, string $choice): array {
    // Validate choice
    if ($choice !== 'heads' && $choice !== 'tails') {
        return [
            'success' => false,
            'message' => '❌ انتخاب نامعتبر. لطفاً شیر یا خط انتخاب کنید.',
            'won' => false,
            'win_amount' => 0,
            'result' => ''
        ];
    }
    
    // Check if user has enough points
    if (!hasEnoughPoints($user_id, $bet_amount)) {
        return [
            'success' => false,
            'message' => "❌ امتیاز شما کافی نیست.\n💎 امتیاز شما: " . getUserPoints($user_id),
            'won' => false,
            'win_amount' => 0,
            'result' => ''
        ];
    }
    
    // Deduct bet amount
    if (!deductUserPoints($user_id, $bet_amount)) {
        return [
            'success' => false,
            'message' => '❌ خطا در کسر امتیاز.',
            'won' => false,
            'win_amount' => 0,
            'result' => ''
        ];
    }
    
    // Flip coin (50/50 chance)
    $coin_result = rand(0, 1) === 0 ? 'heads' : 'tails';
    $won = ($choice === $coin_result);
    $win_amount = $won ? calculateWinAmount($bet_amount) : 0;
    
    // Add win amount if won
    if ($won) {
        addUserPoints($user_id, $win_amount);
    }
    
    // Process result
    processGameResult($user_id, 'coin_flip', $bet_amount, $won, $win_amount, [
        'user_choice' => $choice,
        'coin_result' => $coin_result
    ]);
    
    $choice_text = $choice === 'heads' ? 'شیر' : 'خط';
    $result_text = $coin_result === 'heads' ? 'شیر' : 'خط';
    
    $message = $won 
        ? "🎉 تبریک! شما برنده شدید!\n\n🪙 انتخاب شما: {$choice_text}\n🪙 نتیجه: {$result_text}\n💎 امتیاز برنده: {$win_amount}\n\n💎 امتیاز جدید: " . getUserPoints($user_id)
        : "😔 متأسفانه باختید!\n\n🪙 انتخاب شما: {$choice_text}\n🪙 نتیجه: {$result_text}\n💎 امتیاز کسر شده: {$bet_amount}\n\n💎 امتیاز باقی‌مانده: " . getUserPoints($user_id);
    
    return [
        'success' => true,
        'message' => $message,
        'won' => $won,
        'win_amount' => $win_amount,
        'result' => $coin_result
    ];
}

/**
 * Play dice roll game
 * @param int $user_id
 * @param int $bet_amount
 * @param int $guessed_number Number between 1 and 6
 * @return array ['success' => bool, 'message' => string, 'won' => bool, 'win_amount' => int, 'dice_number' => int]
 */
function playDiceRoll(int $user_id, int $bet_amount, int $guessed_number): array {
    // Validate guessed number
    if ($guessed_number < 1 || $guessed_number > 6) {
        return [
            'success' => false,
            'message' => '❌ عدد باید بین 1 تا 6 باشد.',
            'won' => false,
            'win_amount' => 0,
            'dice_number' => 0
        ];
    }
    
    // Check if user has enough points
    if (!hasEnoughPoints($user_id, $bet_amount)) {
        return [
            'success' => false,
            'message' => "❌ امتیاز شما کافی نیست.\n💎 امتیاز شما: " . getUserPoints($user_id),
            'won' => false,
            'win_amount' => 0,
            'dice_number' => 0
        ];
    }
    
    // Deduct bet amount
    if (!deductUserPoints($user_id, $bet_amount)) {
        return [
            'success' => false,
            'message' => '❌ خطا در کسر امتیاز.',
            'won' => false,
            'win_amount' => 0,
            'dice_number' => 0
        ];
    }
    
    // Roll dice (1-6)
    $dice_number = rand(1, 6);
    $won = ($guessed_number === $dice_number);
    $win_amount = $won ? calculateWinAmount($bet_amount) : 0;
    
    // Add win amount if won
    if ($won) {
        addUserPoints($user_id, $win_amount);
    }
    
    // Process result
    processGameResult($user_id, 'dice_roll', $bet_amount, $won, $win_amount, [
        'guessed_number' => $guessed_number,
        'dice_number' => $dice_number
    ]);
    
    $message = $won 
        ? "🎉 تبریک! شما برنده شدید!\n\n🎯 عدد شما: {$guessed_number}\n🎲 تاس: {$dice_number}\n💎 امتیاز برنده: {$win_amount}\n\n💎 امتیاز جدید: " . getUserPoints($user_id)
        : "😔 متأسفانه باختید!\n\n🎯 عدد شما: {$guessed_number}\n🎲 تاس: {$dice_number}\n💎 امتیاز کسر شده: {$bet_amount}\n\n💎 امتیاز باقی‌مانده: " . getUserPoints($user_id);
    
    return [
        'success' => true,
        'message' => $message,
        'won' => $won,
        'win_amount' => $win_amount,
        'dice_number' => $dice_number
    ];
}

/**
 * Get user gambling statistics
 * @param int $user_id
 * @return array
 */
function getUserGamblingStats(int $user_id): array {
    $stats = dbQueryOne(
        "SELECT * FROM user_gambling_stats WHERE user_id = ?",
        [$user_id]
    );
    
    if ($stats === false) {
        return [
            'total_games' => 0,
            'total_bet' => 0,
            'total_won' => 0,
            'total_lost' => 0,
            'win_rate' => 0.0,
            'profit' => 0
        ];
    }
    
    $total_games = (int)$stats['total_games'];
    $win_rate = $total_games > 0 ? ((int)$stats['total_won'] / $total_games) * 100 : 0.0;
    $profit = (int)$stats['total_won'] - (int)$stats['total_bet'];
    
    return [
        'total_games' => $total_games,
        'total_bet' => (int)$stats['total_bet'],
        'total_won' => (int)$stats['total_won'],
        'total_lost' => (int)$stats['total_lost'],
        'win_rate' => round($win_rate, 2),
        'profit' => $profit
    ];
}

/**
 * Display gambling statistics
 * @param int $chat_id
 * @param int $user_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayGamblingStats(int $chat_id, int $user_id, ?int $message_id = null): void {
    require_once __DIR__ . '/../auth.php';
    require_once __DIR__ . '/channel-handler.php';
    
    // Check forced join first
    if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
        return; // Message already sent
    }
    
    $stats = getUserGamblingStats($user_id);
    
    $message = "📊 آمار بازی‌های شما:\n\n";
    $message .= "🎮 تعداد کل بازی‌ها: {$stats['total_games']}\n";
    $message .= "💰 کل شرط‌بندی: {$stats['total_bet']} امتیاز\n";
    $message .= "🎉 کل برد: {$stats['total_won']} امتیاز\n";
    $message .= "😔 کل باخت: {$stats['total_lost']} امتیاز\n";
    $message .= "📈 نرخ برد: {$stats['win_rate']}%\n";
    
    $profit_text = $stats['profit'] >= 0 ? "✅" : "❌";
    $message .= "💎 سود/زیان: {$profit_text} " . abs($stats['profit']) . " امتیاز\n\n";
    
    $message .= "💡 برای بازی بیشتر، از منوی بازی استفاده کنید.";
    
    // Check if admin is in user mode - use appropriate keyboard
    $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
        ? getInlineKeyboard('gambling_stats_menu', 1, ['user_id' => $user_id])
        : getInlineKeyboard('gambling_stats_menu');
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

