<?php
declare(strict_types=1);

/**
 * Media Management Handler
 * Handles media management operations
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/upload-handler.php';

/**
 * Toggle thumbnail for file
 * @param int $file_id
 * @param bool $enabled
 * @return bool
 */
function toggleFileThumbnail(int $file_id, bool $enabled): bool {
    $result = dbExecute(
        "UPDATE files SET has_thumbnail = ? WHERE id = ?",
        [$enabled ? 1 : 0, $file_id]
    );
    
    return $result !== false;
}

/**
 * Toggle link for file
 * @param int $file_id
 * @param bool $enabled
 * @return bool
 */
function toggleFileLink(int $file_id, bool $enabled): bool {
    $result = dbExecute(
        "UPDATE files SET has_link = ? WHERE id = ?",
        [$enabled ? 1 : 0, $file_id]
    );
    
    return $result !== false;
}

/**
 * Get media information
 * @param int $file_id
 * @return array|false
 */
function getMediaInfo(int $file_id): array|false {
    $file = getFileById($file_id);
    
    if ($file === false) {
        return false;
    }
    
    // Get forced join channels
    $forced_join = [];
    if (!empty($file['forced_join_channels'])) {
        $forced_join = json_decode($file['forced_join_channels'], true) ?: [];
    }
    
    // Get channel info
    $channels_info = [];
    if (!empty($forced_join)) {
        $placeholders = implode(',', array_fill(0, count($forced_join), '?'));
        $channels = dbQuery(
            "SELECT * FROM channels WHERE channel_id IN ({$placeholders})",
            $forced_join
        );
        
        if ($channels !== false) {
            $channels_info = $channels;
        }
    }
    
    return [
        'file' => $file,
        'forced_join_channels' => $channels_info,
        'download_count' => (int)$file['download_count']
    ];
}

/**
 * Display media information
 * @param int $chat_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayMediaInfo(int $chat_id, ?int $message_id = null): void {
    // For now, show a message asking for file ID
    // In future, this can be enhanced to show file list
    $message = "📁 اطلاعات مدیا\n\n";
    $message .= "لطفاً شناسه فایل را ارسال کنید یا به پیام فایل reply کنید.";
    
    $keyboard = getInlineKeyboard('back_only');
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Format file size
 * @param int $bytes
 * @return string
 */
function formatFileSize(int $bytes): string {
    if ($bytes == 0) {
        return '0 B';
    }
    
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $unit_index = 0;
    $size = $bytes;
    
    while ($size >= 1024 && $unit_index < count($units) - 1) {
        $size /= 1024;
        $unit_index++;
    }
    
    return round($size, 2) . ' ' . $units[$unit_index];
}

/**
 * List all media files
 * @param int $chat_id
 * @param int $limit
 * @return void
 */
function listMediaFiles(int $chat_id, int $limit = 20): void {
    $files = getAllFiles($limit);
    
    if ($files === false || empty($files)) {
        sendMessage($chat_id, "📋 لیست فایل‌ها خالی است.");
        return;
    }
    
    $message = "📋 لیست فایل‌ها:\n\n";
    
    foreach ($files as $index => $file) {
        $message .= ($index + 1) . ". {$file['file_type']} (ID: {$file['id']})\n";
        $message .= "   📥 دانلودها: {$file['download_count']}\n";
        $message .= "   📅 {$file['created_at']}\n\n";
    }
    
    sendMessage($chat_id, $message);
}

/**
 * Display media list with pagination
 * @param int $chat_id
 * @param int $user_id
 * @param int $page Page number (1-based)
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayMediaList(int $chat_id, int $user_id, int $page = 1, ?int $message_id = null): void {
    require_once __DIR__ . '/upload-handler.php';
    require_once __DIR__ . '/auto-delete-handler.php';
    
    $limit = 10; // Files per page
    $offset = ($page - 1) * $limit;
    
    $files = getAllFiles($limit, $offset, 'date');
    
    if ($files === false || empty($files)) {
        $msg = "📑 هیچ فایلی یافت نشد.";
        $keyboard = getInlineKeyboard('media');
        
        if ($message_id !== null) {
            editMessage($chat_id, $message_id, $msg, $keyboard);
        } else {
            sendMessage($chat_id, $msg, $keyboard);
        }
        return;
    }
    
    $msg = "📑 لیست فایل‌ها:\n\n";
    $msg .= "💡 روی هر فایل کلیک کنید تا اطلاعات کامل و تنظیمات آن را ببینید.\n\n";
    
    $keyboard_buttons = [];
    
    foreach ($files as $index => $file) {
        $file_type_emoji = [
            'photo' => '🖼️',
            'video' => '🎥',
            'document' => '📄',
            'audio' => '🎵',
            'voice' => '🎤',
            'video_note' => '📹',
            'sticker' => '😀',
            'animation' => '🎬'
        ];
        $emoji = $file_type_emoji[$file['file_type']] ?? '📁';
        
        // Check file-specific auto-delete first, then global
        $auto_delete_seconds = null;
        if (!empty($file['auto_delete_at'])) {
            $auto_delete_seconds = (int)$file['auto_delete_at'];
        } else {
            // Check global auto-delete
            $global_seconds = getGlobalAutoDeleteTimer();
            if ($global_seconds !== false) {
                $auto_delete_seconds = $global_seconds;
            }
        }
        
        $auto_delete = $auto_delete_seconds !== null ? "⏱️ {$auto_delete_seconds}s" : "❌";
        
        $file_number = $offset + $index + 1;
        $msg .= "{$file_number}. {$emoji} فایل #{$file['id']}\n";
        $msg .= "   📊 دانلود: {$file['download_count']}\n";
        $msg .= "   {$auto_delete}\n\n";
        
        $keyboard_buttons[] = [
            [
                'text' => "{$emoji} فایل #{$file['id']}",
                'callback_data' => 'media_info_' . $file['id']
            ]
        ];
    }
    
    // Add pagination buttons
    $total_files = dbQueryOne(
        "SELECT COUNT(*) as count FROM files WHERE is_active = 1",
        []
    );
    $total_count = (int)($total_files['count'] ?? 0);
    $total_pages = max(1, (int)ceil($total_count / $limit));
    
    if ($total_pages > 1) {
        $pagination_row = [];
        if ($page > 1) {
            $pagination_row[] = [
                'text' => '◀️ قبلی',
                'callback_data' => 'list_media_page_' . ($page - 1)
            ];
        }
        $pagination_row[] = [
            'text' => "📄 {$page}/{$total_pages}",
            'callback_data' => 'list_media_page_' . $page
        ];
        if ($page < $total_pages) {
            $pagination_row[] = [
                'text' => '▶️ بعدی',
                'callback_data' => 'list_media_page_' . ($page + 1)
            ];
        }
        $keyboard_buttons[] = $pagination_row;
    }
    
    $keyboard_buttons[] = [
        ['text' => '⬅️ بازگشت', 'callback_data' => 'media']
    ];
    
    $keyboard = [
        'inline_keyboard' => $keyboard_buttons
    ];
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $msg, $keyboard);
    } else {
        sendMessage($chat_id, $msg, $keyboard);
    }
}

