<?php
declare(strict_types=1);

/**
 * Payment Handler
 * Handles Telegram Stars payments
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/point-handler.php';
require_once __DIR__ . '/referral-handler.php';

// Use getSetting from referral-handler
if (!function_exists('getSetting')) {
    function getSetting(string $key, mixed $default = null): mixed {
        require_once __DIR__ . '/referral-handler.php';
        return getSetting($key, $default);
    }
}

/**
 * Handle successful payment
 * @param array $pre_checkout_query
 * @return void
 */
function handleSuccessfulPayment(array $pre_checkout_query): void {
    $user_id = $pre_checkout_query['from']['id'];
    $payment_id = $pre_checkout_query['id'];
    $currency = $pre_checkout_query['currency'] ?? 'XTR';
    $total_amount = $pre_checkout_query['total_amount'] ?? 0;
    
    // Verify payment
    $result = bot('answerPreCheckoutQuery', [
        'pre_checkout_query_id' => $payment_id,
        'ok' => true
    ]);
    
    if ($result === false || !isset($result['ok']) || !$result['ok']) {
        logError("Failed to confirm payment: {$payment_id}");
        return;
    }
    
    // Convert stars to points
    $stars_to_points_rate = getSetting('stars_to_points_rate', 1);
    $points_to_add = (int)($total_amount * $stars_to_points_rate);
    
    // Add points to user
    addUserPoints($user_id, $points_to_add);
    
    // Record payment
    dbExecute(
        "INSERT INTO payments (user_id, stars_amount, points_amount, status, telegram_payment_id) 
         VALUES (?, ?, ?, 'completed', ?)",
        [$user_id, $total_amount, $points_to_add, $payment_id]
    );
    
    // Log analytics
    logAnalytics('payment_completed', $user_id, null, [
        'stars_amount' => $total_amount,
        'points_amount' => $points_to_add
    ]);
    
    // Notify user
    $chat_id = $pre_checkout_query['from']['id'];
    sendMessage($chat_id, "✅ پرداخت با موفقیت انجام شد!\n💎 {$points_to_add} امتیاز به حساب شما اضافه شد.");
}

/**
 * Handle failed payment
 * @param array $pre_checkout_query
 * @return void
 */
function handleFailedPayment(array $pre_checkout_query): void {
    $payment_id = $pre_checkout_query['id'];
    $error_message = $pre_checkout_query['error_message'] ?? 'Unknown error';
    
    // Reject payment
    bot('answerPreCheckoutQuery', [
        'pre_checkout_query_id' => $payment_id,
        'ok' => false,
        'error_message' => 'پرداخت ناموفق بود.'
    ]);
    
    logError("Payment failed: {$payment_id} - {$error_message}");
}

/**
 * Create invoice for stars payment
 * @param int $chat_id
 * @param int $stars_amount
 * @param string $description
 * @return array|false
 */
function createStarsInvoice(int $chat_id, int $stars_amount, string $description = 'خرید امتیاز'): array|false {
    $stars_to_points_rate = getSetting('stars_to_points_rate', 1);
    $points_amount = $stars_amount * $stars_to_points_rate;
    
    $payload = json_encode([
        'type' => 'stars',
        'stars_amount' => $stars_amount,
        'points_amount' => $points_amount
    ], JSON_UNESCAPED_UNICODE);
    
    $result = bot('sendInvoice', [
        'chat_id' => $chat_id,
        'title' => 'خرید امتیاز',
        'description' => "{$description}\n💎 شما {$points_amount} امتیاز دریافت خواهید کرد.",
        'payload' => $payload,
        'provider_token' => '', // Not needed for Telegram Stars
        'currency' => 'XTR', // Telegram Stars currency code
        'prices' => [
            [
                'label' => 'امتیاز',
                'amount' => $stars_amount
            ]
        ]
    ]);
    
    return $result;
}

/**
 * Get payment history for user
 * @param int $user_id
 * @param int $limit
 * @return array|false
 */
function getPaymentHistory(int $user_id, int $limit = 10): array|false {
    return dbQuery(
        "SELECT * FROM payments WHERE user_id = ? ORDER BY created_at DESC LIMIT ?",
        [$user_id, $limit]
    );
}

/**
 * Get all payments (admin only)
 * @param int $limit
 * @return array|false
 */
function getAllPayments(int $limit = 50): array|false {
    return dbQuery(
        "SELECT * FROM payments ORDER BY created_at DESC LIMIT ?",
        [$limit]
    );
}

