<?php
declare(strict_types=1);

/**
 * Point System Handler
 * Handles point management (1 point = 1 download)
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';

/**
 * Get user points
 * @param int $user_id
 * @return int
 */
function getUserPoints(int $user_id): int {
    $result = dbQueryOne(
        "SELECT points FROM points WHERE user_id = ?",
        [$user_id]
    );
    
    if ($result === false) {
        // Initialize points for user
        dbExecute(
            "INSERT INTO points (user_id, points) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id",
            [$user_id]
        );
        return 0;
    }
    
    return (int)$result['points'];
}

/**
 * Add points to user
 * @param int $user_id
 * @param int $points
 * @return bool
 */
function addUserPoints(int $user_id, int $points): bool {
    if ($points <= 0) {
        return false;
    }
    
    // Ensure user has points record
    dbExecute(
        "INSERT INTO points (user_id, points) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id",
        [$user_id]
    );
    
    $result = dbExecute(
        "UPDATE points SET points = points + ? WHERE user_id = ?",
        [$points, $user_id]
    );
    
    return $result !== false;
}

/**
 * Deduct points from user
 * @param int $user_id
 * @param int $points
 * @return bool
 */
function deductUserPoints(int $user_id, int $points): bool {
    if ($points <= 0) {
        return false;
    }
    
    $current_points = getUserPoints($user_id);
    
    if ($current_points < $points) {
        return false; // Not enough points
    }
    
    $result = dbExecute(
        "UPDATE points SET points = points - ? WHERE user_id = ?",
        [$points, $user_id]
    );
    
    return $result !== false;
}

/**
 * Check if user has enough points
 * @param int $user_id
 * @param int $required_points
 * @return bool
 */
function hasEnoughPoints(int $user_id, int $required_points): bool {
    $current_points = getUserPoints($user_id);
    return $current_points >= $required_points;
}

/**
 * Handle download with point deduction
 * @param int $user_id
 * @param int $file_id
 * @param int $chat_id
 * @param bool $skip_forced_join Skip forced join check (already verified)
 * @return array ['success' => bool, 'message' => string, 'sent_message_id' => int|null]
 */
function handleFileDownload(int $user_id, int $file_id, int $chat_id, bool $skip_forced_join = false): array {
    require_once __DIR__ . '/auto-delete-handler.php';
    require_once __DIR__ . '/user-handler.php';
    require_once __DIR__ . '/channel-handler.php';
    
    // Check if user is admin or VIP (needed for backup handling)
    $is_admin = isAdmin($user_id);
    $is_vip = isUserVIP($user_id);
    
    // Check forced join first (file-specific or global) - unless already verified
    if (!$skip_forced_join) {
        $required_channels = getFileForcedJoinChannels($file_id);
        if (!empty($required_channels)) {
            // Use checkForcedJoinAndNotify for consistent glass keyboard
            if (!checkForcedJoinAndNotify($user_id, $chat_id, $required_channels)) {
                return ['success' => false, 'message' => '', 'sent_message_id' => null];
            }
        }
    }
    
    // Check if file exists
    $file = getFileById($file_id);
    
    if ($file === false) {
        return ['success' => false, 'message' => '❌ فایل یافت نشد.', 'sent_message_id' => null];
    }
    
    // Check if this is a backup (bot link + text)
    require_once __DIR__ . '/backup-handler.php';
    if (isBackup($file)) {
        $backup_data = getBackupData($file);
        
        if ($backup_data === false) {
            return ['success' => false, 'message' => '❌ خطا در دریافت اطلاعات بک آپ.', 'sent_message_id' => null];
        }
        
        // Send backup to user
        $backup_result = sendBackupToUser($user_id, $chat_id, $backup_data, $file_id);
        
        if ($backup_result !== false && $backup_result['success']) {
            // Update download count
            dbExecute(
                "UPDATE files SET download_count = download_count + 1 WHERE id = ?",
                [$file_id]
            );
            
            $sent_message_id = $backup_result['message_id'] ?? null;
            
            // Send info message (similar to file download)
            $info_message = "📤 لطفا مدیا را به پیوی دوستان یا سیو مسیج خودتون فوروارد کنید\n\n";
            
            // Check auto-delete timer for info message
            require_once __DIR__ . '/auto-delete-handler.php';
            $auto_delete_timer = getFileAutoDeleteTimer($file_id);
            if ($auto_delete_timer === false || $auto_delete_timer <= 0) {
                $global_timer = getGlobalAutoDeleteTimer();
                if ($global_timer !== false && $global_timer > 0) {
                    $auto_delete_timer = $global_timer;
                }
            }
            
            if ($auto_delete_timer !== false && $auto_delete_timer > 0) {
                $info_message .= "⏱️ پیام بعد از {$auto_delete_timer} ثانیه حذف می‌شود\n\n";
            }
            
            // Add points deduction info if paid mode is enabled and user is not admin/VIP
            if (!$is_admin && !$is_vip) {
                require_once __DIR__ . '/settings-handler.php';
                if (isPaidMode()) {
                    $info_message .= "💎 بابت دریافت این بک آپ از شما 1 سکه/امتیاز کسر شد";
                }
            }
            
            $info_result = sendMessage($chat_id, $info_message);
            $info_message_id = null;
            if ($info_result !== false && isset($info_result['ok']) && $info_result['ok']) {
                $info_message_id = (int)$info_result['result']['message_id'];
            }
            
            // Auto-delete info message if timer is set
            if ($auto_delete_timer !== false && $auto_delete_timer > 0 && $info_message_id !== null) {
                ignore_user_abort(true);
                set_time_limit(0);
                
                sleep($auto_delete_timer);
                
                bot('deleteMessage', [
                    'chat_id' => $chat_id,
                    'message_id' => $info_message_id
                ]);
            }
            
            // Auto-delete backup message if timer is set
            if ($auto_delete_timer !== false && $auto_delete_timer > 0 && $sent_message_id !== null) {
                // Delete backup message after timer expires (in background)
                ignore_user_abort(true);
                set_time_limit(0);
                
                sleep($auto_delete_timer);
                
                bot('deleteMessage', [
                    'chat_id' => $chat_id,
                    'message_id' => $sent_message_id
                ]);
                
                logInfo("Auto-deleted backup message: {$sent_message_id} from chat {$chat_id} after {$auto_delete_timer} seconds");
            }
            
            // Log analytics
            logAnalytics('backup_downloaded', $user_id, $file_id, []);
            
            return [
                'success' => true,
                'message' => '✅ بک آپ با موفقیت ارسال شد!',
                'sent_message_id' => $sent_message_id
            ];
        } else {
            return ['success' => false, 'message' => '❌ خطا در ارسال بک آپ.', 'sent_message_id' => null];
        }
    }
    
    // Check if user has enough points (only for non-admin, non-VIP users)
    // Skip point check if skip_forced_join is true (points already deducted before calling this function)
    if (!$skip_forced_join && !$is_admin && !$is_vip) {
        if (!hasEnoughPoints($user_id, 1)) {
            return [
                'success' => false,
                'message' => "❌ امتیاز شما کافی نیست.\n💡 برای کسب امتیاز از سیستم معرف استفاده کنید یا با Telegram Stars خریداری کنید.",
                'sent_message_id' => null
            ];
        }
        
        // Deduct 1 point (only for non-admin, non-VIP users)
        if (!deductUserPoints($user_id, 1)) {
            return ['success' => false, 'message' => '❌ خطا در کسر امتیاز.', 'sent_message_id' => null];
        }
    }
    
    // Send file to user based on file type
    $sent_message_id = null;
    $result = false;
    
    $file_id_telegram = $file['file_id'];
    $caption = $file['caption'] ?? null;
    
    // Get advertisement button
    require_once __DIR__ . '/advertisement-handler.php';
    $ad_button = getAdvertisementButton();
    
    switch ($file['file_type']) {
        case 'photo':
            $data = [
                'chat_id' => $chat_id,
                'photo' => $file_id_telegram
            ];
            if ($caption !== null) {
                $data['caption'] = $caption;
            }
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendPhoto', $data);
            break;
            
        case 'video':
            $data = [
                'chat_id' => $chat_id,
                'video' => $file_id_telegram
            ];
            if ($caption !== null) {
                $data['caption'] = $caption;
            }
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendVideo', $data);
            break;
            
        case 'document':
            $data = [
                'chat_id' => $chat_id,
                'document' => $file_id_telegram
            ];
            if ($caption !== null) {
                $data['caption'] = $caption;
            }
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendDocument', $data);
            break;
            
        case 'audio':
            $data = [
                'chat_id' => $chat_id,
                'audio' => $file_id_telegram
            ];
            if ($caption !== null) {
                $data['caption'] = $caption;
            }
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendAudio', $data);
            break;
            
        case 'voice':
            $data = [
                'chat_id' => $chat_id,
                'voice' => $file_id_telegram
            ];
            if ($caption !== null) {
                $data['caption'] = $caption;
            }
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendVoice', $data);
            break;
            
        case 'video_note':
            $data = [
                'chat_id' => $chat_id,
                'video_note' => $file_id_telegram
            ];
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendVideoNote', $data);
            break;
            
        case 'sticker':
            $data = [
                'chat_id' => $chat_id,
                'sticker' => $file_id_telegram
            ];
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendSticker', $data);
            break;
            
        case 'animation':
            $data = [
                'chat_id' => $chat_id,
                'animation' => $file_id_telegram
            ];
            if ($caption !== null) {
                $data['caption'] = $caption;
            }
            if ($ad_button !== null) {
                $data['reply_markup'] = json_encode($ad_button, JSON_UNESCAPED_UNICODE);
            }
            $result = bot('sendAnimation', $data);
            break;
    }
    
    if ($result !== false && isset($result['ok']) && $result['ok']) {
        $sent_message_id = (int)$result['result']['message_id'];
        
        // Track advertisement click if button was shown
        if ($ad_button !== null) {
            // Get the channel that was used
            $channel = getRandomAdvertisementChannel();
            if ($channel !== false) {
                // Increment click count (will be tracked when user clicks)
                // We can't track clicks directly, but we can track impressions
                // For actual clicks, we'd need to use a bot button with callback_data
                // But URL buttons don't support callbacks, so we'll just track impressions
            }
        }
        
        // Record download
        dbExecute(
            "INSERT INTO downloads (user_id, file_id) VALUES (?, ?)",
            [$user_id, $file_id]
        );
        
        // Update file download count
        dbExecute(
            "UPDATE files SET download_count = download_count + 1 WHERE id = ?",
            [$file_id]
        );
        
        // Update popularity score
        require_once __DIR__ . '/upload-handler.php';
        updateFilePopularity($file_id);
        
        // Log analytics
        logAnalytics('file_downloaded', $user_id, $file_id, [
            'file_type' => $file['file_type']
        ]);
        
        // Check if auto-delete is set (file-specific or global)
        $auto_delete_timer = getFileAutoDeleteTimer($file_id);
        if ($auto_delete_timer === false || $auto_delete_timer <= 0) {
            // If file doesn't have specific timer, check global
            $global_timer = getGlobalAutoDeleteTimer();
            if ($global_timer !== false && $global_timer > 0) {
                $auto_delete_timer = $global_timer;
            }
        }
        
        // Send info message after file is sent
        $info_message = "📤 لطفا مدیا را به پیوی دوستان یا سیو مسیج خودتون فوروارد کنید\n\n";
        
        // Add auto-delete info if timer is set
        if ($auto_delete_timer !== false && $auto_delete_timer > 0) {
            $info_message .= "⏱️ فایل بعد از {$auto_delete_timer} ثانیه حذف می‌شود\n\n";
        }
        
        // Add points deduction info if paid mode is enabled and user is not admin/VIP
        if (!$is_admin && !$is_vip) {
            require_once __DIR__ . '/settings-handler.php';
            if (isPaidMode()) {
                $info_message .= "💎 بابت دریافت این فایل از شما 1 سکه/امتیاز کسر شد";
            }
        }
        
        $info_result = sendMessage($chat_id, $info_message);
        $info_message_id = null;
        if ($info_result !== false && isset($info_result['ok']) && $info_result['ok']) {
            $info_message_id = (int)$info_result['result']['message_id'];
        }
        
        if ($auto_delete_timer !== false && $auto_delete_timer > 0) {
            // IMPORTANT: Only delete the message from user's chat, NOT from database!
            // The file remains in database and can be downloaded again by other users.
            // Delete message after timer expires (in background)
            ignore_user_abort(true);
            set_time_limit(0);
            
            // Wait for timer
            sleep($auto_delete_timer);
            
            // Delete the file message
            bot('deleteMessage', [
                'chat_id' => $chat_id,
                'message_id' => $sent_message_id
            ]);
            
            // Delete the info message as well
            if ($info_message_id !== null) {
                bot('deleteMessage', [
                    'chat_id' => $chat_id,
                    'message_id' => $info_message_id
                ]);
            }
            
            logInfo("Auto-deleted message: {$sent_message_id} from chat {$chat_id} after {$auto_delete_timer} seconds (file remains in database)");
            
            // Show advertisement after file is deleted
            require_once __DIR__ . '/advertisement-handler.php';
            showAdvertisementToUser($user_id);
        }
        
        return [
            'success' => true,
            'message' => "✅ فایل با موفقیت دانلود شد!\n💎 امتیاز باقی‌مانده: " . getUserPoints($user_id),
            'file' => $file,
            'sent_message_id' => $sent_message_id
        ];
    } else {
        // File sending failed
        $error_message = '❌ خطا در ارسال فایل.';
        if ($result !== false && isset($result['description'])) {
            $error_message .= "\n" . $result['description'];
        }
        
        return [
            'success' => false,
            'message' => $error_message,
            'file' => $file,
            'sent_message_id' => null
        ];
    }
}

/**
 * Display user points
 * @param int $chat_id
 * @param int $user_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayUserPoints(int $chat_id, int $user_id, ?int $message_id = null): void {
    require_once __DIR__ . '/../auth.php';
    require_once __DIR__ . '/channel-handler.php';
    
    // Check forced join first
    if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
        return; // Message already sent
    }
    
    $points = getUserPoints($user_id);
    
    $message = "💎 امتیاز شما: {$points}\n\n";
    $message .= "💡 هر 1 امتیاز = 1 دانلود\n";
    $message .= "📥 برای دانلود فایل، 1 امتیاز از شما کسر می‌شود.";
    
    // Check if admin is in user mode - use appropriate keyboard
    $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
        ? getInlineKeyboard('user_points_menu', 1, ['user_id' => $user_id])
        : getInlineKeyboard('user_points_menu');
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Set user points (admin only)
 * @param int $user_id
 * @param int $points
 * @return bool
 */
function setUserPoints(int $user_id, int $points): bool {
    if ($points < 0) {
        return false;
    }
    
    // Ensure user has points record
    dbExecute(
        "INSERT INTO points (user_id, points) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id",
        [$user_id]
    );
    
    $result = dbExecute(
        "UPDATE points SET points = ? WHERE user_id = ?",
        [$points, $user_id]
    );
    
    return $result !== false;
}

/**
 * Handle add points from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param int $target_user_id
 * @param int $points
 * @return void
 */
function handleAddPointsFromPanel(int $chat_id, int $admin_id, int $target_user_id, int $points): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند امتیاز اضافه کنند.");
        return;
    }
    
    if ($points <= 0) {
        sendMessage($chat_id, "❌ مقدار امتیاز باید بزرگ‌تر از صفر باشد.");
        return;
    }
    
    if (addUserPoints($target_user_id, $points)) {
        $new_points = getUserPoints($target_user_id);
        sendMessage($chat_id, "✅ {$points} امتیاز به کاربر اضافه شد!\n\n💎 امتیاز جدید: {$new_points}");
        logInfo("Admin {$admin_id} added {$points} points to user {$target_user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در افزودن امتیاز.");
    }
}

/**
 * Handle deduct points from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param int $target_user_id
 * @param int $points
 * @return void
 */
function handleDeductPointsFromPanel(int $chat_id, int $admin_id, int $target_user_id, int $points): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند امتیاز کسر کنند.");
        return;
    }
    
    if ($points <= 0) {
        sendMessage($chat_id, "❌ مقدار امتیاز باید بزرگ‌تر از صفر باشد.");
        return;
    }
    
    $current_points = getUserPoints($target_user_id);
    
    if ($current_points < $points) {
        sendMessage($chat_id, "❌ امتیاز کاربر کافی نیست.\n💎 امتیاز فعلی: {$current_points}");
        return;
    }
    
    if (deductUserPoints($target_user_id, $points)) {
        $new_points = getUserPoints($target_user_id);
        sendMessage($chat_id, "✅ {$points} امتیاز از کاربر کسر شد!\n\n💎 امتیاز جدید: {$new_points}");
        logInfo("Admin {$admin_id} deducted {$points} points from user {$target_user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در کسر امتیاز.");
    }
}

/**
 * Handle set points from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param int $target_user_id
 * @param int $points
 * @return void
 */
function handleSetPointsFromPanel(int $chat_id, int $admin_id, int $target_user_id, int $points): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند امتیاز تنظیم کنند.");
        return;
    }
    
    if ($points < 0) {
        sendMessage($chat_id, "❌ مقدار امتیاز نمی‌تواند منفی باشد.");
        return;
    }
    
    if (setUserPoints($target_user_id, $points)) {
        sendMessage($chat_id, "✅ امتیاز کاربر به {$points} تنظیم شد!");
        logInfo("Admin {$admin_id} set points to {$points} for user {$target_user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در تنظیم امتیاز.");
    }
}

