<?php
declare(strict_types=1);

/**
 * Promo Code Handler
 * Handles promo codes (gift codes, discount codes, coin codes)
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/point-handler.php';

/**
 * Create promo code
 * @param string $code Promo code
 * @param string $type 'gift' (points), 'discount' (percentage), 'coin' (stars)
 * @param int $value Value (points, percentage, or stars)
 * @param int $usage_limit Maximum usage limit (0 = unlimited)
 * @param int $expires_at Timestamp when code expires (0 = never)
 * @param int $created_by Admin ID who created the code
 * @return bool
 */
function createPromoCode(string $code, string $type, int $value, int $usage_limit = 0, int $expires_at = 0, int $created_by = 0): bool {
    // Validate code
    if (empty($code) || strlen($code) < 3) {
        return false;
    }
    
    // Validate type
    if (!in_array($type, ['gift', 'discount', 'coin'], true)) {
        return false;
    }
    
    // Check if code already exists
    $existing = dbQueryOne(
        "SELECT id FROM promo_codes WHERE code = ?",
        [$code]
    );
    
    if ($existing !== false) {
        return false; // Code already exists
    }
    
    $expires_at_sql = $expires_at > 0 ? date('Y-m-d H:i:s', $expires_at) : null;
    
    $result = dbExecute(
        "INSERT INTO promo_codes (code, type, value, usage_limit, usage_count, expires_at, created_by, is_active) 
         VALUES (?, ?, ?, ?, 0, ?, ?, 1)",
        [$code, $type, $value, $usage_limit, $expires_at_sql, $created_by]
    );
    
    return $result !== false;
}

/**
 * Use promo code
 * @param int $user_id User ID
 * @param string $code Promo code
 * @return array ['success' => bool, 'message' => string, 'points' => int]
 */
function usePromoCode(int $user_id, string $code): array {
    // Check if promo codes are enabled
    require_once __DIR__ . '/settings-handler.php';
    $promo_codes_enabled = (bool)getSettingValue('promo_codes_enabled', true);
    
    if (!$promo_codes_enabled) {
        return ['success' => false, 'message' => '❌ کدهای جایزه غیرفعال است.', 'points' => 0];
    }
    
    $code = strtoupper(trim($code));
    
    // Get promo code
    $promo = dbQueryOne(
        "SELECT * FROM promo_codes WHERE code = ? AND is_active = 1",
        [$code]
    );
    
    if ($promo === false) {
        return ['success' => false, 'message' => '❌ کد نامعتبر است.', 'points' => 0];
    }
    
    // Check if expired
    if (!empty($promo['expires_at'])) {
        $expires_at = strtotime($promo['expires_at']);
        if (time() > $expires_at) {
            return ['success' => false, 'message' => '❌ کد منقضی شده است.', 'points' => 0];
        }
    }
    
    // Check usage limit
    if ($promo['usage_limit'] > 0 && $promo['usage_count'] >= $promo['usage_limit']) {
        return ['success' => false, 'message' => '❌ کد به پایان رسیده است.', 'points' => 0];
    }
    
    // Check if user already used this code
    $used = dbQueryOne(
        "SELECT id FROM promo_code_usage WHERE user_id = ? AND promo_code_id = ?",
        [$user_id, $promo['id']]
    );
    
    if ($used !== false) {
        return ['success' => false, 'message' => '❌ شما قبلاً از این کد استفاده کرده‌اید.', 'points' => 0];
    }
    
    $points_given = 0;
    $message = '';
    
    // Apply promo code based on type
    switch ($promo['type']) {
        case 'gift':
            // Add points
            $points_given = (int)$promo['value'];
            if (addUserPoints($user_id, $points_given)) {
                $message = "✅ کد جایزه با موفقیت اعمال شد!\n💎 {$points_given} امتیاز به حساب شما اضافه شد.";
            } else {
                return ['success' => false, 'message' => '❌ خطا در اعمال کد.', 'points' => 0];
            }
            break;
            
        case 'discount':
            // Discount is for future use (e.g., discount on purchases)
            $discount = (int)$promo['value'];
            $message = "✅ کد تخفیف با موفقیت اعمال شد!\n💰 {$discount}% تخفیف برای شما ثبت شد.";
            break;
            
        case 'coin':
            // Add stars/coins (points equivalent)
            $points_given = (int)$promo['value'];
            if (addUserPoints($user_id, $points_given)) {
                $message = "✅ کد سکه با موفقیت اعمال شد!\n⭐ {$points_given} سکه به حساب شما اضافه شد.";
            } else {
                return ['success' => false, 'message' => '❌ خطا در اعمال کد.', 'points' => 0];
            }
            break;
    }
    
    // Record usage
    dbExecute(
        "INSERT INTO promo_code_usage (user_id, promo_code_id) VALUES (?, ?)",
        [$user_id, $promo['id']]
    );
    
    // Update usage count
    dbExecute(
        "UPDATE promo_codes SET usage_count = usage_count + 1 WHERE id = ?",
        [$promo['id']]
    );
    
    // Log analytics
    logAnalytics('promo_code_used', $user_id, null, [
        'code' => $code,
        'type' => $promo['type'],
        'value' => $promo['value']
    ]);
    
    return ['success' => true, 'message' => $message, 'points' => $points_given];
}

/**
 * List all promo codes
 * @return array|false
 */
function listPromoCodes(): array|false {
    return dbQuery(
        "SELECT * FROM promo_codes ORDER BY created_at DESC"
    );
}

/**
 * Delete promo code by code string
 * @param string $code Promo code string
 * @return array ['success' => bool, 'message' => string]
 */
function deletePromoCode(string $code): array {
    $code = strtoupper(trim($code));
    
    // Check if code exists
    $existing = dbQueryOne(
        "SELECT id FROM promo_codes WHERE code = ?",
        [$code]
    );
    
    if ($existing === false) {
        return ['success' => false, 'message' => "❌ کدی با این نام یافت نشد."];
    }
    
    $result = dbExecute(
        "DELETE FROM promo_codes WHERE code = ?",
        [$code]
    );
    
    if ($result !== false) {
        logInfo("Promo code deleted: {$code}");
        return ['success' => true, 'message' => "✅ کد {$code} با موفقیت حذف شد."];
    } else {
        logError("Failed to delete promo code: {$code}");
        return ['success' => false, 'message' => "❌ خطا در حذف کد."];
    }
}

/**
 * Get promo code stats
 * @param int $code_id Code ID
 * @return array|false
 */
function getPromoCodeStats(int $code_id): array|false {
    $code = dbQueryOne(
        "SELECT * FROM promo_codes WHERE id = ?",
        [$code_id]
    );
    
    if ($code === false) {
        return false;
    }
    
    $usage_count = dbQueryOne(
        "SELECT COUNT(*) as count FROM promo_code_usage WHERE promo_code_id = ?",
        [$code_id]
    );
    
    return [
        'code' => $code,
        'usage_count' => (int)($usage_count['count'] ?? 0)
    ];
}

