<?php
declare(strict_types=1);

/**
 * Referral System Handler
 * Handles referral system for earning points
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/point-handler.php';

/**
 * Get or create referral code for user
 * @param int $user_id
 * @return string|false
 */
function getUserReferralCode(int $user_id): string|false {
    $user = dbQueryOne(
        "SELECT referral_code FROM users WHERE id = ?",
        [$user_id]
    );
    
    if ($user === false) {
        return false;
    }
    
    // If user already has referral code, return it
    if (!empty($user['referral_code'])) {
        return $user['referral_code'];
    }
    
    // Generate new referral code
    $code = generateReferralCode($user_id);
    
    // Save to database
    $result = dbExecute(
        "UPDATE users SET referral_code = ? WHERE id = ?",
        [$code, $user_id]
    );
    
    if ($result === false) {
        return false;
    }
    
    return $code;
}

/**
 * Handle referral link usage (multi-level support)
 * @param int $user_id New user ID
 * @param string $referral_code Referral code
 * @return array ['success' => bool, 'referrer_id' => int|null, 'points_given' => array]
 */
function handleReferralUsage(int $user_id, string $referral_code): array {
    // Check if referral system is enabled
    require_once __DIR__ . '/settings-handler.php';
    $referral_enabled = (bool)getSettingValue('referral_enabled', true);
    
    if (!$referral_enabled) {
        return ['success' => false, 'referrer_id' => null, 'points_given' => []];
    }
    
    // Find referrer
    $referrer = dbQueryOne(
        "SELECT id FROM users WHERE referral_code = ?",
        [$referral_code]
    );
    
    if ($referrer === false) {
        return ['success' => false, 'referrer_id' => null, 'points_given' => []];
    }
    
    $referrer_id = (int)$referrer['id'];
    
    // Check if user was already referred
    $existing = dbQueryOne(
        "SELECT id FROM referrals WHERE referred_id = ?",
        [$user_id]
    );
    
    if ($existing !== false) {
        return ['success' => false, 'referrer_id' => $referrer_id, 'points_given' => []];
    }
    
    // Use multi-level referral handler
    return handleMultiLevelReferral($user_id, $referral_code);
}

/**
 * Handle multi-level referral (3 levels)
 * @param int $user_id New user ID
 * @param string $referral_code Referral code
 * @return array ['success' => bool, 'referrer_id' => int|null, 'points_given' => array]
 */
function handleMultiLevelReferral(int $user_id, string $referral_code): array {
    // Find direct referrer (level 1)
    $referrer = dbQueryOne(
        "SELECT id FROM users WHERE referral_code = ?",
        [$referral_code]
    );
    
    if ($referrer === false) {
        return ['success' => false, 'referrer_id' => null, 'points_given' => []];
    }
    
    $level1_referrer_id = (int)$referrer['id'];
    
    // Get referral points from settings
    require_once __DIR__ . '/settings-handler.php';
    $points_level1 = (int)getSettingValue('referral_points_level_1', 5);
    $points_level2 = (int)getSettingValue('referral_points_level_2', 3);
    $points_level3 = (int)getSettingValue('referral_points_level_3', 1);
    $points_referred = (int)getSettingValue('referral_points_referred', 3);
    
    $total_points_given = [];
    
    // Give points to level 1 referrer
    addUserPoints($level1_referrer_id, $points_level1);
    $total_points_given['level_1'] = $points_level1;
    
    // Record level 1 referral
    dbExecute(
        "INSERT INTO referrals (referrer_id, referred_id, points_given_referrer, points_given_referred, referral_level) 
         VALUES (?, ?, ?, ?, 1)",
        [$level1_referrer_id, $user_id, $points_level1, $points_referred]
    );
    
    // Update user's referred_by
    dbExecute(
        "UPDATE users SET referred_by = ? WHERE id = ?",
        [$level1_referrer_id, $user_id]
    );
    
    // Find level 2 referrer (referrer of level 1)
    $level2_referrer = dbQueryOne(
        "SELECT referred_by FROM users WHERE id = ?",
        [$level1_referrer_id]
    );
    
    if ($level2_referrer !== false && !empty($level2_referrer['referred_by'])) {
        $level2_referrer_id = (int)$level2_referrer['referred_by'];
        
        // Give points to level 2 referrer
        addUserPoints($level2_referrer_id, $points_level2);
        $total_points_given['level_2'] = $points_level2;
        
        // Record level 2 referral (indirect)
        dbExecute(
            "INSERT INTO referrals (referrer_id, referred_id, points_given_referrer, points_given_referred, referral_level) 
             VALUES (?, ?, ?, ?, 2)",
            [$level2_referrer_id, $user_id, $points_level2, 0]
        );
        
        // Find level 3 referrer (referrer of level 2)
        $level3_referrer = dbQueryOne(
            "SELECT referred_by FROM users WHERE id = ?",
            [$level2_referrer_id]
        );
        
        if ($level3_referrer !== false && !empty($level3_referrer['referred_by'])) {
            $level3_referrer_id = (int)$level3_referrer['referred_by'];
            
            // Give points to level 3 referrer
            addUserPoints($level3_referrer_id, $points_level3);
            $total_points_given['level_3'] = $points_level3;
            
            // Record level 3 referral (indirect)
            dbExecute(
                "INSERT INTO referrals (referrer_id, referred_id, points_given_referrer, points_given_referred, referral_level) 
                 VALUES (?, ?, ?, ?, 3)",
                [$level3_referrer_id, $user_id, $points_level3, 0]
            );
        }
    }
    
    // Give points to referred user
    addUserPoints($user_id, $points_referred);
    $total_points_given['referred'] = $points_referred;
    
    // Log analytics
    logAnalytics('referral_used', $user_id, null, [
        'referrer_id' => $level1_referrer_id,
        'points_given' => $total_points_given,
        'multi_level' => true
    ]);
    
    return [
        'success' => true,
        'referrer_id' => $level1_referrer_id,
        'points_given' => $total_points_given
    ];
}

/**
 * Calculate referral points for a specific level
 * @param int $level Referral level (1, 2, or 3)
 * @return int Points for this level
 */
function calculateReferralPoints(int $level): int {
    require_once __DIR__ . '/settings-handler.php';
    
    if ($level === 1) {
        return (int)getSettingValue('referral_points_level_1', 5);
    } elseif ($level === 2) {
        return (int)getSettingValue('referral_points_level_2', 3);
    } elseif ($level === 3) {
        return (int)getSettingValue('referral_points_level_3', 1);
    }
    
    return 0;
}

/**
 * Get referral statistics for user
 * @param int $user_id
 * @return array
 */
function getReferralStats(int $user_id): array {
    $referrals = dbQuery(
        "SELECT COUNT(*) as total_referrals, 
                SUM(points_given_referrer) as total_points_earned 
         FROM referrals WHERE referrer_id = ?",
        [$user_id]
    );
    
    $stats = [
        'total_referrals' => 0,
        'total_points_earned' => 0
    ];
    
    if ($referrals !== false && !empty($referrals)) {
        $stats['total_referrals'] = (int)$referrals[0]['total_referrals'];
        $stats['total_points_earned'] = (int)($referrals[0]['total_points_earned'] ?? 0);
    }
    
    return $stats;
}

/**
 * Get advanced referral statistics with level breakdown
 * @param int $user_id
 * @return array
 */
function getReferralStatsAdvanced(int $user_id): array {
    // Level 1 stats (direct referrals)
    $level1_stats = dbQueryOne(
        "SELECT COUNT(*) as count, SUM(points_given_referrer) as points 
         FROM referrals WHERE referrer_id = ? AND referral_level = 1",
        [$user_id]
    );
    
    // Level 2 stats (indirect referrals)
    $level2_stats = dbQueryOne(
        "SELECT COUNT(*) as count, SUM(points_given_referrer) as points 
         FROM referrals WHERE referrer_id = ? AND referral_level = 2",
        [$user_id]
    );
    
    // Level 3 stats (indirect referrals)
    $level3_stats = dbQueryOne(
        "SELECT COUNT(*) as count, SUM(points_given_referrer) as points 
         FROM referrals WHERE referrer_id = ? AND referral_level = 3",
        [$user_id]
    );
    
    return [
        'level_1' => [
            'count' => (int)($level1_stats['count'] ?? 0),
            'points' => (int)($level1_stats['points'] ?? 0)
        ],
        'level_2' => [
            'count' => (int)($level2_stats['count'] ?? 0),
            'points' => (int)($level2_stats['points'] ?? 0)
        ],
        'level_3' => [
            'count' => (int)($level3_stats['count'] ?? 0),
            'points' => (int)($level3_stats['points'] ?? 0)
        ],
        'total_referrals' => (int)($level1_stats['count'] ?? 0) + (int)($level2_stats['count'] ?? 0) + (int)($level3_stats['count'] ?? 0),
        'total_points_earned' => (int)($level1_stats['points'] ?? 0) + (int)($level2_stats['points'] ?? 0) + (int)($level3_stats['points'] ?? 0)
    ];
}

/**
 * Get referral tree for user
 * @param int $user_id
 * @param int $max_depth Maximum depth (default: 3)
 * @return array Tree structure with referrals
 */
function getReferralTree(int $user_id, int $max_depth = 3): array {
    $tree = [
        'user_id' => $user_id,
        'level' => 0,
        'children' => []
    ];
    
    // Get direct referrals (level 1)
    $direct_referrals = dbQuery(
        "SELECT referred_id FROM referrals WHERE referrer_id = ? AND referral_level = 1",
        [$user_id]
    );
    
    if ($direct_referrals === false || empty($direct_referrals)) {
        return $tree;
    }
    
    foreach ($direct_referrals as $referral) {
        $referred_id = (int)$referral['referred_id'];
        
        $child = [
            'user_id' => $referred_id,
            'level' => 1,
            'children' => []
        ];
        
        // Recursively get children if depth allows
        if ($max_depth > 1) {
            $child['children'] = getReferralTreeRecursive($referred_id, 2, $max_depth);
        }
        
        $tree['children'][] = $child;
    }
    
    return $tree;
}

/**
 * Recursive helper for building referral tree
 * @param int $user_id
 * @param int $current_level
 * @param int $max_depth
 * @return array
 */
function getReferralTreeRecursive(int $user_id, int $current_level, int $max_depth): array {
    if ($current_level > $max_depth) {
        return [];
    }
    
    $children = [];
    
    $direct_referrals = dbQuery(
        "SELECT referred_id FROM referrals WHERE referrer_id = ? AND referral_level = 1",
        [$user_id]
    );
    
    if ($direct_referrals === false || empty($direct_referrals)) {
        return $children;
    }
    
    foreach ($direct_referrals as $referral) {
        $referred_id = (int)$referral['referred_id'];
        
        $child = [
            'user_id' => $referred_id,
            'level' => $current_level,
            'children' => []
        ];
        
        // Recursively get children if depth allows
        if ($current_level < $max_depth) {
            $child['children'] = getReferralTreeRecursive($referred_id, $current_level + 1, $max_depth);
        }
        
        $children[] = $child;
    }
    
    return $children;
}

/**
 * Display referral tree
 * @param int $chat_id
 * @param int $user_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayReferralTree(int $chat_id, int $user_id, ?int $message_id = null): void {
    require_once __DIR__ . '/../auth.php';
    require_once __DIR__ . '/channel-handler.php';
    
    // Check forced join first
    if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
        return; // Message already sent
    }
    
    $tree = getReferralTree($user_id, 3);
    $stats = getReferralStatsAdvanced($user_id);
    
    $message = "🌳 درخت معرف شما:\n\n";
    
    // Display stats
    $message .= "📊 آمار کلی:\n";
    $message .= "👥 کل معرف‌ها: {$stats['total_referrals']}\n";
    $message .= "💎 کل امتیاز کسب شده: {$stats['total_points_earned']}\n\n";
    
    $message .= "📈 تفکیک بر اساس سطح:\n";
    $message .= "   🔹 سطح 1 (مستقیم): {$stats['level_1']['count']} نفر - {$stats['level_1']['points']} امتیاز\n";
    $message .= "   🔹 سطح 2: {$stats['level_2']['count']} نفر - {$stats['level_2']['points']} امتیاز\n";
    $message .= "   🔹 سطح 3: {$stats['level_3']['count']} نفر - {$stats['level_3']['points']} امتیاز\n\n";
    
    // Display tree structure
    if (!empty($tree['children'])) {
        $message .= "🌳 ساختار درخت:\n";
        $message .= buildTreeString($tree, 0);
    } else {
        $message .= "🌳 شما هنوز هیچ معرفی ندارید.";
    }
    
    // Check if admin is in user mode - use appropriate keyboard
    $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
        ? getInlineKeyboard('user_referral_menu', 1, ['user_id' => $user_id])
        : getInlineKeyboard('user_referral_menu');
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Build tree string representation
 * @param array $tree Tree structure
 * @param int $indent Current indentation level
 * @return string
 */
function buildTreeString(array $tree, int $indent = 0): string {
    $result = '';
    $prefix = str_repeat('  ', $indent);
    
    if ($indent > 0) {
        $result .= $prefix . "└─ ";
    }
    
    // Get user info
    $user = dbQueryOne(
        "SELECT first_name, username FROM users WHERE id = ?",
        [$tree['user_id']]
    );
    
    if ($user !== false) {
        $name = $user['first_name'] ?? 'Unknown';
        $username = !empty($user['username']) ? '@' . $user['username'] : '';
        $result .= "{$name} {$username} (ID: {$tree['user_id']})";
        if ($tree['level'] > 0) {
            $result .= " [سطح {$tree['level']}]";
        }
        $result .= "\n";
    }
    
    // Add children
    foreach ($tree['children'] as $child) {
        $result .= buildTreeString($child, $indent + 1);
    }
    
    return $result;
}

/**
 * Display referral link and stats
 * @param int $chat_id
 * @param int $user_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayReferralInfo(int $chat_id, int $user_id, ?int $message_id = null): void {
    require_once __DIR__ . '/../auth.php';
    require_once __DIR__ . '/channel-handler.php';
    
    // Check forced join first
    if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
        return; // Message already sent
    }
    
    $code = getUserReferralCode($user_id);
    
    if ($code === false) {
        $msg = "❌ خطا در ایجاد کد معرف.";
        $keyboard = getInlineKeyboard('back_only');
        if ($message_id !== null) {
            editMessage($chat_id, $message_id, $msg, $keyboard);
        } else {
            sendMessage($chat_id, $msg, $keyboard);
        }
        return;
    }
    
    $bot_username = BOT_USERNAME ?? 'your_bot';
    $referral_link = "https://t.me/{$bot_username}?start={$code}";
    
    // Get advanced stats with level breakdown
    $stats = getReferralStatsAdvanced($user_id);
    
    $message = "🔗 لینک معرف شما:\n";
    $message .= "{$referral_link}\n\n";
    $message .= "📊 آمار معرف:\n";
    $message .= "👥 کل معرف‌ها: {$stats['total_referrals']}\n";
    $message .= "💎 کل امتیاز کسب شده: {$stats['total_points_earned']}\n\n";
    
    $message .= "📈 تفکیک بر اساس سطح:\n";
    $message .= "   🔹 سطح 1 (مستقیم): {$stats['level_1']['count']} نفر - {$stats['level_1']['points']} امتیاز\n";
    $message .= "   🔹 سطح 2: {$stats['level_2']['count']} نفر - {$stats['level_2']['points']} امتیاز\n";
    $message .= "   🔹 سطح 3: {$stats['level_3']['count']} نفر - {$stats['level_3']['points']} امتیاز\n\n";
    
    $message .= "💡 با هر معرف، شما و معرف‌شده امتیاز دریافت می‌کنید!\n";
    $message .= "💡 معرف‌های سطح 2 و 3 نیز برای شما امتیاز می‌آورند!";
    
    // Check if admin is in user mode - use appropriate keyboard
    $keyboard = (isRealAdmin($user_id) && isInUserMode($user_id)) 
        ? getInlineKeyboard('user_referral_menu', 1, ['user_id' => $user_id])
        : getInlineKeyboard('user_referral_menu');
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Get setting value
 * @param string $key
 * @param mixed $default
 * @return mixed
 */
function getSetting(string $key, mixed $default = null): mixed {
    $setting = dbQueryOne(
        "SELECT setting_value FROM settings WHERE setting_key = ?",
        [$key]
    );
    
    if ($setting === false) {
        return $default;
    }
    
    $value = $setting['setting_value'];
    
    // Try to convert to int if numeric
    if (is_numeric($value)) {
        return (int)$value;
    }
    
    return $value ?: $default;
}

