<?php
declare(strict_types=1);

/**
 * Settings Handler
 * Handles bot settings and configuration
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/referral-handler.php';

/**
 * Get setting value
 * @param string $key
 * @param mixed $default
 * @return mixed
 */
function getSettingValue(string $key, mixed $default = null): mixed {
    $setting = dbQueryOne(
        "SELECT setting_value FROM settings WHERE setting_key = ?",
        [$key]
    );
    
    if ($setting === false) {
        return $default;
    }
    
    $value = $setting['setting_value'];
    
    // Convert to boolean if needed
    if ($value === '0' || $value === '1') {
        return (bool)(int)$value;
    }
    
    // Convert to int if numeric
    if (is_numeric($value)) {
        return (int)$value;
    }
    
    return $value ?: $default;
}

/**
 * Set setting value
 * @param string $key
 * @param mixed $value
 * @return bool
 */
function setSettingValue(string $key, mixed $value): bool {
    // Convert boolean to string
    if (is_bool($value)) {
        $value = $value ? '1' : '0';
    } else {
        $value = (string)$value;
    }
    
    $result = dbExecute(
        "UPDATE settings SET setting_value = ? WHERE setting_key = ?",
        [$value, $key]
    );
    
    if ($result === false) {
        // Insert if doesn't exist
        return dbExecute(
            "INSERT INTO settings (setting_key, setting_value) VALUES (?, ?)",
            [$key, $value]
        ) !== false;
    }
    
    return true;
}

/**
 * Check if bot is in paid mode
 * @return bool
 */
function isPaidMode(): bool {
    return (bool)getSettingValue('is_paid_mode', false);
}

/**
 * Toggle paid mode
 * @param bool $enabled
 * @return bool
 */
function togglePaidMode(bool $enabled): bool {
    return setSettingValue('is_paid_mode', $enabled);
}

/**
 * Check if mandatory seen is enabled
 * @return bool
 */
function isMandatorySeenEnabled(): bool {
    return (bool)getSettingValue('mandatory_seen_enabled', false);
}

/**
 * Toggle mandatory seen
 * @param bool $enabled
 * @return bool
 */
function toggleMandatorySeen(bool $enabled): bool {
    return setSettingValue('mandatory_seen_enabled', $enabled);
}

/**
 * Send message with mandatory seen
 * @param int $chat_id
 * @param string $text
 * @param array|null $keyboard
 * @return array|false
 */
function sendMessageWithMandatorySeen(int $chat_id, string $text, ?array $keyboard = null): array|false {
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'message_effect_id' => '👀' // Mandatory seen effect
    ];
    
    if ($keyboard) {
        $data['reply_markup'] = json_encode($keyboard, JSON_UNESCAPED_UNICODE);
    }
    
    return bot('sendMessage', $data);
}

/**
 * Get all feature toggles
 * @return array
 */
function getAllFeatureToggles(): array {
    return [
        [
            'key' => 'is_paid_mode',
            'name' => '💰 حالت پولی',
            'callback' => 'toggle_feature_paid_mode'
        ],
        [
            'key' => 'advertisement_enabled',
            'name' => '🎯 تبلیغات',
            'callback' => 'toggle_feature_advertisement'
        ],
        [
            'key' => 'forced_join_enabled',
            'name' => '📢 جوین اجباری',
            'callback' => 'toggle_feature_forced_join'
        ],
        [
            'key' => 'daily_spin_enabled',
            'name' => '🎰 اسپین روزانه',
            'callback' => 'toggle_feature_daily_spin'
        ],
        [
            'key' => 'referral_enabled',
            'name' => '🎁 سیستم معرف',
            'callback' => 'toggle_feature_referral'
        ],
        [
            'key' => 'mandatory_seen_enabled',
            'name' => '👁 Mandatory Seen',
            'callback' => 'toggle_feature_mandatory_seen'
        ],
        [
            'key' => 'reaction_channels_enabled',
            'name' => '💬 کانال‌های ری‌اکشن',
            'callback' => 'toggle_feature_reaction_channels'
        ],
        [
            'key' => 'promo_codes_enabled',
            'name' => '🎫 کدهای جایزه',
            'callback' => 'toggle_feature_promo_codes'
        ],
        [
            'key' => 'gambling_enabled',
            'name' => '🎲 بازی و شرط‌بندی',
            'callback' => 'toggle_feature_gambling'
        ],
    ];
}

/**
 * Display settings
 * @param int $chat_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displaySettings(int $chat_id, ?int $message_id = null): void {
    $paid_mode = isPaidMode();
    $mandatory_seen = isMandatorySeenEnabled();
    $forced_join_text = getForcedJoinText();
    $stars_rate = getSettingValue('stars_to_points_rate', 1);
    $referral_points_referrer = getSettingValue('referral_points_referrer', 5);
    $referral_points_referred = getSettingValue('referral_points_referred', 3);
    $referral_points_level_1 = getSettingValue('referral_points_level_1', 5);
    $referral_points_level_2 = getSettingValue('referral_points_level_2', 3);
    $referral_points_level_3 = getSettingValue('referral_points_level_3', 1);
    $gambling_enabled = (bool)getSettingValue('gambling_enabled', true);
    $gambling_min_bet = (int)getSettingValue('gambling_min_bet', 1);
    $gambling_max_bet = (int)getSettingValue('gambling_max_bet', 100);
    $gambling_win_multiplier = (float)getSettingValue('gambling_win_multiplier', 2.0);
    $gambling_daily_limit = (int)getSettingValue('gambling_daily_limit', 0);
    $gambling_cooldown_seconds = (int)getSettingValue('gambling_cooldown_seconds', 0);
    
    $message = "⚙️ تنظیمات ربات:\n\n";
    $message .= "💰 حالت پولی: " . ($paid_mode ? '✅ فعال' : '❌ غیرفعال') . "\n";
    $message .= "👁 Mandatory Seen: " . ($mandatory_seen ? '✅ فعال' : '❌ غیرفعال') . "\n";
    $message .= "📢 متن Forced Join: {$forced_join_text}\n";
    $message .= "⭐ نرخ Stars به امتیاز: {$stars_rate}\n";
    $message .= "🎁 امتیاز معرف (معرف): {$referral_points_referrer}\n";
    $message .= "🎁 امتیاز معرف (معرف‌شده): {$referral_points_referred}\n\n";
    $message .= "🎁 امتیاز رفرال چند سطحی:\n";
    $message .= "   🔹 سطح 1: {$referral_points_level_1}\n";
    $message .= "   🔹 سطح 2: {$referral_points_level_2}\n";
    $message .= "   🔹 سطح 3: {$referral_points_level_3}\n\n";
    $message .= "🎲 تنظیمات بازی:\n";
    $message .= "   فعال: " . ($gambling_enabled ? '✅' : '❌') . "\n";
    $message .= "   حداقل شرط: {$gambling_min_bet}\n";
    $message .= "   حداکثر شرط: {$gambling_max_bet}\n";
    $message .= "   نرخ برد: {$gambling_win_multiplier}x\n";
    $message .= "   محدودیت روزانه: " . ($gambling_daily_limit > 0 ? $gambling_daily_limit : 'بدون محدودیت') . "\n";
    $message .= "   زمان انتظار: " . ($gambling_cooldown_seconds > 0 ? $gambling_cooldown_seconds . ' ثانیه' : 'بدون انتظار') . "\n";
    
    $keyboard = getInlineKeyboard('settings');
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Display features list with toggles
 * @param int $chat_id
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayFeaturesList(int $chat_id, ?int $message_id = null): void {
    $features = getAllFeatureToggles();
    $message = "🔄 لیست امکانات ربات:\n\n";
    
    $keyboard_rows = [];
    
    foreach ($features as $feature) {
        $is_enabled = (bool)getSettingValue($feature['key'], false);
        $status_emoji = $is_enabled ? '✅' : '❌';
        $status_text = $is_enabled ? 'فعال' : 'غیرفعال';
        
        $message .= "{$status_emoji} {$feature['name']}: {$status_text}\n";
        
        // Add toggle button for each feature
        $keyboard_rows[] = [
            [
                'text' => "{$status_emoji} {$feature['name']}",
                'callback_data' => $feature['callback']
            ]
        ];
    }
    
    // Add back button
    $keyboard_rows[] = [
        ['text' => '⬅️ بازگشت', 'callback_data' => 'settings']
    ];
    
    $keyboard = [
        'inline_keyboard' => $keyboard_rows
    ];
    
    if ($message_id !== null) {
        // Always try to edit - editMessage will handle "message is not modified" error
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Toggle a feature
 * @param string $feature_key
 * @param int $chat_id
 * @param int|null $message_id
 * @return void
 */
function toggleFeature(string $feature_key, int $chat_id, ?int $message_id = null): void {
    $current = (bool)getSettingValue($feature_key, false);
    $new = !$current;
    setSettingValue($feature_key, $new);
    
    // Refresh features list
    displayFeaturesList($chat_id, $message_id);
}

/**
 * Handle update forced join text
 * @param array $message
 * @return void
 */
function handleUpdateForcedJoinText(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند تنظیمات را تغییر دهند.");
        return;
    }
    
    // Extract new text
    $text = $message['text'] ?? '';
    $new_text = preg_replace('/^\/\w+\s*/', '', $text);
    $new_text = trim($new_text);
    
    if (empty($new_text)) {
        sendMessage($chat_id, "❌ لطفاً متن جدید را ارسال کنید.");
        return;
    }
    
    if (updateForcedJoinText($new_text)) {
        sendMessage($chat_id, "✅ متن Forced Join با موفقیت به‌روزرسانی شد!");
    } else {
        sendMessage($chat_id, "❌ خطا در به‌روزرسانی متن.");
    }
}

/**
 * Handle set stars rate from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param int $rate
 * @return void
 */
function handleSetStarsRateFromPanel(int $chat_id, int $admin_id, int $rate): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند تنظیمات را تغییر دهند.");
        return;
    }
    
    if ($rate <= 0) {
        sendMessage($chat_id, "❌ نرخ باید بزرگ‌تر از صفر باشد.");
        return;
    }
    
    if (setSettingValue('stars_to_points_rate', $rate)) {
        sendMessage($chat_id, "✅ نرخ Stars به امتیاز به {$rate} تنظیم شد!");
        logInfo("Admin {$admin_id} set stars rate to {$rate}");
    } else {
        sendMessage($chat_id, "❌ خطا در تنظیم نرخ.");
    }
}

/**
 * Handle set referral points from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param int $referrer_points
 * @param int $referred_points
 * @return void
 */
function handleSetReferralPointsFromPanel(int $chat_id, int $admin_id, int $referrer_points, int $referred_points): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند تنظیمات را تغییر دهند.");
        return;
    }
    
    if ($referrer_points < 0 || $referred_points < 0) {
        sendMessage($chat_id, "❌ امتیازها نمی‌توانند منفی باشند.");
        return;
    }
    
    if (setSettingValue('referral_points_referrer', $referrer_points) && 
        setSettingValue('referral_points_referred', $referred_points)) {
        sendMessage($chat_id, "✅ امتیازهای معرف به‌روزرسانی شد!\n\n🎁 معرف: {$referrer_points}\n🎁 معرف‌شده: {$referred_points}");
        logInfo("Admin {$admin_id} set referral points: referrer={$referrer_points}, referred={$referred_points}");
    } else {
        sendMessage($chat_id, "❌ خطا در تنظیم امتیازها.");
    }
}

/**
 * Handle set forced join text from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param string $text
 * @return void
 */
function handleSetForcedJoinTextFromPanel(int $chat_id, int $admin_id, string $text): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند تنظیمات را تغییر دهند.");
        return;
    }
    
    $text = trim($text);
    
    if (empty($text)) {
        sendMessage($chat_id, "❌ متن نمی‌تواند خالی باشد.");
        return;
    }
    
    if (updateForcedJoinText($text)) {
        sendMessage($chat_id, "✅ متن Forced Join با موفقیت به‌روزرسانی شد!");
        logInfo("Admin {$admin_id} updated forced join text");
    } else {
        sendMessage($chat_id, "❌ خطا در به‌روزرسانی متن.");
    }
}

