<?php
declare(strict_types=1);

/**
 * File Upload Handler
 * Handles file uploads (single and album) by admins
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';

/**
 * Handle single file upload
 * @param array $message Telegram message object
 * @return void
 */
function handleFileUpload(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    $user_data = $message['from'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند فایل آپلود کنند.");
        return;
    }
    
    // Ensure user exists
    ensureUserExists($user_id, $user_data);
    
    // Extract file information
    $file_info = extractFileInfo($message);
    
    if ($file_info === false) {
        sendMessage($chat_id, "❌ خطا در دریافت اطلاعات فایل.");
        return;
    }
    
    // Save file to database
    $file_id_db = saveFileToDatabase($file_info, $user_id);
    
    if ($file_id_db === false) {
        sendMessage($chat_id, "❌ خطا در ذخیره فایل در دیتابیس.");
        return;
    }
    
    // Log analytics
    logAnalytics('file_uploaded', $user_id, $file_id_db, [
        'file_type' => $file_info['file_type'],
        'file_size' => $file_info['file_size'] ?? 0
    ]);
    
    // Get file code for download link
    $file = getFileById($file_id_db);
    $file_code = $file['file_code'] ?? '';
    
    // Get bot username
    $bot_username = BOT_USERNAME ?? '';
    if (empty($bot_username)) {
        // Try to get from bot info
        $bot_info = bot('getMe', []);
        if ($bot_info !== false && isset($bot_info['ok']) && $bot_info['ok']) {
            $bot_username = $bot_info['result']['username'] ?? '';
        }
    }
    
    // Create download link
    $download_link = '';
    if (!empty($bot_username) && !empty($file_code)) {
        $download_link = "https://t.me/{$bot_username}?start={$file_code}";
    }
    
    $message_text = "✅ فایل با موفقیت آپلود شد!\n";
    $message_text .= "📁 شناسه فایل: {$file_id_db}\n";
    if (!empty($download_link)) {
        $message_text .= "\n🔗 لینک دانلود:\n{$download_link}";
    }
    
    // Create keyboard with download link and auto-delete settings
    $keyboard = [
        'inline_keyboard' => [
            [
                ['text' => '⏱️ تنظیم تایم حذف', 'callback_data' => 'file_autodelete_' . $file_id_db]
            ],
            [
                ['text' => '⬅️ بازگشت', 'callback_data' => 'back_main']
            ]
        ]
    ];
    
    // Try to edit previous message if exists (from upload callback)
    $user_state = dbQueryOne(
        "SELECT state_data FROM user_states WHERE user_id = ? AND state = 'uploading'",
        [$user_id]
    );
    
    if ($user_state !== false && !empty($user_state['state_data'])) {
        $state_data = json_decode($user_state['state_data'], true);
        if ($state_data !== null && isset($state_data['upload_message_id'])) {
            $prev_message_id = (int)$state_data['upload_message_id'];
            // Try to edit previous message
            $edit_result = editMessage($chat_id, $prev_message_id, $message_text, $keyboard);
            if ($edit_result !== false && isset($edit_result['ok']) && $edit_result['ok']) {
                // Clear state
                dbExecute(
                    "UPDATE user_states SET state = NULL, state_data = NULL WHERE user_id = ?",
                    [$user_id]
                );
                return; // Message edited, don't send new one
            }
        }
    }
    
    // If edit failed or no previous message, send new message
    sendMessage($chat_id, $message_text, $keyboard);
    
    // Clear uploading state
    dbExecute(
        "UPDATE user_states SET state = NULL, state_data = NULL WHERE user_id = ? AND state = 'uploading'",
        [$user_id]
    );
}

/**
 * Handle album upload
 * @param array $message Telegram message object
 * @return void
 */
function handleAlbumUpload(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    $user_data = $message['from'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند فایل آپلود کنند.");
        return;
    }
    
    // Ensure user exists
    ensureUserExists($user_id, $user_data);
    
    // Extract album files
    $media_group_id = $message['media_group_id'] ?? null;
    
    if (!$media_group_id) {
        sendMessage($chat_id, "❌ خطا در دریافت اطلاعات آلبوم.");
        return;
    }
    
    $file_info = extractFileInfo($message);
    
    if ($file_info === false) {
        return;
    }
    
    // Save file to database with media_group_id
    $file_id_db = saveFileToDatabase($file_info, $user_id, $media_group_id);
    
    if ($file_id_db === false) {
        return;
    }
    
    // Log analytics
    logAnalytics('file_uploaded', $user_id, $file_id_db, [
        'file_type' => $file_info['file_type'],
        'media_group_id' => $media_group_id,
        'is_album' => true
    ]);
}

/**
 * Extract file information from Telegram message
 * @param array $message
 * @return array|false
 */
function extractFileInfo(array $message): array|false {
    $file_id = null;
    $file_type = null;
    $file_size = null;
    $caption = $message['caption'] ?? null;
    $thumbnail_file_id = null;
    
    // Check for photo
    if (isset($message['photo']) && is_array($message['photo'])) {
        // Get largest photo
        $photos = $message['photo'];
        $largest_photo = end($photos);
        $file_id = $largest_photo['file_id'] ?? null;
        $file_type = 'photo';
        $file_size = $largest_photo['file_size'] ?? null;
    }
    // Check for video
    elseif (isset($message['video'])) {
        $file_id = $message['video']['file_id'] ?? null;
        $file_type = 'video';
        $file_size = $message['video']['file_size'] ?? null;
        $thumbnail_file_id = $message['video']['thumb']['file_id'] ?? null;
    }
    // Check for document
    elseif (isset($message['document'])) {
        $file_id = $message['document']['file_id'] ?? null;
        $file_type = 'document';
        $file_size = $message['document']['file_size'] ?? null;
        $thumbnail_file_id = $message['document']['thumb']['file_id'] ?? null;
    }
    // Check for audio
    elseif (isset($message['audio'])) {
        $file_id = $message['audio']['file_id'] ?? null;
        $file_type = 'audio';
        $file_size = $message['audio']['file_size'] ?? null;
    }
    // Check for voice
    elseif (isset($message['voice'])) {
        $file_id = $message['voice']['file_id'] ?? null;
        $file_type = 'voice';
        $file_size = $message['voice']['file_size'] ?? null;
    }
    // Check for video_note
    elseif (isset($message['video_note'])) {
        $file_id = $message['video_note']['file_id'] ?? null;
        $file_type = 'video_note';
        $file_size = $message['video_note']['file_size'] ?? null;
    }
    // Check for sticker
    elseif (isset($message['sticker'])) {
        $file_id = $message['sticker']['file_id'] ?? null;
        $file_type = 'sticker';
        $file_size = $message['sticker']['file_size'] ?? null;
    }
    // Check for animation
    elseif (isset($message['animation'])) {
        $file_id = $message['animation']['file_id'] ?? null;
        $file_type = 'animation';
        $file_size = $message['animation']['file_size'] ?? null;
        $thumbnail_file_id = $message['animation']['thumb']['file_id'] ?? null;
    }
    
    if ($file_id === null || $file_type === null) {
        return false;
    }
    
    return [
        'file_id' => $file_id,
        'file_type' => $file_type,
        'file_size' => $file_size,
        'caption' => $caption,
        'thumbnail_file_id' => $thumbnail_file_id
    ];
}

/**
 * Save file to database
 * @param array $file_info
 * @param int $uploaded_by
 * @param string|null $media_group_id
 * @return int|false File ID in database
 */
function saveFileToDatabase(array $file_info, int $uploaded_by, ?string $media_group_id = null): int|false {
    $file_id = $file_info['file_id'];
    $file_type = $file_info['file_type'];
    $file_size = $file_info['file_size'];
    $caption = $file_info['caption'];
    $thumbnail_file_id = $file_info['thumbnail_file_id'];
    
    // Validate file type
    if (!validateFileType($file_type)) {
        return false;
    }
    
    // Generate unique file code
    $file_code = generateFileCode();
    
    $result = dbExecute(
        "INSERT INTO files (file_id, file_code, file_type, file_size, caption, thumbnail_file_id, uploaded_by) 
         VALUES (?, ?, ?, ?, ?, ?, ?)",
        [$file_id, $file_code, $file_type, $file_size, $caption, $thumbnail_file_id, $uploaded_by]
    );
    
    if ($result === false) {
        return false;
    }
    
    $db_file_id = dbLastInsertId();
    return $db_file_id ? (int)$db_file_id : false;
}

/**
 * Get file by database ID
 * @param int $file_id
 * @return array|false
 */
function getFileById(int $file_id): array|false {
    return dbQueryOne(
        "SELECT * FROM files WHERE id = ? AND is_active = 1",
        [$file_id]
    );
}

/**
 * Get file by Telegram file_id
 * @param string $telegram_file_id
 * @return array|false
 */
function getFileByTelegramFileId(string $telegram_file_id): array|false {
    return dbQueryOne(
        "SELECT * FROM files WHERE file_id = ? AND is_active = 1",
        [$telegram_file_id]
    );
}

/**
 * Get file by file code
 * @param string $file_code
 * @return array|false
 */
function getFileByCode(string $file_code): array|false {
    return dbQueryOne(
        "SELECT * FROM files WHERE file_code = ? AND is_active = 1",
        [$file_code]
    );
}

/**
 * Get all files
 * @param int $limit
 * @param int $offset
 * @param string|null $sort_by Sort by: 'date', 'popularity', 'downloads', null (default: date)
 * @return array|false
 */
function getAllFiles(int $limit = 50, int $offset = 0, ?string $sort_by = null): array|false {
    $order_by = "ORDER BY created_at DESC";
    
    if ($sort_by === 'popularity') {
        $order_by = "ORDER BY popularity_score DESC, download_count DESC";
    } elseif ($sort_by === 'downloads') {
        $order_by = "ORDER BY download_count DESC, created_at DESC";
    } elseif ($sort_by === 'date') {
        $order_by = "ORDER BY created_at DESC";
    }
    
    return dbQuery(
        "SELECT * FROM files WHERE is_active = 1 {$order_by} LIMIT ? OFFSET ?",
        [$limit, $offset]
    );
}

/**
 * Get files sorted by specific criteria
 * @param string $sort_by Sort by: 'popularity', 'date', 'downloads'
 * @param int $limit
 * @param int $offset
 * @return array|false
 */
function getFilesSortedBy(string $sort_by, int $limit = 50, int $offset = 0): array|false {
    return getAllFiles($limit, $offset, $sort_by);
}

/**
 * Calculate file popularity score
 * Formula: (download_count * 10) - (hours_since_upload * 0.1)
 * This gives more weight to downloads but decreases over time
 * @param int $file_id
 * @return float
 */
function calculateFilePopularity(int $file_id): float {
    $file = getFileById($file_id);
    
    if ($file === false) {
        return 0.0;
    }
    
    $download_count = (int)($file['download_count'] ?? 0);
    $created_at = $file['created_at'] ?? date('Y-m-d H:i:s');
    
    // Calculate hours since upload
    $created_timestamp = strtotime($created_at);
    $current_timestamp = time();
    $hours_since_upload = max(0, ($current_timestamp - $created_timestamp) / 3600);
    
    // Calculate popularity score
    // Downloads are weighted more heavily, but score decreases over time
    $popularity_score = ($download_count * 10.0) - ($hours_since_upload * 0.1);
    
    return max(0.0, $popularity_score);
}

/**
 * Update file popularity score
 * @param int $file_id
 * @return bool
 */
function updateFilePopularity(int $file_id): bool {
    $popularity_score = calculateFilePopularity($file_id);
    
    $result = dbExecute(
        "UPDATE files SET popularity_score = ? WHERE id = ?",
        [$popularity_score, $file_id]
    );
    
    return $result !== false;
}

/**
 * Get recommended files for user
 * Algorithm: Mix of popular files (70%) and new files (30%)
 * @param int $user_id User ID (for future personalization)
 * @param int $limit Number of files to return
 * @return array|false
 */
function getRecommendedFiles(int $user_id, int $limit = 20): array|false {
    // Get 70% popular files
    $popular_limit = (int)ceil($limit * 0.7);
    $popular_files = getFilesSortedBy('popularity', $popular_limit, 0);
    
    // Get 30% new files
    $new_limit = $limit - $popular_limit;
    $new_files = getFilesSortedBy('date', $new_limit, 0);
    
    $recommended = [];
    $file_ids = [];
    
    // Add popular files first
    if ($popular_files !== false) {
        foreach ($popular_files as $file) {
            if (!in_array($file['id'], $file_ids)) {
                $recommended[] = $file;
                $file_ids[] = $file['id'];
            }
        }
    }
    
    // Add new files (avoid duplicates)
    if ($new_files !== false) {
        foreach ($new_files as $file) {
            if (!in_array($file['id'], $file_ids) && count($recommended) < $limit) {
                $recommended[] = $file;
                $file_ids[] = $file['id'];
            }
        }
    }
    
    return !empty($recommended) ? $recommended : false;
}

/**
 * Display file list with pagination
 * @param int $chat_id
 * @param int $user_id
 * @param string $sort_by Sort by: 'popularity', 'date', 'downloads'
 * @param int $page Page number (1-based)
 * @param int|null $message_id Message ID to edit (optional)
 * @return void
 */
function displayFileList(int $chat_id, int $user_id, string $sort_by = 'date', int $page = 1, ?int $message_id = null): void {
    require_once __DIR__ . '/../auth.php';
    require_once __DIR__ . '/channel-handler.php';
    
    // Check forced join first
    if (!checkForcedJoinAndNotify($user_id, $chat_id)) {
        return; // Message already sent
    }
    
    $limit = 10; // Files per page
    $offset = ($page - 1) * $limit;
    
    $files = getFilesSortedBy($sort_by, $limit, $offset);
    
    if ($files === false || empty($files)) {
        $msg = "📑 هیچ فایلی یافت نشد.";
        $keyboard = getInlineKeyboard('file_sorting_menu');
        if ($message_id !== null) {
            editMessage($chat_id, $message_id, $msg, $keyboard);
        } else {
            sendMessage($chat_id, $msg, $keyboard);
        }
        return;
    }
    
    $sort_names = [
        'popularity' => 'محبوبیت',
        'date' => 'تاریخ',
        'downloads' => 'تعداد دانلود'
    ];
    
    $sort_name = $sort_names[$sort_by] ?? 'تاریخ';
    
    $msg = "📑 لیست فایل‌ها (مرتب شده بر اساس {$sort_name}):\n\n";
    $msg .= "💡 روی هر فایل کلیک کنید تا دانلود کنید.\n\n";
    
    $keyboard_buttons = [];
    
    foreach ($files as $index => $file) {
        $file_type_emoji = [
            'photo' => '🖼️',
            'video' => '🎥',
            'document' => '📄',
            'audio' => '🎵',
            'voice' => '🎤',
            'video_note' => '📹',
            'sticker' => '😀',
            'animation' => '🎬'
        ];
        $emoji = $file_type_emoji[$file['file_type']] ?? '📁';
        
        $file_number = $offset + $index + 1;
        $msg .= "{$file_number}. {$emoji} فایل #{$file['id']}\n";
        $msg .= "   📊 دانلود: {$file['download_count']}\n";
        if ($sort_by === 'popularity') {
            $popularity = number_format((float)($file['popularity_score'] ?? 0), 2);
            $msg .= "   ⭐ محبوبیت: {$popularity}\n";
        }
        $msg .= "\n";
        
        $keyboard_buttons[] = [
            [
                'text' => "{$emoji} فایل #{$file['id']}",
                'callback_data' => 'download_file_' . $file['id']
            ]
        ];
    }
    
    // Add pagination buttons
    $total_files = dbQueryOne(
        "SELECT COUNT(*) as count FROM files WHERE is_active = 1",
        []
    );
    $total_count = (int)($total_files['count'] ?? 0);
    $total_pages = max(1, (int)ceil($total_count / $limit));
    
    if ($total_pages > 1) {
        $pagination_row = [];
        if ($page > 1) {
            $pagination_row[] = [
                'text' => '◀️ قبلی',
                'callback_data' => 'sort_files_' . $sort_by . '_page_' . ($page - 1)
            ];
        }
        $pagination_row[] = [
            'text' => "📄 {$page}/{$total_pages}",
            'callback_data' => 'sort_files_' . $sort_by . '_page_' . $page
        ];
        if ($page < $total_pages) {
            $pagination_row[] = [
                'text' => '▶️ بعدی',
                'callback_data' => 'sort_files_' . $sort_by . '_page_' . ($page + 1)
            ];
        }
        $keyboard_buttons[] = $pagination_row;
    }
    
    // Add sorting options
    $keyboard_buttons[] = [
        ['text' => '🔥 محبوبیت', 'callback_data' => 'sort_files_popular_page_1'],
        ['text' => '📅 تاریخ', 'callback_data' => 'sort_files_date_page_1']
    ];
    $keyboard_buttons[] = [
        ['text' => '📊 دانلود', 'callback_data' => 'sort_files_downloads_page_1'],
        ['text' => '⭐ پیشنهادی', 'callback_data' => 'recommended_files']
    ];
    
    // Add back button
    $keyboard_buttons[] = [
        ['text' => '⬅️ بازگشت', 'callback_data' => 'download_file']
    ];
    
    $keyboard = [
        'inline_keyboard' => $keyboard_buttons
    ];
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $msg, $keyboard);
    } else {
        sendMessage($chat_id, $msg, $keyboard);
    }
}

/**
 * Delete file (soft delete)
 * @param int $file_id
 * @return bool
 */
function deleteFile(int $file_id): bool {
    $result = dbExecute(
        "UPDATE files SET is_active = 0 WHERE id = ?",
        [$file_id]
    );
    
    return $result !== false;
}

/**
 * Log analytics event
 * @param string $event_type
 * @param int|null $user_id
 * @param int|null $file_id
 * @param array|null $metadata
 * @return void
 */
function logAnalytics(string $event_type, ?int $user_id = null, ?int $file_id = null, ?array $metadata = null): void {
    $metadata_json = $metadata ? json_encode($metadata, JSON_UNESCAPED_UNICODE) : null;
    
    dbExecute(
        "INSERT INTO analytics (event_type, user_id, file_id, metadata) VALUES (?, ?, ?, ?)",
        [$event_type, $user_id, $file_id, $metadata_json]
    );
}

