<?php
declare(strict_types=1);

/**
 * User Management Handler
 * Handles user management operations
 */

require_once __DIR__ . '/../functions.php';
require_once __DIR__ . '/../auth.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../validators.php';
require_once __DIR__ . '/../logger.php';
require_once __DIR__ . '/point-handler.php';
require_once __DIR__ . '/referral-handler.php';

/**
 * Search users
 * @param string $query Search query (username, name, or user ID)
 * @return array|false
 */
function searchUsers(string $query): array|false {
    $query = trim($query);
    
    // If query is numeric, search by ID
    if (is_numeric($query)) {
        $user_id = validateUserId($query);
        if ($user_id === false) {
            return false;
        }
        
        $user = dbQueryOne(
            "SELECT * FROM users WHERE id = ?",
            [$user_id]
        );
        
        return $user !== false ? [$user] : false;
    }
    
    // Search by username or name
    $search_term = "%{$query}%";
    
    $users = dbQuery(
        "SELECT * FROM users 
         WHERE username LIKE ? OR first_name LIKE ? OR last_name LIKE ?
         LIMIT 20",
        [$search_term, $search_term, $search_term]
    );
    
    return $users;
}

/**
 * Get user information
 * @param int $user_id
 * @return array|false
 */
function getUserInformation(int $user_id): array|false {
    $user = dbQueryOne(
        "SELECT * FROM users WHERE id = ?",
        [$user_id]
    );
    
    if ($user === false) {
        return false;
    }
    
    // Get points
    $points = getUserPoints($user_id);
    
    // Get download count
    $downloads = dbQueryOne(
        "SELECT COUNT(*) as count FROM downloads WHERE user_id = ?",
        [$user_id]
    );
    $download_count = (int)($downloads['count'] ?? 0);
    
    // Get referral stats
    $referral_stats = getReferralStats($user_id);
    
    return [
        'user' => $user,
        'points' => $points,
        'download_count' => $download_count,
        'referral_stats' => $referral_stats
    ];
}

/**
 * Display user information
 * @param int $chat_id
 * @param int $user_id
 * @param int|null $message_id Message ID to edit (optional)
 * @param bool $show_management_buttons Show management buttons (default: true)
 * @return void
 */
function displayUserInfo(int $chat_id, int $user_id, ?int $message_id = null, bool $show_management_buttons = true): void {
    $info = getUserInformation($user_id);
    
    if ($info === false) {
        $msg = "❌ کاربر یافت نشد.";
        if ($message_id !== null) {
            editMessage($chat_id, $message_id, $msg);
        } else {
            sendMessage($chat_id, $msg);
        }
        return;
    }
    
    $user = $info['user'];
    $name = $user['first_name'] . ($user['last_name'] ? ' ' . $user['last_name'] : '');
    $username = $user['username'] ? '@' . $user['username'] : 'بدون یوزرنیم';
    
    $message = "👤 اطلاعات کاربر:\n\n";
    $message .= "👤 نام: {$name}\n";
    $message .= "🆔 شناسه: {$user['id']}\n";
    $message .= "📱 یوزرنیم: {$username}\n";
    $message .= "💎 امتیاز: {$info['points']}\n";
    $message .= "📥 دانلودها: {$info['download_count']}\n";
    $message .= "📅 تاریخ عضویت: {$user['created_at']}\n";
    $message .= "🚫 وضعیت: " . ($user['is_blocked'] ? '🔴 بلاک شده' : '🟢 عادی') . "\n";
    $message .= "⭐ VIP: " . (($user['is_vip'] ?? 0) ? '✅' : '❌') . "\n";
    
    if ($info['referral_stats']['total_referrals'] > 0) {
        $message .= "\n📊 آمار معرف:\n";
        $message .= "   👥 معرف‌ها: {$info['referral_stats']['total_referrals']}\n";
        $message .= "   💎 امتیاز کسب شده: {$info['referral_stats']['total_points_earned']}\n";
    }
    
    $keyboard = $show_management_buttons ? getInlineKeyboard('user_info_menu', 1, ['user_id' => $user_id]) : getInlineKeyboard('back_only');
    
    if ($message_id !== null) {
        editMessage($chat_id, $message_id, $message, $keyboard);
    } else {
        sendMessage($chat_id, $message, $keyboard);
    }
}

/**
 * Handle block user from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param int $target_user_id
 * @return void
 */
function handleBlockUserFromPanel(int $chat_id, int $admin_id, int $target_user_id): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کاربر بلاک کنند.");
        return;
    }
    
    // Cannot block admins
    if (isRealAdmin($target_user_id)) {
        sendMessage($chat_id, "❌ نمی‌توانید ادمین‌ها را بلاک کنید.");
        return;
    }
    
    if (blockUser($target_user_id)) {
        sendMessage($chat_id, "✅ کاربر با موفقیت بلاک شد!");
        logInfo("Admin {$admin_id} blocked user {$target_user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در بلاک کردن کاربر.");
    }
}

/**
 * Handle unblock user from admin panel
 * @param int $chat_id
 * @param int $admin_id
 * @param int $target_user_id
 * @return void
 */
function handleUnblockUserFromPanel(int $chat_id, int $admin_id, int $target_user_id): void {
    if (!isAdmin($admin_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کاربر آنبلاک کنند.");
        return;
    }
    
    if (unblockUser($target_user_id)) {
        sendMessage($chat_id, "✅ کاربر با موفقیت آنبلاک شد!");
        logInfo("Admin {$admin_id} unblocked user {$target_user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در آنبلاک کردن کاربر.");
    }
}

/**
 * Handle search user command
 * @param array $message
 * @return void
 */
function handleSearchUser(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کاربر جستجو کنند.");
        return;
    }
    
    // Extract search query
    $text = $message['text'] ?? '';
    $query = preg_replace('/^\/\w+\s*/', '', $text);
    $query = trim($query);
    
    if (empty($query)) {
        sendMessage($chat_id, "❌ لطفاً یوزرنیم، نام یا شناسه کاربر را ارسال کنید.");
        return;
    }
    
    $users = searchUsers($query);
    
    if ($users === false || empty($users)) {
        sendMessage($chat_id, "❌ کاربری یافت نشد.");
        return;
    }
    
    if (count($users) === 1) {
        // Show detailed info for single result
        displayUserInfo($chat_id, (int)$users[0]['id']);
    } else {
        // Show list of results
        $message = "🔍 نتایج جستجو:\n\n";
        
        foreach ($users as $index => $user) {
            $name = $user['first_name'] . ($user['last_name'] ? ' ' . $user['last_name'] : '');
            $username = $user['username'] ? '@' . $user['username'] : 'بدون یوزرنیم';
            
            $message .= ($index + 1) . ". {$name}\n";
            $message .= "   🆔 {$user['id']}\n";
            $message .= "   📱 {$username}\n\n";
        }
        
        sendMessage($chat_id, $message);
    }
}

/**
 * Check if user is VIP
 * @param int $user_id
 * @return bool
 */
function isUserVIP(int $user_id): bool {
    $user = dbQueryOne(
        "SELECT is_vip FROM users WHERE id = ?",
        [$user_id]
    );
    
    if ($user === false) {
        return false;
    }
    
    return (bool)$user['is_vip'];
}

/**
 * Set user as VIP
 * @param int $user_id
 * @return bool
 */
function setUserVIP(int $user_id): bool {
    // Don't allow setting admins as VIP
    require_once __DIR__ . '/../auth.php';
    if (isRealAdmin($user_id)) {
        return false;
    }
    
    ensureUserExists($user_id);
    
    $result = dbExecute(
        "UPDATE users SET is_vip = 1 WHERE id = ?",
        [$user_id]
    );
    
    return $result !== false;
}

/**
 * Remove VIP from user
 * @param int $user_id
 * @return bool
 */
function removeUserVIP(int $user_id): bool {
    $result = dbExecute(
        "UPDATE users SET is_vip = 0 WHERE id = ?",
        [$user_id]
    );
    
    return $result !== false;
}

/**
 * Handle set VIP command
 * @param array $message
 * @return void
 */
function handleSetVIP(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند کاربر VIP کنند.");
        return;
    }
    
    // Extract user ID from message
    $target_user_id = extractUserIdFromMessage($message);
    
    if ($target_user_id === false) {
        sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
        return;
    }
    
    if (setUserVIP($target_user_id)) {
        sendMessage($chat_id, "✅ کاربر با موفقیت VIP شد!");
    } else {
        sendMessage($chat_id, "❌ خطا در VIP کردن کاربر.");
    }
}

/**
 * Handle remove VIP command
 * @param array $message
 * @return void
 */
function handleRemoveVIP(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند VIP را حذف کنند.");
        return;
    }
    
    // Extract user ID from message
    $target_user_id = extractUserIdFromMessage($message);
    
    if ($target_user_id === false) {
        sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
        return;
    }
    
    if (removeUserVIP($target_user_id)) {
        sendMessage($chat_id, "✅ VIP کاربر با موفقیت حذف شد!");
    } else {
        sendMessage($chat_id, "❌ خطا در حذف VIP کاربر.");
    }
}

/**
 * Extract user ID from message (if not already defined)
 * @param array $message
 * @return int|false
 */
if (!function_exists('extractUserIdFromMessage')) {
    function extractUserIdFromMessage(array $message): int|false {
        // Check if message is a reply
        if (isset($message['reply_to_message']['from']['id'])) {
            return (int)$message['reply_to_message']['from']['id'];
        }
        
        // Check if message contains forwarded user
        if (isset($message['forward_from']['id'])) {
            return (int)$message['forward_from']['id'];
        }
        
        // Check if message text contains user ID
        if (isset($message['text'])) {
            $text = trim($message['text']);
            // Remove command part if exists
            $text = preg_replace('/^\/\w+\s*/', '', $text);
            
            if (is_numeric($text)) {
                $user_id = validateUserId($text);
                if ($user_id !== false) {
                    return $user_id;
                }
            }
        }
        
        return false;
    }
}

