<?php
declare(strict_types=1);

/**
 * Database Migrations
 * Creates all required tables for the uploader bot system
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/logger.php';

/**
 * Run all migrations
 * @return bool
 */
function runMigrations(): bool {
    $db = initDatabase();
    if (!$db) {
        logError("Cannot run migrations: Database connection failed");
        return false;
    }
    
    // Don't use transaction - CREATE TABLE IF NOT EXISTS is safe
    // and some operations might fail if tables already exist
    
    try {
        // Create users table
        createUsersTable($db);
        
        // Create files table
        createFilesTable($db);
        
        // Create admins table
        createAdminsTable($db);
        
        // Create channels table
        createChannelsTable($db);
        
        // Create points table
        createPointsTable($db);
        
        // Create referrals table
        createReferralsTable($db);
        
        // Create settings table
        createSettingsTable($db);
        
        // Create analytics table
        createAnalyticsTable($db);
        
        // Create payments table
        createPaymentsTable($db);
        
        // Create advertisements table
        createAdvertisementsTable($db);
        
        // Create downloads table (for tracking downloads)
        createDownloadsTable($db);
        
        // Create user_states table (for managing user states)
        createUserStatesTable($db);
        
        // Create sent_messages table (for auto-delete)
        createSentMessagesTable($db);
        
        // Create promo_codes table
        createPromoCodesTable($db);
        
        // Create promo_code_usage table
        createPromoCodeUsageTable($db);
        
        // Create daily_spins table
        createDailySpinsTable($db);
        
        // Create advertisement_channels table
        createAdvertisementChannelsTable($db);
        
        // Create reaction_channels table
        createReactionChannelsTable($db);
        
        createUserReactionsTable($db);
        createGamblingTables($db);
        
        logInfo("All migrations completed successfully");
        return true;
    } catch (PDOException $e) {
        logError("Migration failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Create users table
 */
function createUsersTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS users (
        id BIGINT UNSIGNED PRIMARY KEY,
        username VARCHAR(255) NULL,
        first_name VARCHAR(255) NOT NULL,
        last_name VARCHAR(255) NULL,
        is_blocked TINYINT(1) DEFAULT 0,
        is_vip TINYINT(1) DEFAULT 0,
        referral_code VARCHAR(50) UNIQUE NULL,
        referred_by BIGINT UNSIGNED NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_username (username),
        INDEX idx_referral_code (referral_code),
        INDEX idx_referred_by (referred_by),
        INDEX idx_is_vip (is_vip)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Add is_vip column if it doesn't exist (for existing databases)
    try {
        $columns = $db->query("SHOW COLUMNS FROM users LIKE 'is_vip'")->fetchAll();
        if (empty($columns)) {
            $db->exec("ALTER TABLE users ADD COLUMN is_vip TINYINT(1) DEFAULT 0 AFTER is_blocked");
            $db->exec("ALTER TABLE users ADD INDEX idx_is_vip (is_vip)");
        }
    } catch (PDOException $e) {
        // Column might already exist, ignore error
    }
}

/**
 * Create files table
 */
function createFilesTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS files (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        file_id VARCHAR(255) NOT NULL,
        file_code VARCHAR(50) UNIQUE NOT NULL,
        file_type ENUM('photo', 'video', 'document', 'audio', 'voice', 'video_note', 'sticker', 'animation') NOT NULL,
        file_size BIGINT UNSIGNED NULL,
        caption TEXT NULL,
        thumbnail_file_id VARCHAR(255) NULL,
        has_thumbnail TINYINT(1) DEFAULT 1,
        has_link TINYINT(1) DEFAULT 1,
        uploaded_by BIGINT UNSIGNED NOT NULL,
        forced_join_channels JSON NULL,
        auto_delete_at INT UNSIGNED NULL,
        is_active TINYINT(1) DEFAULT 1,
        download_count INT UNSIGNED DEFAULT 0,
        popularity_score DECIMAL(10,2) DEFAULT 0.0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_file_id (file_id),
        INDEX idx_file_code (file_code),
        INDEX idx_uploaded_by (uploaded_by),
        INDEX idx_auto_delete_at (auto_delete_at),
        INDEX idx_is_active (is_active),
        INDEX idx_popularity_score (popularity_score)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Add file_code column if it doesn't exist (for existing databases)
    // Check if column exists first to avoid errors
    try {
        $columns = $db->query("SHOW COLUMNS FROM files LIKE 'file_code'")->fetchAll();
        if (empty($columns)) {
            // Column doesn't exist, add it
            $db->exec("ALTER TABLE files ADD COLUMN file_code VARCHAR(50) UNIQUE NULL AFTER file_id");
            $db->exec("ALTER TABLE files ADD INDEX idx_file_code (file_code)");
            
            // Generate file_code for existing files
            $existing_files = $db->query("SELECT id FROM files WHERE file_code IS NULL")->fetchAll(PDO::FETCH_ASSOC);
            if (!empty($existing_files)) {
                require_once __DIR__ . '/validators.php';
                foreach ($existing_files as $file) {
                    $file_code = generateFileCode();
                    $db->prepare("UPDATE files SET file_code = ? WHERE id = ?")->execute([$file_code, $file['id']]);
                }
            }
            
            // Make file_code NOT NULL after populating
            $db->exec("ALTER TABLE files MODIFY COLUMN file_code VARCHAR(50) NOT NULL");
        }
    } catch (PDOException $e) {
        // Column might already exist or table doesn't exist yet, ignore error
        // This is safe because CREATE TABLE IF NOT EXISTS will handle it
    }
    
    // Migrate auto_delete_at from TIMESTAMP to INT (for existing databases)
    try {
        $column_info = $db->query("SHOW COLUMNS FROM files WHERE Field = 'auto_delete_at'")->fetch(PDO::FETCH_ASSOC);
        if (!empty($column_info) && strpos(strtoupper($column_info['Type']), 'TIMESTAMP') !== false) {
            // Column exists and is TIMESTAMP, convert to INT
            // First, clear any invalid timestamp values (they will be NULL)
            $db->exec("UPDATE files SET auto_delete_at = NULL WHERE auto_delete_at IS NOT NULL AND auto_delete_at NOT REGEXP '^[0-9]+$'");
            
            // Drop index if exists
            try {
                $db->exec("ALTER TABLE files DROP INDEX idx_auto_delete_at");
            } catch (PDOException $e) {
                // Index might not exist, ignore
            }
            
            // Change column type to INT
            $db->exec("ALTER TABLE files MODIFY COLUMN auto_delete_at INT UNSIGNED NULL");
            
            // Recreate index
            $db->exec("ALTER TABLE files ADD INDEX idx_auto_delete_at (auto_delete_at)");
        }
    } catch (PDOException $e) {
        // Migration might fail if column doesn't exist or already migrated, ignore error
        logError("Auto-delete column migration warning: " . $e->getMessage());
    }
    
    // Add popularity_score column if it doesn't exist (for existing databases)
    try {
        $columns = $db->query("SHOW COLUMNS FROM files LIKE 'popularity_score'")->fetchAll();
        if (empty($columns)) {
            $db->exec("ALTER TABLE files ADD COLUMN popularity_score DECIMAL(10,2) DEFAULT 0.0 AFTER download_count");
            $db->exec("ALTER TABLE files ADD INDEX idx_popularity_score (popularity_score)");
            
            // Calculate initial popularity scores for existing files
            $db->exec("UPDATE files SET popularity_score = (download_count * 10.0) + (TIMESTAMPDIFF(HOUR, created_at, NOW()) * -0.1) WHERE is_active = 1");
        }
    } catch (PDOException $e) {
        // Column might already exist, ignore error
        logError("Popularity score column migration warning: " . $e->getMessage());
    }
}

/**
 * Create admins table
 */
function createAdminsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS admins (
        id BIGINT UNSIGNED PRIMARY KEY,
        added_by BIGINT UNSIGNED NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_added_by (added_by)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create channels table
 */
function createChannelsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS channels (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        channel_id BIGINT NOT NULL,
        channel_username VARCHAR(255) NULL,
        channel_title VARCHAR(255) NOT NULL,
        channel_link VARCHAR(500) NULL,
        is_active TINYINT(1) DEFAULT 1,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_channel_id (channel_id),
        INDEX idx_is_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create points table
 */
function createPointsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS points (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        points INT NOT NULL DEFAULT 0,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY unique_user (user_id),
        INDEX idx_user_id (user_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create referrals table
 */
function createReferralsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS referrals (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        referrer_id BIGINT UNSIGNED NOT NULL,
        referred_id BIGINT UNSIGNED NOT NULL,
        points_given_referrer INT DEFAULT 0,
        points_given_referred INT DEFAULT 0,
        referral_level TINYINT UNSIGNED DEFAULT 1,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_referrer_id (referrer_id),
        INDEX idx_referred_id (referred_id),
        INDEX idx_referral_level (referral_level)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Add referral_level column if it doesn't exist (for existing databases)
    try {
        $columns = $db->query("SHOW COLUMNS FROM referrals LIKE 'referral_level'")->fetchAll();
        if (empty($columns)) {
            $db->exec("ALTER TABLE referrals ADD COLUMN referral_level TINYINT UNSIGNED DEFAULT 1 AFTER points_given_referred");
            $db->exec("ALTER TABLE referrals ADD INDEX idx_referral_level (referral_level)");
            
            // Set default level 1 for existing referrals
            $db->exec("UPDATE referrals SET referral_level = 1 WHERE referral_level IS NULL OR referral_level = 0");
        }
    } catch (PDOException $e) {
        // Column might already exist, ignore error
        logError("Referral level column migration warning: " . $e->getMessage());
    }
}

/**
 * Create settings table
 */
function createSettingsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS settings (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT NULL,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_setting_key (setting_key)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Insert default settings
    $defaultSettings = [
        ['is_paid_mode', '0'],
        ['forced_join_text', 'لطفاً ابتدا در کانال‌های زیر عضو شوید:'],
        ['mandatory_seen_enabled', '0'],
        ['stars_to_points_rate', '1'], // 1 star = 1 point
        ['referral_points_referrer', '5'],
        ['referral_points_referred', '3'],
        ['referral_points_level_1', '5'], // Level 1 referral points
        ['referral_points_level_2', '3'], // Level 2 referral points
        ['referral_points_level_3', '1'], // Level 3 referral points
        ['global_auto_delete_seconds', '0'], // 0 = disabled, >0 = seconds
        ['global_forced_join_channels', '[]'], // JSON array of channel IDs
        ['advertisement_enabled', '1'], // Enable advertisements by default
        ['gambling_enabled', '1'], // Enable gambling by default
        ['gambling_min_bet', '1'], // Minimum bet amount
        ['gambling_max_bet', '100'], // Maximum bet amount
        ['gambling_win_multiplier', '2.0'], // Win multiplier (2x = double)
        ['gambling_daily_limit', '0'], // Daily game limit (0 = unlimited)
        ['gambling_cooldown_seconds', '0'], // Cooldown between games in seconds (0 = no cooldown)
    ];
    
    $stmt = $db->prepare("INSERT IGNORE INTO settings (setting_key, setting_value) VALUES (?, ?)");
    foreach ($defaultSettings as $setting) {
        $stmt->execute($setting);
    }
}

/**
 * Create analytics table
 */
function createAnalyticsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS analytics (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        event_type VARCHAR(50) NOT NULL,
        user_id BIGINT UNSIGNED NULL,
        file_id INT UNSIGNED NULL,
        metadata JSON NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_event_type (event_type),
        INDEX idx_user_id (user_id),
        INDEX idx_file_id (file_id),
        INDEX idx_created_at (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create payments table
 */
function createPaymentsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS payments (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        stars_amount INT UNSIGNED NOT NULL,
        points_amount INT UNSIGNED NOT NULL,
        status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
        telegram_payment_id VARCHAR(255) NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_status (status),
        INDEX idx_created_at (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create advertisements table
 */
function createAdvertisementsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS advertisements (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        content TEXT NOT NULL,
        caption TEXT NULL,
        media_type VARCHAR(50) NULL,
        media_file_id VARCHAR(255) NULL,
        ad_url VARCHAR(500) NULL,
        button_text VARCHAR(100) DEFAULT 'خرید تبلیغات',
        is_active TINYINT(1) DEFAULT 1,
        show_count INT UNSIGNED DEFAULT 0,
        click_count INT UNSIGNED DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_is_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Add new columns if they don't exist (for existing databases)
    try {
        $db->exec("ALTER TABLE advertisements ADD COLUMN IF NOT EXISTS caption TEXT NULL AFTER content");
        $db->exec("ALTER TABLE advertisements ADD COLUMN IF NOT EXISTS ad_url VARCHAR(500) NULL AFTER media_file_id");
        $db->exec("ALTER TABLE advertisements ADD COLUMN IF NOT EXISTS button_text VARCHAR(100) DEFAULT 'خرید تبلیغات' AFTER ad_url");
    } catch (PDOException $e) {
        // Columns might already exist, ignore error
    }
}

/**
 * Create downloads table
 */
function createDownloadsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS downloads (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        file_id INT UNSIGNED NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_file_id (file_id),
        INDEX idx_created_at (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create user_states table
 */
function createUserStatesTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS user_states (
        user_id BIGINT UNSIGNED PRIMARY KEY,
        state VARCHAR(100) NULL,
        state_data JSON NULL,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create sent_messages table (for auto-delete)
 */
function createSentMessagesTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS sent_messages (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        chat_id BIGINT NOT NULL,
        message_id INT NOT NULL,
        file_id INT UNSIGNED NULL,
        delete_at TIMESTAMP NOT NULL,
        is_deleted TINYINT(1) DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_chat_id (chat_id),
        INDEX idx_delete_at (delete_at),
        INDEX idx_is_deleted (is_deleted)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create promo_codes table
 */
function createPromoCodesTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS promo_codes (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        code VARCHAR(50) UNIQUE NOT NULL,
        type ENUM('gift', 'discount', 'coin') NOT NULL,
        value INT UNSIGNED NOT NULL,
        usage_limit INT UNSIGNED DEFAULT 0,
        usage_count INT UNSIGNED DEFAULT 0,
        expires_at DATETIME NULL,
        created_by BIGINT UNSIGNED NULL,
        is_active TINYINT(1) DEFAULT 1,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_code (code),
        INDEX idx_type (type),
        INDEX idx_is_active (is_active),
        INDEX idx_expires_at (expires_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create promo_code_usage table
 */
function createPromoCodeUsageTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS promo_code_usage (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        promo_code_id INT UNSIGNED NOT NULL,
        user_id BIGINT UNSIGNED NOT NULL,
        used_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (promo_code_id) REFERENCES promo_codes(id) ON DELETE CASCADE,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        UNIQUE KEY unique_user_code (promo_code_id, user_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create daily_spins table
 */
function createDailySpinsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS daily_spins (
        user_id BIGINT UNSIGNED PRIMARY KEY,
        last_spin_date DATE NOT NULL,
        points_earned INT UNSIGNED DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create advertisement_channels table
 */
function createAdvertisementChannelsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS advertisement_channels (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        channel_id BIGINT NOT NULL,
        channel_username VARCHAR(255) NULL,
        channel_title VARCHAR(255) NOT NULL,
        channel_link VARCHAR(500) NULL,
        button_text VARCHAR(100) DEFAULT '📢 کانال ما',
        display_order INT UNSIGNED DEFAULT 0,
        is_active TINYINT(1) DEFAULT 1,
        click_count INT UNSIGNED DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_channel_id (channel_id),
        INDEX idx_is_active (is_active),
        INDEX idx_display_order (display_order)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create reaction channels table
 */
function createReactionChannelsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS reaction_channels (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        channel_id BIGINT NOT NULL,
        channel_title VARCHAR(255) NOT NULL,
        channel_username VARCHAR(255) NULL,
        channel_link VARCHAR(500) NULL,
        is_active TINYINT(1) DEFAULT 1,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_channel_id (channel_id),
        INDEX idx_is_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create user_reactions table (for tracking reactions on channel posts)
 */
function createUserReactionsTable(PDO $db): void {
    $sql = "CREATE TABLE IF NOT EXISTS user_reactions (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        channel_id BIGINT NOT NULL,
        message_id INT NOT NULL,
        reaction_type VARCHAR(50) NOT NULL DEFAULT 'emoji',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY unique_user_channel_message (user_id, channel_id, message_id),
        INDEX idx_user_id (user_id),
        INDEX idx_channel_id (channel_id),
        INDEX idx_message_id (message_id),
        INDEX idx_user_channel (user_id, channel_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

/**
 * Create gambling tables (games and user_gambling_stats)
 */
function createGamblingTables(PDO $db): void {
    // Create games table
    $sql = "CREATE TABLE IF NOT EXISTS games (
        id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT UNSIGNED NOT NULL,
        game_type VARCHAR(50) NOT NULL,
        bet_amount INT UNSIGNED NOT NULL,
        win_amount INT UNSIGNED DEFAULT 0,
        result ENUM('win', 'lose') NOT NULL,
        game_data JSON NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_game_type (game_type),
        INDEX idx_created_at (created_at),
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
    
    // Create user_gambling_stats table
    $sql = "CREATE TABLE IF NOT EXISTS user_gambling_stats (
        user_id BIGINT UNSIGNED PRIMARY KEY,
        total_games INT UNSIGNED DEFAULT 0,
        total_bet INT UNSIGNED DEFAULT 0,
        total_won INT UNSIGNED DEFAULT 0,
        total_lost INT UNSIGNED DEFAULT 0,
        last_game_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $db->exec($sql);
}

