<?php
declare(strict_types=1);

/**
 * Update Processor
 * Processes Telegram webhook updates and extracts relevant data
 */

require_once __DIR__ . '/auth.php';

/**
 * Process and validate Telegram update
 * @return array|false Returns processed update data or false on failure
 */
function processUpdate(): array|false {
    // Get update from Telegram
    $update = json_decode(file_get_contents("php://input"), true);
    
    // Debug logging (only in development - remove in production)
    // file_put_contents("botlog.txt", print_r($update, true));
    
    if (!$update) {
        http_response_code(400);
        return false;
    }
    
    // Verify webhook secret token if set
    if (defined('WEBHOOK_SECRET') && !empty(WEBHOOK_SECRET)) {
        $headers = getallheaders();
        $secret = $headers['X-Telegram-Bot-Api-Secret-Token'] ?? '';
        
        if ($secret !== WEBHOOK_SECRET) {
            http_response_code(403);
            return false;
        }
    }
    
    // Extract update data
    $message = $update['message'] ?? null;
    $callback_query = $update['callback_query'] ?? null;
    $pre_checkout_query = $update['pre_checkout_query'] ?? null;
    $message_reaction = $update['message_reaction'] ?? null;
    
    $chat_id = null;
    $user_id = null;
    $user_data = null;
    $text = null;
    $chat_type = null;
    $callback_message_id = null;
    
    if ($message) {
        $chat_id = $message['chat']['id'];
        $user_id = $message['from']['id'];
        $user_data = $message['from'];
        $text = $message['text'] ?? null;
        $chat_type = $message['chat']['type'];
    }
    
    if ($callback_query) {
        $chat_id = $callback_query['message']['chat']['id'];
        $user_id = $callback_query['from']['id'];
        $user_data = $callback_query['from'];
        $text = $callback_query['data'] ?? null;
        $callback_message_id = $callback_query['message']['message_id'] ?? null;
    }
    
    // Ensure user exists in database
    if ($user_id) {
        ensureUserExists($user_id, $user_data);
    }
    
    return [
        'update' => $update,
        'message' => $message,
        'callback_query' => $callback_query,
        'pre_checkout_query' => $pre_checkout_query,
        'message_reaction' => $message_reaction,
        'chat_id' => $chat_id,
        'user_id' => $user_id,
        'user_data' => $user_data,
        'text' => $text,
        'chat_type' => $chat_type,
        'callback_message_id' => $callback_message_id
    ];
}

