<?php
declare(strict_types=1);

require_once __DIR__ . '/../src/helpers.php';

if (!function_exists('tg_config')) {
    function tg_config(): array
    {
        static $config = null;
        if ($config === null) {
            $config = config('app');
        }
        return $config;
    }
}

if (!function_exists('tg_api_base')) {
    function tg_api_base(): string
    {
        $config = tg_config();
        $token = $config['bot_token'] ?? '';
        if ($token === '' || $token === 'your-bot-token') {
            throw new RuntimeException('Telegram bot token is missing. Set TELEGRAM_BOT_TOKEN in .env');
        }
        return 'https://api.telegram.org/bot' . $token . '/';
    }
}

if (!function_exists('tg_request')) {
    function tg_request(string $method, array $payload = []): array
    {
        $url = tg_api_base() . $method;
        $ch = curl_init($url);
        if ($ch === false) {
            throw new RuntimeException('Failed to initialize curl');
        }

        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POSTFIELDS => tg_prepare_payload($payload),
            CURLOPT_HTTPHEADER => ['Expect:'],
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_TIMEOUT => 30,
        ]);

        $response = curl_exec($ch);
        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Telegram API request failed: ' . $error);
        }

        $status = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        curl_close($ch);

        $decoded = json_decode($response, true);
        if (!is_array($decoded) || ($decoded['ok'] ?? false) !== true) {
            $description = $decoded['description'] ?? 'Unknown error';
            throw new RuntimeException(sprintf('Telegram API error (%d): %s', $status, $description));
        }

        return $decoded['result'];
    }
}

if (!function_exists('tg_prepare_payload')) {
    function tg_prepare_payload(array $payload): array
    {
        foreach ($payload as $key => $value) {
            if (is_array($value)) {
                $payload[$key] = json_encode($value, JSON_UNESCAPED_UNICODE);
            }
        }
        return $payload;
    }
}

if (!function_exists('tg_send_message')) {
    function tg_send_message(int|string $chatId, string $text, array $options = []): array
    {
        $payload = array_merge([
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ], $options);

        return tg_request('sendMessage', $payload);
    }
}

if (!function_exists('tg_send_photo')) {
    function tg_send_photo(int|string $chatId, string $photo, array $options = []): array
    {
        $payload = array_merge([
            'chat_id' => $chatId,
            'photo' => $photo,
        ], $options);

        return tg_request('sendPhoto', $payload);
    }
}

if (!function_exists('tg_send_document')) {
    function tg_send_document(int|string $chatId, string $document, array $options = []): array
    {
        $payload = array_merge([
            'chat_id' => $chatId,
            'document' => $document,
        ], $options);

        return tg_request('sendDocument', $payload);
    }
}

if (!function_exists('tg_send_video')) {
    function tg_send_video(int|string $chatId, string $video, array $options = []): array
    {
        $payload = array_merge([
            'chat_id' => $chatId,
            'video' => $video,
        ], $options);

        return tg_request('sendVideo', $payload);
    }
}

if (!function_exists('tg_send_media_group')) {
    function tg_send_media_group(int|string $chatId, array $mediaItems, array $options = []): array
    {
        $payload = array_merge([
            'chat_id' => $chatId,
            'media' => $mediaItems,
        ], $options);

        return tg_request('sendMediaGroup', $payload);
    }
}

if (!function_exists('tg_delete_message')) {
    function tg_delete_message(int|string $chatId, int $messageId): array
    {
        return tg_request('deleteMessage', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
        ]);
    }
}

if (!function_exists('tg_answer_callback')) {
    function tg_answer_callback(string $callbackId, string $text = '', bool $alert = false): array
    {
        $payload = [
            'callback_query_id' => $callbackId,
            'show_alert' => $alert,
        ];
        if ($text !== '') {
            $payload['text'] = $text;
        }

        return tg_request('answerCallbackQuery', $payload);
    }
}

if (!function_exists('tg_edit_text')) {
    function tg_edit_text(int|string $chatId, int $messageId, string $text, array $options = []): array
    {
        $payload = array_merge([
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ], $options);

        return tg_request('editMessageText', $payload);
    }
}

if (!function_exists('tg_edit_markup')) {
    function tg_edit_markup(int|string $chatId, int $messageId, array $markup): array
    {
        return tg_request('editMessageReplyMarkup', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'reply_markup' => $markup,
        ]);
    }
}

if (!function_exists('tg_chat_action')) {
    function tg_chat_action(int|string $chatId, string $action): array
    {
        return tg_request('sendChatAction', [
            'chat_id' => $chatId,
            'action' => $action,
        ]);
    }
}

if (!function_exists('tg_get_file')) {
    function tg_get_file(string $fileId): array
    {
        return tg_request('getFile', [
            'file_id' => $fileId,
        ]);
    }
}

if (!function_exists('tg_download_file')) {
    function tg_download_file(string $filePath, ?string $destination = null): string
    {
        $config = tg_config();
        $token = $config['bot_token'] ?? '';
        $url = sprintf('https://api.telegram.org/file/bot%s/%s', $token, ltrim($filePath, '/'));

        $data = file_get_contents($url);
        if ($data === false) {
            throw new RuntimeException('Failed to download telegram file: ' . $filePath);
        }

        if ($destination !== null) {
            file_put_contents($destination, $data);
        }

        return $data;
    }
}

