<?php
declare(strict_types=1);

final class Router
{
    /** @var array<string, callable> */
    private array $commands = [];

    /** @var array<string, callable> */
    private array $callbacks = [];

    public function register(string $command, callable $handler): void
    {
        $this->commands[$command] = $handler;
    }

    public function registerCallback(string $prefix, callable $handler): void
    {
        $this->callbacks[$prefix] = $handler;
    }

    public function dispatch(array $update): void
    {
        if (isset($update['callback_query'])) {
            $this->dispatchCallback($update['callback_query']);
            return;
        }

        $message = $update['message'] ?? $update['edited_message'] ?? null;
        if (!$message) {
            return;
        }
        $text = trim((string)($message['text'] ?? ''));
        if ($text !== '' && str_starts_with($text, '/')) {
            $parts = explode(' ', $text, 2);
            $command = strtolower($parts[0]);
            if (isset($this->commands[$command])) {
                ($this->commands[$command])($message, $parts[1] ?? '');
                return;
            }
        }

        if (isset($this->commands['*'])) {
            ($this->commands['*'])($message, $text);
        }
    }

    private function dispatchCallback(array $callback): void
    {
        $data = (string)($callback['data'] ?? '');
        if ($data === '') {
            return;
        }
        foreach ($this->callbacks as $prefix => $handler) {
            if (str_starts_with($data, $prefix)) {
                $payload = substr($data, strlen($prefix));
                $handler($callback, $payload);
                return;
            }
        }
    }
}

