<?php
declare(strict_types=1);

final class TelegramClient
{
    private string $apiBase;

    public function __construct(private readonly string $token)
    {
        $this->apiBase = 'https://api.telegram.org/bot' . $this->token . '/';
    }

    public function sendMessage(int|string $chatId, string $text, array $options = []): array
    {
        return $this->request('sendMessage', array_merge([
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ], $options));
    }

    public function sendPhoto(int|string $chatId, string $photo, array $options = []): array
    {
        return $this->request('sendPhoto', array_merge([
            'chat_id' => $chatId,
            'photo' => $photo,
        ], $options));
    }

    public function sendDocument(int|string $chatId, string $fileId, array $options = []): array
    {
        return $this->request('sendDocument', array_merge([
            'chat_id' => $chatId,
            'document' => $fileId,
        ], $options));
    }

    public function deleteMessage(int|string $chatId, int $messageId): array
    {
        return $this->request('deleteMessage', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
        ]);
    }

    public function answerCallback(string $callbackId, string $text = ''): array
    {
        $payload = ['callback_query_id' => $callbackId];
        if ($text !== '') {
            $payload['text'] = $text;
        }
        return $this->request('answerCallbackQuery', $payload);
    }

    public function editMessageReplyMarkup(int|string $chatId, int $messageId, array $markup): array
    {
        return $this->request('editMessageReplyMarkup', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'reply_markup' => json_encode($markup, JSON_UNESCAPED_UNICODE),
        ]);
    }

    private function request(string $method, array $payload): array
    {
        $url = $this->apiBase . $method;
        $ch = curl_init($url);
        if ($ch === false) {
            throw new RuntimeException('Curl init failed');
        }

        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POSTFIELDS => $this->normalizePayload($payload),
            CURLOPT_HTTPHEADER => ['Expect:'],
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_TIMEOUT => 30,
        ]);

        $response = curl_exec($ch);
        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Telegram request error: ' . $error);
        }

        $status = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        curl_close($ch);

        $decoded = json_decode($response, true);
        if (!is_array($decoded) || ($decoded['ok'] ?? false) !== true) {
            $description = $decoded['description'] ?? 'Unknown error';
            throw new RuntimeException(sprintf('Telegram API error (%d): %s', $status, $description));
        }

        return $decoded['result'];
    }

    private function normalizePayload(array $payload): array
    {
        foreach ($payload as $key => $value) {
            if (is_array($value)) {
                $payload[$key] = json_encode($value, JSON_UNESCAPED_UNICODE);
            }
        }

        return $payload;
    }
}

