<?php
declare(strict_types=1);

final class ChannelRepository
{
    public function __construct(private readonly Database $db)
    {
    }

    public function all(): array
    {
        return $this->db->fetchAll('SELECT * FROM channels ORDER BY is_default DESC, title ASC');
    }

    public function find(int|string $id): ?array
    {
        return $this->db->fetch('SELECT * FROM channels WHERE id = :id', [':id' => $id]);
    }

    public function upsert(array $channel): void
    {
        $sql = 'INSERT INTO channels (id, title, type, active, is_default) 
                VALUES (:id, :title, :type, :active, :is_default)
                ON DUPLICATE KEY UPDATE title = VALUES(title), type = VALUES(type), active = VALUES(active)';
        $this->db->query($sql, [
            ':id' => $channel['id'],
            ':title' => $channel['title'],
            ':type' => $channel['type'],
            ':active' => $channel['active'],
            ':is_default' => $channel['is_default'] ?? 0,
        ]);
    }

    public function markDefault(int|string $id): void
    {
        $this->db->query('UPDATE channels SET is_default = CASE WHEN id = :id THEN 1 ELSE 0 END', [':id' => $id]);
    }

    public function getDefault(): ?array
    {
        return $this->db->fetch('SELECT * FROM channels WHERE is_default = 1 LIMIT 1');
    }
}

