<?php
declare(strict_types=1);

final class UserRepository
{
    public function __construct(private readonly Database $db)
    {
    }

    public function find(int $id): ?array
    {
        return $this->db->fetch('SELECT * FROM users WHERE id = :id', [':id' => $id]);
    }

    public function create(array $data): array
    {
        $sql = 'INSERT INTO users (id, username, role, download_limit, subscription_expires_at, keyboard_type, theme) 
                VALUES (:id, :username, :role, :download_limit, :subscription_expires_at, :keyboard_type, :theme)';
        $this->db->query($sql, [
            ':id' => $data['id'],
            ':username' => $data['username'],
            ':role' => $data['role'],
            ':download_limit' => $data['download_limit'],
            ':subscription_expires_at' => $data['subscription_expires_at'],
            ':keyboard_type' => $data['keyboard_type'],
            ':theme' => $data['theme'],
        ]);
        return $this->find($data['id']);
    }

    public function updatePreferences(int $userId, array $preferences): void
    {
        $allowed = ['keyboard_type', 'theme'];
        $fields = [];
        $params = [':id' => $userId];
        foreach ($preferences as $key => $value) {
            if (!in_array($key, $allowed, true)) {
                continue;
            }
            $fields[] = sprintf('%s = :%s', $key, $key);
            $params[':' . $key] = $value;
        }
        if ($fields === []) {
            return;
        }
        $sql = 'UPDATE users SET ' . implode(', ', $fields) . ' WHERE id = :id';
        $this->db->query($sql, $params);
    }

    public function recordDownload(int $userId, int $mediaId): void
    {
        $sql = 'INSERT INTO downloads (user_id, media_id) VALUES (:user_id, :media_id)';
        $this->db->query($sql, [
            ':user_id' => $userId,
            ':media_id' => $mediaId,
        ]);
    }

    public function downloadsCount(int $userId, string $since): int
    {
        $result = $this->db->fetch(
            'SELECT COUNT(*) AS total FROM downloads WHERE user_id = :user_id AND created_at >= :since',
            [
                ':user_id' => $userId,
                ':since' => $since,
            ]
        );
        return (int)($result['total'] ?? 0);
    }
}

