<?php
declare(strict_types=1);

final class ChannelService
{
    /** @var array<string> */
    private array $allowedTypes = ['announce', 'advertise', 'fun'];

    public function __construct(
        private readonly ChannelRepository $repository,
        private readonly array $appConfig
    ) {
    }

    public function summary(): string
    {
        $channels = $this->repository->all();
        if ($channels === []) {
            return 'هیچ کانالی ثبت نشده است. از دستور /addchannel استفاده کنید.';
        }
        $lines = [];
        foreach ($channels as $channel) {
            $marker = ((int)$channel['is_default'] === 1) ? '✅' : '▫️';
            $lines[] = sprintf(
                '%s %s (%s) - ID: %s',
                $marker,
                $channel['title'],
                $channel['type'],
                $channel['id']
            );
        }
        return implode(PHP_EOL, $lines);
    }

    public function addChannel(int|string $chatId, string $title, string $type = 'announce'): void
    {
        $type = $this->normalizeType($type);
        $this->repository->upsert([
            'id' => $chatId,
            'title' => $title,
            'type' => $type,
            'active' => 1,
            'is_default' => 0,
        ]);
    }

    public function setDefault(int|string $chatId): bool
    {
        if (!$this->repository->find($chatId)) {
            return false;
        }
        $this->repository->markDefault($chatId);
        return true;
    }

    public function defaultChannelId(): int|string
    {
        $channel = $this->repository->getDefault();
        return $channel['id'] ?? $this->appConfig['default_channel_id'];
    }

    private function normalizeType(string $type): string
    {
        $type = strtolower(trim($type));
        return in_array($type, $this->allowedTypes, true) ? $type : 'announce';
    }
}

