<?php
declare(strict_types=1);

final class MediaService
{
    public function __construct(
        private readonly MediaRepository $repository,
        private readonly TelegramClient $client,
        private readonly ChannelService $channelService,
        private readonly array $appConfig
    ) {
    }

    public function handleNewUpload(array $message): void
    {
        $from = $message['from'] ?? [];
        $ownerId = (int)($from['id'] ?? 0);
        $file = $message['video'] ?? $message['document'] ?? null;
        if (!$file) {
            return;
        }
        $fileId = $file['file_id'];
        $caption = $message['caption'] ?? 'بدون توضیحات';

        $channelId = $this->channelService->defaultChannelId();
        $mediaId = $this->repository->store([
            'file_id' => $fileId,
            'title' => $caption,
            'description' => $caption,
            'cover' => '',
            'owner_id' => $ownerId,
            'channel_id' => $channelId,
        ]);

        $text = "آپلود ذخیره شد. شناسه: #{$mediaId}\nبرای اعلان کانال از /publish {$mediaId} استفاده کنید.";
        $this->client->sendMessage($ownerId, $text);
        $this->publishToChannel($mediaId, (string)$channelId);
    }

    public function latestList(int $limit = 5): string
    {
        $items = $this->repository->latest($limit);
        if ($items === []) {
            return '';
        }
        $lines = [];
        foreach ($items as $item) {
            $lines[] = sprintf("#%d - %s", $item['id'], $item['title']);
        }
        $lines[] = '';
        $lines[] = 'برای دریافت هر آیتم از دستور /get <id> استفاده کنید.';
        return implode(PHP_EOL, $lines);
    }

    public function sendMediaToUser(int|string $chatId, int $mediaId): bool
    {
        $media = $this->repository->find($mediaId);
        if (!$media) {
            return false;
        }
        $this->client->sendDocument($chatId, $media['file_id'], [
            'caption' => $media['description'] ?? $media['title'],
        ]);
        return true;
    }

    public function publishToChannel(int $mediaId, int|string|null $channelId = null): bool
    {
        $media = $this->repository->find($mediaId);
        if (!$media) {
            return false;
        }
        $targetChannelId = $channelId ?? $media['channel_id'] ?? $this->channelService->defaultChannelId();
        $text = $this->buildAnnouncement($media);
        $this->client->sendMessage($targetChannelId, $text);
        return true;
    }

    private function buildAnnouncement(array $media): string
    {
        $botUsername = $this->appConfig['bot_username'];
        return sprintf(
            "%s\n\n%s\n\nدریافت: پیام به @%s با دستور /get %d",
            $media['title'],
            $media['description'] ?? '',
            $botUsername,
            $media['id']
        );
    }
}

