<?php
declare(strict_types=1);

require_once __DIR__ . '/../../inc/components.php';

final class SettingsService
{
    private array $themes;
    private array $keyboardTypes;

    public function __construct(private readonly UserRepository $users)
    {
        $this->themes = component_themes();
        $this->keyboardTypes = component_keyboard_types();
    }

    public function themeOptions(): array
    {
        return $this->themes;
    }

    public function keyboardOptions(): array
    {
        return $this->keyboardTypes;
    }

    public function setTheme(int $userId, string $theme): bool
    {
        if (!isset($this->themes[$theme])) {
            return false;
        }
        $this->users->updatePreferences($userId, ['theme' => $theme]);
        return true;
    }

    public function setKeyboard(int $userId, string $type): bool
    {
        if (!isset($this->keyboardTypes[$type])) {
            return false;
        }
        $this->users->updatePreferences($userId, ['keyboard_type' => $type]);
        return true;
    }

    public function detectTheme(string $input): ?string
    {
        foreach ($this->themes as $key => $info) {
            if (strcasecmp($key, $input) === 0) {
                return $key;
            }
            foreach ($info['aliases'] as $alias) {
                if (mb_strtolower($alias) === mb_strtolower($input)) {
                    return $key;
                }
            }
        }
        return null;
    }

    public function detectKeyboard(string $input): ?string
    {
        foreach ($this->keyboardTypes as $key => $info) {
            if (strcasecmp($key, $input) === 0) {
                return $key;
            }
            foreach ($info['aliases'] as $alias) {
                if (mb_strtolower($alias) === mb_strtolower($input)) {
                    return $key;
                }
            }
        }
        return null;
    }

    public function describeTheme(string $theme): string
    {
        return $this->themes[$theme]['description'] ?? '';
    }

    public function describeKeyboard(string $type): string
    {
        return $this->keyboardTypes[$type]['description'] ?? '';
    }

    public function summary(array $user): string
    {
        $themeKey = $user['theme'] ?? 'classic';
        $keyboardKey = $user['keyboard_type'] ?? 'reply';
        $themeLabel = $this->themes[$themeKey]['label'] ?? $themeKey;
        $keyboardLabel = $this->keyboardTypes[$keyboardKey]['label'] ?? $keyboardKey;

        return sprintf(
            "تنظیمات فعلی:\n• تم: %s\n• کیبورد: %s\n\nبرای تغییر، یکی از دکمه‌ها را بزن.",
            $themeLabel,
            $keyboardLabel
        );
    }

    public function decorate(string $text, ?string $theme): string
    {
        $theme = $theme ?? 'classic';
        $config = $this->themes[$theme] ?? $this->themes['classic'];
        $prefix = $config['prefix'] !== '' ? $config['prefix'] . ' ' : '';
        $suffix = $config['suffix'] !== '' ? ' ' . $config['suffix'] : '';
        return $prefix . $text . $suffix;
    }
}

