<?php
declare(strict_types=1);

/**
 * Admin Management Handler
 * Handles admin management operations
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/validators.php';
require_once __DIR__ . '/../inc/logger.php';

/**
 * Handle add admin command
 * @param array $message Telegram message object
 * @return void
 */
function handleAddAdmin(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is main admin
    if (!isMainAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین اصلی می‌تواند ادمین اضافه کند.");
        return;
    }
    
    // Extract user ID from message (reply to message or text)
    $new_admin_id = extractUserIdFromMessage($message);
    
    if ($new_admin_id === false) {
        sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
        return;
    }
    
    // Add admin
    $result = addAdmin($new_admin_id, $user_id);
    
    if ($result) {
        sendMessage($chat_id, "✅ ادمین با موفقیت اضافه شد!");
        logInfo("Admin added: {$new_admin_id} by {$user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در افزودن ادمین. ممکن است این کاربر قبلاً ادمین باشد.");
    }
}

/**
 * Handle remove admin command
 * @param array $message Telegram message object
 * @return void
 */
function handleRemoveAdmin(array $message): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is main admin
    if (!isMainAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین اصلی می‌تواند ادمین حذف کند.");
        return;
    }
    
    // Extract user ID from message
    $admin_to_remove = extractUserIdFromMessage($message);
    
    if ($admin_to_remove === false) {
        sendMessage($chat_id, "❌ لطفاً یک کاربر را forward کنید یا شناسه کاربر را ارسال کنید.");
        return;
    }
    
    // Cannot remove main admin
    if (isMainAdmin($admin_to_remove)) {
        sendMessage($chat_id, "❌ نمی‌توانید ادمین اصلی را حذف کنید.");
        return;
    }
    
    // Remove admin
    $result = removeAdmin($admin_to_remove);
    
    if ($result) {
        sendMessage($chat_id, "✅ ادمین با موفقیت حذف شد!");
        logInfo("Admin removed: {$admin_to_remove} by {$user_id}");
    } else {
        sendMessage($chat_id, "❌ خطا در حذف ادمین.");
    }
}

/**
 * Handle list admins command
 * @param int $chat_id
 * @return void
 */
function handleListAdmins(int $chat_id): void {
    $admins = getAllAdmins();
    
    if (empty($admins)) {
        sendMessage($chat_id, "📋 لیست ادمین‌ها خالی است.");
        return;
    }
    
    $message = "📋 لیست ادمین‌ها:\n\n";
    
    foreach ($admins as $index => $admin) {
        $admin_info = getUserInfo($admin['id']);
        $name = $admin_info['first_name'] ?? 'Unknown';
        $username = $admin_info['username'] ? '@' . $admin_info['username'] : 'بدون یوزرنیم';
        $type = $admin['is_main'] ? '🔴 ادمین اصلی' : '🟢 ادمین';
        
        $message .= ($index + 1) . ". {$type}\n";
        $message .= "   👤 نام: {$name}\n";
        $message .= "   🆔 شناسه: {$admin['id']}\n";
        $message .= "   📱 یوزرنیم: {$username}\n\n";
    }
    
    sendMessage($chat_id, $message);
}

/**
 * Extract user ID from message
 * @param array $message
 * @return int|false
 */
function extractUserIdFromMessage(array $message): int|false {
    // Check if message is a reply
    if (isset($message['reply_to_message']['from']['id'])) {
        return (int)$message['reply_to_message']['from']['id'];
    }
    
    // Check if message contains forwarded user
    if (isset($message['forward_from']['id'])) {
        return (int)$message['forward_from']['id'];
    }
    
    // Check if message text contains user ID
    if (isset($message['text'])) {
        $text = trim($message['text']);
        // Remove command part if exists
        $text = preg_replace('/^\/\w+\s*/', '', $text);
        
        if (is_numeric($text)) {
            $user_id = validateUserId($text);
            if ($user_id !== false) {
                return $user_id;
            }
        }
    }
    
    return false;
}

/**
 * Get user info from database
 * @param int $user_id
 * @return array
 */
function getUserInfo(int $user_id): array {
    $user = dbQueryOne(
        "SELECT * FROM users WHERE id = ?",
        [$user_id]
    );
    
    if ($user === false) {
        return [];
    }
    
    return $user;
}

