<?php
declare(strict_types=1);

/**
 * Advertisement Handler
 * Handles advertisement management
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/logger.php';

/**
 * Add advertisement
 * @param array $message Admin's message containing ad
 * @param string $title Ad title
 * @return int|false Advertisement ID
 */
function addAdvertisement(array $message, string $title): int|false {
    $content = $message['text'] ?? '';
    $media_type = null;
    $media_file_id = null;
    
    // Extract media if exists
    if (isset($message['photo'])) {
        $photos = is_array($message['photo']) ? $message['photo'] : [$message['photo']];
        $largest_photo = end($photos);
        $media_type = 'photo';
        $media_file_id = $largest_photo['file_id'];
        $content = $message['caption'] ?? $content;
    } elseif (isset($message['video'])) {
        $media_type = 'video';
        $media_file_id = $message['video']['file_id'];
        $content = $message['caption'] ?? $content;
    } elseif (isset($message['document'])) {
        $media_type = 'document';
        $media_file_id = $message['document']['file_id'];
        $content = $message['caption'] ?? $content;
    }
    
    $result = dbExecute(
        "INSERT INTO advertisements (title, content, media_type, media_file_id) 
         VALUES (?, ?, ?, ?)",
        [$title, $content, $media_type, $media_file_id]
    );
    
    if ($result === false) {
        return false;
    }
    
    $ad_id = dbLastInsertId();
    return $ad_id ? (int)$ad_id : false;
}

/**
 * Delete advertisement
 * @param int $ad_id
 * @return bool
 */
function deleteAdvertisement(int $ad_id): bool {
    $result = dbExecute(
        "DELETE FROM advertisements WHERE id = ?",
        [$ad_id]
    );
    
    return $result !== false;
}

/**
 * Get active advertisements
 * @return array|false
 */
function getActiveAdvertisements(): array|false {
    return dbQuery(
        "SELECT * FROM advertisements WHERE is_active = 1 ORDER BY created_at DESC"
    );
}

/**
 * Get advertisement by ID
 * @param int $ad_id
 * @return array|false
 */
function getAdvertisementById(int $ad_id): array|false {
    return dbQueryOne(
        "SELECT * FROM advertisements WHERE id = ?",
        [$ad_id]
    );
}

/**
 * Toggle advertisement active status
 * @param int $ad_id
 * @return bool
 */
function toggleAdvertisement(int $ad_id): bool {
    $ad = getAdvertisementById($ad_id);
    
    if ($ad === false) {
        return false;
    }
    
    $new_status = $ad['is_active'] ? 0 : 1;
    
    $result = dbExecute(
        "UPDATE advertisements SET is_active = ? WHERE id = ?",
        [$new_status, $ad_id]
    );
    
    return $result !== false;
}

/**
 * Show advertisement to user
 * @param int $user_id
 * @return bool
 */
function showAdvertisementToUser(int $user_id): bool {
    $ads = getActiveAdvertisements();
    
    if ($ads === false || empty($ads)) {
        return false;
    }
    
    // Get random ad or first ad
    $ad = $ads[array_rand($ads)];
    
    // Send ad to user
    $result = false;
    
    if ($ad['media_type'] === 'photo' && $ad['media_file_id']) {
        $result = bot('sendPhoto', [
            'chat_id' => $user_id,
            'photo' => $ad['media_file_id'],
            'caption' => $ad['content']
        ]);
    } elseif ($ad['media_type'] === 'video' && $ad['media_file_id']) {
        $result = bot('sendVideo', [
            'chat_id' => $user_id,
            'video' => $ad['media_file_id'],
            'caption' => $ad['content']
        ]);
    } elseif ($ad['media_type'] === 'document' && $ad['media_file_id']) {
        $result = bot('sendDocument', [
            'chat_id' => $user_id,
            'document' => $ad['media_file_id'],
            'caption' => $ad['content']
        ]);
    } elseif (!empty($ad['content'])) {
        $result = sendMessage($user_id, $ad['content']);
    }
    
    if ($result !== false && isset($result['ok']) && $result['ok']) {
        // Update show count
        dbExecute(
            "UPDATE advertisements SET show_count = show_count + 1 WHERE id = ?",
            [$ad['id']]
        );
        
        // Log analytics
        logAnalytics('advertisement_shown', $user_id, null, [
            'ad_id' => $ad['id']
        ]);
        
        return true;
    }
    
    return false;
}

/**
 * List all advertisements
 * @param int $chat_id
 * @return void
 */
function listAdvertisements(int $chat_id): void {
    $ads = dbQuery("SELECT * FROM advertisements ORDER BY created_at DESC");
    
    if ($ads === false || empty($ads)) {
        sendMessage($chat_id, "📋 لیست تبلیغات خالی است.");
        return;
    }
    
    $message = "📋 لیست تبلیغات:\n\n";
    
    foreach ($ads as $index => $ad) {
        $status = $ad['is_active'] ? '✅ فعال' : '❌ غیرفعال';
        $message .= ($index + 1) . ". {$ad['title']} ({$status})\n";
        $message .= "   👁 نمایش: {$ad['show_count']}\n";
        $message .= "   🆔 شناسه: {$ad['id']}\n\n";
    }
    
    sendMessage($chat_id, $message);
}

