<?php
declare(strict_types=1);

/**
 * Analytics Handler
 * Handles analytics and statistics
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/logger.php';

/**
 * Get download statistics
 * @return array
 */
function getDownloadStats(): array {
    $stats = [];
    
    // Total downloads
    $total = dbQueryOne("SELECT COUNT(*) as count FROM downloads");
    $stats['total_downloads'] = (int)($total['count'] ?? 0);
    
    // Downloads per file
    $per_file = dbQuery(
        "SELECT file_id, COUNT(*) as count 
         FROM downloads 
         GROUP BY file_id 
         ORDER BY count DESC 
         LIMIT 10"
    );
    $stats['downloads_per_file'] = $per_file !== false ? $per_file : [];
    
    // Downloads today
    $today = dbQueryOne(
        "SELECT COUNT(*) as count FROM downloads WHERE DATE(created_at) = CURDATE()"
    );
    $stats['downloads_today'] = (int)($today['count'] ?? 0);
    
    // Downloads this week
    $week = dbQueryOne(
        "SELECT COUNT(*) as count FROM downloads WHERE WEEK(created_at) = WEEK(NOW())"
    );
    $stats['downloads_this_week'] = (int)($week['count'] ?? 0);
    
    // Downloads this month
    $month = dbQueryOne(
        "SELECT COUNT(*) as count FROM downloads WHERE MONTH(created_at) = MONTH(NOW())"
    );
    $stats['downloads_this_month'] = (int)($month['count'] ?? 0);
    
    return $stats;
}

/**
 * Get user statistics
 * @return array
 */
function getUserStats(): array {
    $stats = [];
    
    // Total users
    $total = dbQueryOne("SELECT COUNT(*) as count FROM users");
    $stats['total_users'] = (int)($total['count'] ?? 0);
    
    // Active users (users who downloaded at least once)
    $active = dbQueryOne(
        "SELECT COUNT(DISTINCT user_id) as count FROM downloads"
    );
    $stats['active_users'] = (int)($active['count'] ?? 0);
    
    // New users today
    $today = dbQueryOne(
        "SELECT COUNT(*) as count FROM users WHERE DATE(created_at) = CURDATE()"
    );
    $stats['new_users_today'] = (int)($today['count'] ?? 0);
    
    // New users this week
    $week = dbQueryOne(
        "SELECT COUNT(*) as count FROM users WHERE WEEK(created_at) = WEEK(NOW())"
    );
    $stats['new_users_this_week'] = (int)($week['count'] ?? 0);
    
    // New users this month
    $month = dbQueryOne(
        "SELECT COUNT(*) as count FROM users WHERE MONTH(created_at) = MONTH(NOW())"
    );
    $stats['new_users_this_month'] = (int)($month['count'] ?? 0);
    
    // Users with points distribution
    $points_dist = dbQuery(
        "SELECT 
            CASE 
                WHEN points = 0 THEN '0'
                WHEN points BETWEEN 1 AND 10 THEN '1-10'
                WHEN points BETWEEN 11 AND 50 THEN '11-50'
                WHEN points BETWEEN 51 AND 100 THEN '51-100'
                ELSE '100+'
            END as range,
            COUNT(*) as count
         FROM points
         GROUP BY range"
    );
    $stats['points_distribution'] = $points_dist !== false ? $points_dist : [];
    
    return $stats;
}

/**
 * Get file statistics
 * @return array
 */
function getFileStats(): array {
    $stats = [];
    
    // Total files
    $total = dbQueryOne("SELECT COUNT(*) as count FROM files WHERE is_active = 1");
    $stats['total_files'] = (int)($total['count'] ?? 0);
    
    // Files by type
    $by_type = dbQuery(
        "SELECT file_type, COUNT(*) as count 
         FROM files 
         WHERE is_active = 1 
         GROUP BY file_type"
    );
    $stats['files_by_type'] = $by_type !== false ? $by_type : [];
    
    // Most popular files
    $popular = dbQuery(
        "SELECT f.id, f.file_type, f.download_count, f.created_at
         FROM files f
         WHERE f.is_active = 1
         ORDER BY f.download_count DESC
         LIMIT 10"
    );
    $stats['most_popular_files'] = $popular !== false ? $popular : [];
    
    // Files uploaded today
    $today = dbQueryOne(
        "SELECT COUNT(*) as count FROM files WHERE DATE(created_at) = CURDATE() AND is_active = 1"
    );
    $stats['files_uploaded_today'] = (int)($today['count'] ?? 0);
    
    return $stats;
}

/**
 * Get advanced analytics report
 * @return array
 */
function getAdvancedAnalytics(): array {
    $report = [];
    
    // Daily statistics for last 7 days
    $daily = dbQuery(
        "SELECT DATE(created_at) as date, COUNT(*) as count
         FROM downloads
         WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
         GROUP BY DATE(created_at)
         ORDER BY date DESC"
    );
    $report['daily_downloads'] = $daily !== false ? $daily : [];
    
    // Weekly statistics for last 4 weeks
    $weekly = dbQuery(
        "SELECT WEEK(created_at) as week, COUNT(*) as count
         FROM downloads
         WHERE created_at >= DATE_SUB(NOW(), INTERVAL 4 WEEK)
         GROUP BY WEEK(created_at)
         ORDER BY week DESC"
    );
    $report['weekly_downloads'] = $weekly !== false ? $weekly : [];
    
    // Monthly statistics for last 6 months
    $monthly = dbQuery(
        "SELECT DATE_FORMAT(created_at, '%Y-%m') as month, COUNT(*) as count
         FROM downloads
         WHERE created_at >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
         GROUP BY DATE_FORMAT(created_at, '%Y-%m')
         ORDER BY month DESC"
    );
    $report['monthly_downloads'] = $monthly !== false ? $monthly : [];
    
    // Referral statistics
    $referral_stats = dbQuery(
        "SELECT COUNT(*) as total_referrals, SUM(points_given_referrer) as total_points
         FROM referrals"
    );
    $report['referral_stats'] = $referral_stats !== false && !empty($referral_stats) ? $referral_stats[0] : [];
    
    return $report;
}

/**
 * Display analytics to admin
 * @param int $chat_id
 * @return void
 */
function displayAnalytics(int $chat_id): void {
    $download_stats = getDownloadStats();
    $user_stats = getUserStats();
    $file_stats = getFileStats();
    
    $message = "📊 آمار ربات\n\n";
    
    $message .= "📥 آمار دانلودها:\n";
    $message .= "   • کل دانلودها: {$download_stats['total_downloads']}\n";
    $message .= "   • امروز: {$download_stats['downloads_today']}\n";
    $message .= "   • این هفته: {$download_stats['downloads_this_week']}\n";
    $message .= "   • این ماه: {$download_stats['downloads_this_month']}\n\n";
    
    $message .= "👥 آمار کاربران:\n";
    $message .= "   • کل کاربران: {$user_stats['total_users']}\n";
    $message .= "   • کاربران فعال: {$user_stats['active_users']}\n";
    $message .= "   • کاربران جدید امروز: {$user_stats['new_users_today']}\n";
    $message .= "   • کاربران جدید این هفته: {$user_stats['new_users_this_week']}\n";
    $message .= "   • کاربران جدید این ماه: {$user_stats['new_users_this_month']}\n\n";
    
    $message .= "📁 آمار فایل‌ها:\n";
    $message .= "   • کل فایل‌ها: {$file_stats['total_files']}\n";
    $message .= "   • فایل‌های آپلود شده امروز: {$file_stats['files_uploaded_today']}\n";
    
    if (!empty($file_stats['files_by_type'])) {
        $message .= "   • بر اساس نوع:\n";
        foreach ($file_stats['files_by_type'] as $type) {
            $message .= "     - {$type['file_type']}: {$type['count']}\n";
        }
    }
    
    sendMessage($chat_id, $message);
}

