<?php
declare(strict_types=1);

/**
 * Auto-Delete System Handler
 * Handles automatic message deletion using Telegram deleteMessage API
 * 
 * IMPORTANT: This only deletes messages from user's chat, NOT from database!
 * Files remain in database and can be downloaded again by other users.
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/validators.php';
require_once __DIR__ . '/../inc/logger.php';
require_once __DIR__ . '/upload-handler.php';

/**
 * Set auto-delete timer for file (in seconds)
 * @param int $file_id
 * @param int $seconds Seconds until deletion
 * @return bool
 */
function setFileAutoDelete(int $file_id, int $seconds): bool {
    if ($seconds <= 0) {
        return false;
    }
    
    $delete_at = date('Y-m-d H:i:s', time() + $seconds);
    
    $result = dbExecute(
        "UPDATE files SET auto_delete_at = ? WHERE id = ?",
        [$delete_at, $file_id]
    );
    
    return $result !== false;
}

/**
 * Set auto-delete timer for file (in hours)
 * @param int $file_id
 * @param int $hours Hours until deletion
 * @return bool
 */
function setFileAutoDeleteHours(int $file_id, int $hours): bool {
    return setFileAutoDelete($file_id, $hours * 3600);
}

/**
 * Remove auto-delete timer for file
 * @param int $file_id
 * @return bool
 */
function removeFileAutoDelete(int $file_id): bool {
    $result = dbExecute(
        "UPDATE files SET auto_delete_at = NULL WHERE id = ?",
        [$file_id]
    );
    
    return $result !== false;
}

// Note: Auto-delete is now handled directly in handleFileDownload using sleep()
// No need for recordSentMessageForAutoDelete function anymore

// Note: Auto-delete is now handled directly in handleFileDownload using sleep()
// No need for processAutoDeleteMessages function anymore

/**
 * Get auto-delete timer for file (in seconds)
 * @param int $file_id
 * @return int|false Seconds until deletion, or false if not set
 */
function getFileAutoDeleteTimer(int $file_id): int|false {
    $file = getFileById($file_id);
    
    if ($file === false || empty($file['auto_delete_at'])) {
        return false;
    }
    
    $delete_at = strtotime($file['auto_delete_at']);
    $now = time();
    
    if ($delete_at <= $now) {
        return 0; // Should be deleted now
    }
    
    return $delete_at - $now;
}

/**
 * Handle set auto-delete command
 * @param array $message Telegram message object
 * @param int $seconds Seconds until deletion
 * @return void
 */
function handleSetAutoDelete(array $message, int $seconds): void {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    
    // Check if user is admin
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ فقط ادمین‌ها می‌توانند timer حذف خودکار تنظیم کنند.");
        return;
    }
    
    // Check if message is reply to a file
    if (!isset($message['reply_to_message'])) {
        sendMessage($chat_id, "❌ لطفاً به پیام فایلی که می‌خواهید timer برای آن تنظیم کنید reply کنید.");
        return;
    }
    
    $replied_message = $message['reply_to_message'];
    $file_info = extractFileInfo($replied_message);
    
    if ($file_info === false) {
        sendMessage($chat_id, "❌ خطا در دریافت اطلاعات فایل.");
        return;
    }
    
    // Find file in database
    $file = dbQueryOne(
        "SELECT id FROM files WHERE file_id = ? AND is_active = 1 ORDER BY created_at DESC LIMIT 1",
        [$file_info['file_id']]
    );
    
    if ($file === false) {
        sendMessage($chat_id, "❌ فایل در دیتابیس یافت نشد.");
        return;
    }
    
    $file_id = (int)$file['id'];
    
    // Set auto-delete
    if (setFileAutoDelete($file_id, $seconds)) {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;
        
        $time_str = '';
        if ($hours > 0) {
            $time_str .= "{$hours} ساعت ";
        }
        if ($minutes > 0) {
            $time_str .= "{$minutes} دقیقه ";
        }
        if ($secs > 0 || $time_str === '') {
            $time_str .= "{$secs} ثانیه";
        }
        
        sendMessage($chat_id, "✅ Timer حذف خودکار تنظیم شد!\n⏰ پیام‌های ارسال شده به کاربران با این تنظیمات بعد از {$time_str} از چت کاربران حذف خواهند شد.\n📝 توجه: فایل در دیتابیس باقی می‌ماند و قابل دانلود مجدد است.");
    } else {
        sendMessage($chat_id, "❌ خطا در تنظیم timer حذف خودکار.");
    }
}
