<?php
declare(strict_types=1);

/**
 * Media Management Handler
 * Handles media management operations
 */

require_once __DIR__ . '/../inc/functions.php';
require_once __DIR__ . '/../inc/auth.php';
require_once __DIR__ . '/../inc/database.php';
require_once __DIR__ . '/../inc/logger.php';
require_once __DIR__ . '/upload-handler.php';

/**
 * Toggle thumbnail for file
 * @param int $file_id
 * @param bool $enabled
 * @return bool
 */
function toggleFileThumbnail(int $file_id, bool $enabled): bool {
    $result = dbExecute(
        "UPDATE files SET has_thumbnail = ? WHERE id = ?",
        [$enabled ? 1 : 0, $file_id]
    );
    
    return $result !== false;
}

/**
 * Toggle link for file
 * @param int $file_id
 * @param bool $enabled
 * @return bool
 */
function toggleFileLink(int $file_id, bool $enabled): bool {
    $result = dbExecute(
        "UPDATE files SET has_link = ? WHERE id = ?",
        [$enabled ? 1 : 0, $file_id]
    );
    
    return $result !== false;
}

/**
 * Get media information
 * @param int $file_id
 * @return array|false
 */
function getMediaInfo(int $file_id): array|false {
    $file = getFileById($file_id);
    
    if ($file === false) {
        return false;
    }
    
    // Get forced join channels
    $forced_join = [];
    if (!empty($file['forced_join_channels'])) {
        $forced_join = json_decode($file['forced_join_channels'], true) ?: [];
    }
    
    // Get channel info
    $channels_info = [];
    if (!empty($forced_join)) {
        $placeholders = implode(',', array_fill(0, count($forced_join), '?'));
        $channels = dbQuery(
            "SELECT * FROM channels WHERE channel_id IN ({$placeholders})",
            $forced_join
        );
        
        if ($channels !== false) {
            $channels_info = $channels;
        }
    }
    
    return [
        'file' => $file,
        'forced_join_channels' => $channels_info,
        'download_count' => (int)$file['download_count']
    ];
}

/**
 * Display media information
 * @param int $chat_id
 * @param int $file_id
 * @return void
 */
function displayMediaInfo(int $chat_id, int $file_id): void {
    $info = getMediaInfo($file_id);
    
    if ($info === false) {
        sendMessage($chat_id, "❌ فایل یافت نشد.");
        return;
    }
    
    $file = $info['file'];
    
    $message = "📁 اطلاعات فایل:\n\n";
    $message .= "🆔 شناسه: {$file['id']}\n";
    $message .= "📄 نوع: {$file['file_type']}\n";
    $message .= "💾 حجم: " . formatFileSize($file['file_size'] ?? 0) . "\n";
    $message .= "📥 دانلودها: {$file['download_count']}\n";
    $message .= "📅 تاریخ آپلود: {$file['created_at']}\n";
    $message .= "🖼 Thumbnail: " . ($file['has_thumbnail'] ? '✅' : '❌') . "\n";
    $message .= "🔗 Link: " . ($file['has_link'] ? '✅' : '❌') . "\n";
    
    if (!empty($file['auto_delete_at'])) {
        $message .= "⏰ حذف خودکار: {$file['auto_delete_at']}\n";
    }
    
    if (!empty($info['forced_join_channels'])) {
        $message .= "\n📢 کانال‌های Forced Join:\n";
        foreach ($info['forced_join_channels'] as $channel) {
            $message .= "   • {$channel['channel_title']}\n";
        }
    }
    
    if (!empty($file['caption'])) {
        $message .= "\n📝 کپشن: {$file['caption']}";
    }
    
    sendMessage($chat_id, $message);
}

/**
 * Format file size
 * @param int $bytes
 * @return string
 */
function formatFileSize(int $bytes): string {
    if ($bytes == 0) {
        return '0 B';
    }
    
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $unit_index = 0;
    $size = $bytes;
    
    while ($size >= 1024 && $unit_index < count($units) - 1) {
        $size /= 1024;
        $unit_index++;
    }
    
    return round($size, 2) . ' ' . $units[$unit_index];
}

/**
 * List all media files
 * @param int $chat_id
 * @param int $limit
 * @return void
 */
function listMediaFiles(int $chat_id, int $limit = 20): void {
    $files = getAllFiles($limit);
    
    if ($files === false || empty($files)) {
        sendMessage($chat_id, "📋 لیست فایل‌ها خالی است.");
        return;
    }
    
    $message = "📋 لیست فایل‌ها:\n\n";
    
    foreach ($files as $index => $file) {
        $message .= ($index + 1) . ". {$file['file_type']} (ID: {$file['id']})\n";
        $message .= "   📥 دانلودها: {$file['download_count']}\n";
        $message .= "   📅 {$file['created_at']}\n\n";
    }
    
    sendMessage($chat_id, $message);
}

