<?php
declare(strict_types=1);

/**
 * Validation Functions
 * Provides data validation and sanitization functions
 */

/**
 * Sanitize string input
 * @param string $input
 * @return string
 */
function sanitizeString(string $input): string {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate user ID
 * @param mixed $user_id
 * @return int|false
 */
function validateUserId(mixed $user_id): int|false {
    if (!is_numeric($user_id)) {
        return false;
    }
    
    $id = (int)$user_id;
    if ($id <= 0) {
        return false;
    }
    
    return $id;
}

/**
 * Validate channel ID or username
 * @param mixed $channel
 * @return string|false
 */
function validateChannel(mixed $channel): string|false {
    if (!is_string($channel)) {
        return false;
    }
    
    $channel = trim($channel);
    
    // Check if it's a channel ID (numeric)
    if (is_numeric($channel)) {
        return $channel;
    }
    
    // Check if it's a username (starts with @)
    if (preg_match('/^@[a-zA-Z0-9_]{5,32}$/', $channel)) {
        return $channel;
    }
    
    return false;
}

/**
 * Validate file type
 * @param string $file_type
 * @return bool
 */
function validateFileType(string $file_type): bool {
    $allowed_types = ['photo', 'video', 'document', 'audio', 'voice', 'video_note', 'sticker', 'animation'];
    return in_array($file_type, $allowed_types, true);
}

/**
 * Validate points amount
 * @param mixed $points
 * @return int|false
 */
function validatePoints(mixed $points): int|false {
    if (!is_numeric($points)) {
        return false;
    }
    
    $points = (int)$points;
    if ($points < 0) {
        return false;
    }
    
    return $points;
}

/**
 * Validate stars amount
 * @param mixed $stars
 * @return int|false
 */
function validateStars(mixed $stars): int|false {
    if (!is_numeric($stars)) {
        return false;
    }
    
    $stars = (int)$stars;
    if ($stars <= 0) {
        return false;
    }
    
    return $stars;
}

/**
 * Validate referral code
 * @param string $code
 * @return bool
 */
function validateReferralCode(string $code): bool {
    return preg_match('/^[a-zA-Z0-9]{8,32}$/', $code) === 1;
}

/**
 * Generate referral code
 * @param int $user_id
 * @return string
 */
function generateReferralCode(int $user_id): string {
    return base64_encode((string)$user_id) . bin2hex(random_bytes(4));
}

/**
 * Generate unique file code
 * @return string
 */
function generateFileCode(): string {
    require_once __DIR__ . '/database.php';
    
    // Generate a unique code similar to: b7qxlyohbabmjb6
    $characters = 'abcdefghijklmnopqrstuvwxyz0123456789';
    $code = '';
    $length = 16;
    
    // Try up to 10 times to generate unique code
    for ($attempt = 0; $attempt < 10; $attempt++) {
        $code = '';
        for ($i = 0; $i < $length; $i++) {
            $code .= $characters[random_int(0, strlen($characters) - 1)];
        }
        
        // Check if code already exists
        $existing = dbQueryOne("SELECT id FROM files WHERE file_code = ?", [$code]);
        if ($existing === false) {
            // Code is unique
            return $code;
        }
    }
    
    // If all attempts failed, add timestamp to make it unique
    return $code . substr(md5(time() . random_bytes(8)), 0, 8);
}

/**
 * Validate JSON string
 * @param string $json
 * @return array|false
 */
function validateJson(string $json): array|false {
    $decoded = json_decode($json, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return false;
    }
    return is_array($decoded) ? $decoded : false;
}

/**
 * Validate timestamp
 * @param mixed $timestamp
 * @return int|false
 */
function validateTimestamp(mixed $timestamp): int|false {
    if (!is_numeric($timestamp)) {
        return false;
    }
    
    $ts = (int)$timestamp;
    if ($ts <= 0) {
        return false;
    }
    
    return $ts;
}

