<?php
/**
 * Migration Script: تولید UUID برای مدیاهای موجود
 * 
 * این اسکریپت باید به صورت دستی اجرا شود یا در migration SQL قرار گیرد
 * اما چون نیاز به استفاده از PHP برای تولید UUID داریم، بهتر است به صورت جداگانه اجرا شود
 * 
 * استفاده:
 * php database/migrates/migration_generate_uuids_for_existing_media.php
 */

declare(strict_types=1);

require __DIR__ . '/../../vendor/autoload.php';

use Dotenv\Dotenv;

// Load environment variables
$dotenv = Dotenv::createImmutable(__DIR__ . '/../..');
$dotenv->load();

$dbHost = $_ENV['DB_HOST'] ?? 'localhost';
$dbPort = isset($_ENV['DB_PORT']) ? (int) $_ENV['DB_PORT'] : 3306;
$dbName = $_ENV['DB_DATABASE'] ?? $_ENV['DB_NAME'] ?? '';
$dbUser = $_ENV['DB_USERNAME'] ?? $_ENV['DB_USER'] ?? '';
$dbPass = $_ENV['DB_PASSWORD'] ?? '';

if (empty($dbName) || empty($dbUser)) {
    echo "❌ تنظیمات دیتابیس در فایل .env وارد نشده است.\n";
    exit(1);
}

try {
    $dsn = "mysql:host={$dbHost};port={$dbPort};dbname={$dbName};charset=utf8mb4";
    $pdo = new PDO($dsn, $dbUser, $dbPass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);

    // بررسی وجود فیلد uuid
    $stmt = $pdo->query("SHOW COLUMNS FROM media LIKE 'uuid'");
    if ($stmt->rowCount() === 0) {
        echo "❌ فیلد uuid در جدول media وجود ندارد. ابتدا migration_add_uuid_to_media.sql را اجرا کنید.\n";
        exit(1);
    }

    // دریافت مدیاهای بدون UUID
    $stmt = $pdo->query("SELECT id FROM media WHERE uuid IS NULL");
    $medias = $stmt->fetchAll();

    if (empty($medias)) {
        echo "✅ همه مدیاها UUID دارند.\n";
        exit(0);
    }

    echo "📊 تعداد مدیاهای بدون UUID: " . count($medias) . "\n";
    echo "🔄 در حال تولید UUID...\n\n";

    $pdo->beginTransaction();

    try {
        $updateStmt = $pdo->prepare("UPDATE media SET uuid = :uuid WHERE id = :id");
        $count = 0;

        foreach ($medias as $media) {
            // تولید UUID v4
            $data = random_bytes(16);
            $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // version 4
            $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // variant bits
            
            $uuid = sprintf(
                '%08s-%04s-%04s-%04s-%012s',
                bin2hex(substr($data, 0, 4)),
                bin2hex(substr($data, 4, 2)),
                bin2hex(substr($data, 6, 2)),
                bin2hex(substr($data, 8, 2)),
                bin2hex(substr($data, 10, 6))
            );

            $updateStmt->execute([
                'uuid' => $uuid,
                'id' => $media['id'],
            ]);

            $count++;
            echo "✅ Media ID {$media['id']}: {$uuid}\n";
        }

        $pdo->commit();
        echo "\n✅ {$count} UUID با موفقیت تولید و ذخیره شدند.\n";
    } catch (\Throwable $e) {
        $pdo->rollBack();
        throw $e;
    }
} catch (PDOException $e) {
    echo "❌ خطا در اتصال به دیتابیس: " . $e->getMessage() . "\n";
    exit(1);
} catch (\Throwable $e) {
    echo "❌ خطا: " . $e->getMessage() . "\n";
    exit(1);
}

