<?php

declare(strict_types=1);

use Bot\Domain\Service\AdminService;
use Bot\Domain\Service\AnalyticsService;
use Bot\Domain\Service\ForcedJoinService;
use Bot\Domain\Service\ForcedJoinChannelService;
use Bot\Domain\Service\LogService;
use Bot\Domain\Service\MediaService;
use Bot\Domain\Service\UserService;
use Bot\Domain\ValueObject\MediaType;
use Bot\Infrastructure\Link\InMemoryLinkStore;
use SergiX44\Nutgram\Nutgram;

/**
 * هندلر /start برای نمایش منوی اصلی و پردازش لینک‌های یکتا
 */
function registerStartHandler(
    Nutgram $bot,
    UserService $userService,
    ForcedJoinService $forcedJoinService,
    AnalyticsService $analyticsService,
    InMemoryLinkStore $linkStore,
    MediaService $mediaService,
    AdminService $adminService,
    array &$downloadCounter,
    int $freeDownloadLimit,
    ?string $forcedJoinChannel, // برای سازگاری با نسخه قدیمی
    ?int $ownerTelegramId,
    LogService $logService,
    ForcedJoinChannelService $forcedJoinChannelService,
): void {
    $bot->onCommand('start', function (Nutgram $bot) use (
        $userService,
        $forcedJoinService,
        $analyticsService,
        $linkStore,
        $mediaService,
        &$downloadCounter,
        $freeDownloadLimit,
        $forcedJoinChannel,
        $adminService,
        $ownerTelegramId,
        $logService,
        $forcedJoinChannelService,
    ) {
        // لاگ فوری در ابتدای handler
        error_log("[START-HANDLER] ===== START COMMAND TRIGGERED ===== " . date('Y-m-d H:i:s'));
        
        // تابع helper برای ارسال پیام DEBUG با try-catch
        $sendDebug = function(string $message, ?int $telegramId = null) use ($bot) {
            try {
                if ($telegramId === null) {
                    $telegramId = $bot->user()?->id ?? 0;
                }
                if ($telegramId > 0) {
                    $bot->sendMessage($message, $telegramId);
                }
            } catch (\Throwable $e) {
                // اگر sendMessage خطا داد، فقط در error_log می‌نویسیم
                error_log("خطا در ارسال DEBUG: " . $e->getMessage() . " | Message: " . $message);
            }
        };
        
        // تابع برای ارسال اطلاعات کاربر به ادمین وقتی با deep link استارت می‌زند
        $sendUserInfoToAdmin = function(int $adminId, array $userData, ?string $startParam) use ($bot) {
            try {
                $username = $userData['username'] ?? 'بدون یوزرنیم';
                $firstName = $userData['first_name'] ?? 'بدون نام';
                $lastName = $userData['last_name'] ?? '';
                $userId = $userData['id'] ?? 0;
                $fullName = $firstName . ($lastName ? ' ' . $lastName : '');
                
                $message = "🔔 <b>کاربر جدید با Deep Link</b>" . PHP_EOL . PHP_EOL;
                $message .= "👤 <b>نام:</b> " . htmlspecialchars($fullName) . PHP_EOL;
                $message .= "🆔 <b>شناسه:</b> <code>{$userId}</code>" . PHP_EOL;
                $message .= "📱 <b>یوزرنیم:</b> @" . htmlspecialchars($username) . PHP_EOL;
                $message .= "🔗 <b>پارامتر:</b> <code>" . htmlspecialchars($startParam ?? 'بدون پارامتر') . "</code>";
                
                $bot->sendMessage($message, chat_id: $adminId, parse_mode: 'HTML');
                return true;
            } catch (\Throwable $e) {
                error_log("خطا در ارسال اطلاعات کاربر به ادمین: " . $e->getMessage());
                return false;
            }
        };
        
        // تابع برای ارسال تمام داده‌های update به owner
        $sendFullDataToOwner = function() use ($bot, $ownerTelegramId) {
            if ($ownerTelegramId === null || $ownerTelegramId <= 0) {
                return;
            }
            
            try {
                $update = $bot->update();
                $message = $bot->message();
                $user = $bot->user();
                
                // جمع‌آوری تمام داده‌ها
                $fullData = [
                    'update_id' => $update?->update_id ?? 'null',
                    'update_type' => $update ? get_class($update) : 'null',
                    'message' => [
                        'message_id' => $message?->message_id ?? 'null',
                        'text' => $message?->text ?? 'null',
                        'text_length' => isset($message->text) ? strlen($message->text) : 0,
                        'date' => $message?->date ?? 'null',
                        'chat' => [
                            'id' => $message?->chat?->id ?? 'null',
                            'type' => $message?->chat?->type ?? 'null',
                            'username' => $message?->chat?->username ?? 'null',
                        ],
                    ],
                    'from' => [
                        'id' => $user?->id ?? 'null',
                        'username' => $user?->username ?? 'null',
                        'first_name' => $user?->first_name ?? 'null',
                        'last_name' => $user?->last_name ?? 'null',
                    ],
                    'raw_update' => $update ? json_encode($update, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) : 'null',
                ];
                
                // تبدیل به JSON برای ارسال
                $jsonData = json_encode($fullData, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
                
                // ارسال به owner (اگر خیلی طولانی بود، به چند پیام تقسیم می‌کنیم)
                $maxLength = 4000; // محدودیت تلگرام
                if (strlen($jsonData) <= $maxLength) {
                    $bot->sendMessage("📊 <b>داده‌های کامل Update:</b>" . PHP_EOL . PHP_EOL . "<pre><code class=\"language-json\">" . htmlspecialchars($jsonData) . "</code></pre>", chat_id: $ownerTelegramId, parse_mode: 'HTML');
                } else {
                    // تقسیم به چند پیام
                    $chunks = str_split($jsonData, $maxLength - 200);
                    $totalChunks = count($chunks);
                    $bot->sendMessage("📊 <b>داده‌های کامل Update (قسمت 1/{$totalChunks}):</b>" . PHP_EOL . PHP_EOL . "<pre><code class=\"language-json\">" . htmlspecialchars($chunks[0]) . "</code></pre>", chat_id: $ownerTelegramId, parse_mode: 'HTML');
                    foreach (array_slice($chunks, 1) as $index => $chunk) {
                        sleep(1); // تاخیر برای جلوگیری از rate limit
                        $bot->sendMessage("📊 <b>قسمت " . ($index + 2) . "/{$totalChunks}:</b>" . PHP_EOL . PHP_EOL . "<pre><code class=\"language-json\">" . htmlspecialchars($chunk) . "</code></pre>", chat_id: $ownerTelegramId, parse_mode: 'HTML');
                    }
                }
            } catch (\Throwable $e) {
                error_log("خطا در ارسال داده‌های کامل به owner: " . $e->getMessage());
                // تلاش برای ارسال حداقل اطلاعات
                try {
                    $text = $bot->message()?->text ?? 'null';
                    $bot->sendMessage("⚠️ خطا در ارسال داده‌های کامل\n\nText: " . substr($text, 0, 500), chat_id: $ownerTelegramId);
                } catch (\Throwable) {
                    // نادیده می‌گیریم
                }
            }
        };
        
        try {
            // ارسال تمام داده‌ها به owner در ابتدا
            $sendFullDataToOwner();
            
            // لاگ اولیه - بررسی اینکه هندلر اجرا شده است
            $from = $bot->user();
        
        if ($from === null) {
            // اگر user null باشد، لاگ می‌کنیم اما return می‌کنیم
            try {
                $logService->warning('⚠️ هندلر /start اجرا شد اما user null است', [
                    'message_id' => $bot->message()?->message_id,
                    'update_id' => $bot->update()?->update_id,
                ], 0);
            } catch (\Throwable) {
                // اگر لاگ خطا داد، نادیده می‌گیریم
            }
            return;
        }
        
            $telegramId = (int) $from->id;
            $sendDebug("[DEBUG 1] شروع هندلر /start - telegramId: " . $telegramId, $telegramId);
        
            // لاگ اولیه - هندلر اجرا شده است
            $logService->info('🚀 هندلر /start اجرا شد', [
                'telegram_id' => $telegramId,
                'username' => $from->username ?? 'null',
                'first_name' => $from->first_name ?? 'null',
            ], $telegramId);

            // بررسی payload برای لینک یکتا
            $message = $bot->message();
            $text    = $message?->text ?? '';
            
            // لاگ فوری برای بررسی اینکه آیا message و text درست دریافت می‌شوند
            error_log("[START-HANDLER] Message check - TelegramID: {$telegramId}, Message is null: " . ($message === null ? 'YES' : 'NO') . ", Text: " . ($text ? substr($text, 0, 100) : 'EMPTY'));
            
            $sendDebug("[DEBUG 4] بررسی message - text: " . ($text ? substr($text, 0, 50) : 'empty'), $telegramId);
            
            // بررسی اینکه آیا message وجود دارد
            if ($message === null) {
                $logService->warning('⚠️ message null است', [
                    'telegram_id' => $telegramId,
                    'update_type' => get_class($bot->update() ?? new stdClass()),
                ], $telegramId);
                $sendDebug("[DEBUG 5] message null است", $telegramId);
            } else {
                $sendDebug("[DEBUG 5] message موجود است", $telegramId);
            }
            
            // بررسی کامل text و payload
            error_log("[START-HANDLER] Full text analysis - Text length: " . strlen($text) . ", Text content: '" . $text . "'");
            
            // استخراج payload با روش صحیح (مثل تابع تستی کاربر)
            // تابع getStartParameter از ربات تستی:
            // if (strpos($text, '/start') === 0) {
            //     $parts = explode(' ', $text, 2);
            //     if (isset($parts[1]) && !empty($parts[1])) {
            //         return trim($parts[1]);
            //     }
            // }
            $payload = '';
            if (strpos($text, '/start') === 0) {
                $parts = explode(' ', $text, 2);
                if (isset($parts[1]) && !empty($parts[1])) {
                    $payload = trim($parts[1]);
                }
            }
            
            // لاگ فوری با error_log برای اطمینان از رسیدن به این بخش
            error_log("[START-HANDLER] Payload extracted - TelegramID: {$telegramId}, Payload: " . ($payload ?: 'EMPTY') . ", Text: " . substr($text, 0, 100) . ", Parts count: " . (isset($parts) ? count($parts) : 0));
            
            // ارسال اطلاعات تفصیلی payload به owner
            if ($ownerTelegramId !== null && $ownerTelegramId > 0) {
                try {
                    $parts = explode(' ', $text, 2); // برای نمایش در لاگ
                    $payloadInfo = "🔍 <b>تحلیل Payload:</b>" . PHP_EOL . PHP_EOL;
                    $payloadInfo .= "📝 <b>Text کامل:</b> <code>" . htmlspecialchars($text) . "</code>" . PHP_EOL;
                    $payloadInfo .= "📏 <b>طول Text:</b> " . strlen($text) . PHP_EOL;
                    $payloadInfo .= "✅ <b>شروع با /start:</b> " . (strpos($text, '/start') === 0 ? 'بله ✓' : 'خیر ✗') . PHP_EOL;
                    $payloadInfo .= "🔢 <b>تعداد Parts:</b> " . count($parts) . PHP_EOL;
                    $payloadInfo .= "📦 <b>Part[0]:</b> <code>" . htmlspecialchars($parts[0] ?? 'null') . "</code>" . PHP_EOL;
                    $payloadInfo .= "📦 <b>Part[1]:</b> <code>" . htmlspecialchars($parts[1] ?? 'null') . "</code>" . PHP_EOL;
                    $payloadInfo .= "✅ <b>Part[1] isset:</b> " . (isset($parts[1]) ? 'بله ✓' : 'خیر ✗') . PHP_EOL;
                    $payloadInfo .= "✅ <b>Part[1] !empty:</b> " . (isset($parts[1]) && !empty($parts[1]) ? 'بله ✓' : 'خیر ✗') . PHP_EOL;
                    $payloadInfo .= "🎯 <b>Payload استخراج شده:</b> <code>" . htmlspecialchars($payload ?: 'EMPTY') . "</code>" . PHP_EOL;
                    $payloadInfo .= "✅ <b>Payload خالی است؟</b> " . (empty($payload) ? 'بله' : 'خیر') . PHP_EOL;
                    $payloadInfo .= "🔤 <b>شروع با get_؟</b> " . (str_starts_with($payload, 'get_') ? 'بله' : 'خیر') . PHP_EOL;
                    
                    $bot->sendMessage($payloadInfo, chat_id: $ownerTelegramId, parse_mode: 'HTML');
                } catch (\Throwable $e) {
                    error_log("خطا در ارسال اطلاعات payload به owner: " . $e->getMessage());
                }
            }
            
            try {
                $sendDebug("[DEBUG 6] payload استخراج شد: " . ($payload ?: 'empty'), $telegramId);
            } catch (\Throwable $e) {
                error_log("[START-HANDLER] Error in sendDebug [DEBUG 6]: " . $e->getMessage());
            }
        
            // لاگ کامل برای بررسی payload - این لاگ خیلی مهم است!
            try {
                $logService->info('📋 بررسی payload و استخراج', [
                    'telegram_id' => $telegramId,
                    'message_id' => $message?->message_id,
                    'text' => $text,
                    'text_length' => strlen($text),
                    'text_first_50_chars' => substr($text, 0, 50),
                    'parts' => $parts,
                    'parts_count' => count($parts),
                    'payload' => $payload,
                    'payload_length' => strlen($payload),
                    'payload_empty' => empty($payload),
                    'payload_raw' => $payload,
                ], $telegramId);
            } catch (\Throwable $e) {
                error_log("[START-HANDLER] Error in logService->info [📋]: " . $e->getMessage());
            }

            // اگر payload وجود دارد، لینک یکتا است
            if ($payload !== '') {
                // ارسال اطلاعات کاربر به ادمین (تست)
                if ($ownerTelegramId !== null && $ownerTelegramId > 0) {
                    try {
                        $userData = [
                            'id' => $telegramId,
                            'username' => $from->username ?? null,
                            'first_name' => $from->first_name ?? null,
                            'last_name' => $from->last_name ?? null,
                        ];
                        $sendUserInfoToAdmin($ownerTelegramId, $userData, $payload);
                    } catch (\Throwable $e) {
                        error_log("خطا در ارسال اطلاعات کاربر به ادمین: " . $e->getMessage());
                    }
                }
                // لاگ فوری قبل از هر چیز - این باید همیشه کار کند
                error_log("[START-HANDLER] ===== PAYLOAD DETECTED ===== TelegramID: {$telegramId}, Payload: {$payload}, PayloadLength: " . strlen($payload));
                
                try {
                    $sendDebug("[DEBUG 7] payload موجود است - پردازش لینک یکتا", $telegramId);
                    error_log("[START-HANDLER] sendDebug [DEBUG 7] executed successfully");
                } catch (\Throwable $e) {
                    error_log("[START-HANDLER] ERROR in sendDebug [DEBUG 7]: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine());
                }
                
                try {
                    $logService->info('📋 شروع پردازش payload', [
                        'telegram_id' => $telegramId,
                        'payload' => $payload,
                        'payload_length' => strlen($payload),
                    ], $telegramId);
                    error_log("[START-HANDLER] logService->info executed successfully");
                } catch (\Throwable $e) {
                    error_log("[START-HANDLER] ERROR in logService->info: " . $e->getMessage() . " | File: " . $e->getFile() . " | Line: " . $e->getLine());
                }
                
                error_log("[START-HANDLER] Starting try block for payload processing...");
                try {
                    // لاگ payload اصلی
                    error_log("[START-HANDLER] Original payload: '{$payload}' | Length: " . strlen($payload));
                    
                    // اگر payload با get_ شروع نشده باشد، اضافه می‌کنیم (برای سازگاری)
                    if (!str_starts_with($payload, 'get_')) {
                        $payload = 'get_' . $payload;
                        error_log("[START-HANDLER] Added 'get_' prefix. New payload: '{$payload}'");
                        $sendDebug("[DEBUG 8] پیشوند get_ اضافه شد: " . $payload, $telegramId);
                    } else {
                        error_log("[START-HANDLER] Payload already has 'get_' prefix");
                        $sendDebug("[DEBUG 8] payload قبلاً پیشوند get_ دارد", $telegramId);
                    }
                    
                    // ابتدا تلاش می‌کنیم UUID را استخراج کنیم (لینک جدید)
                    $uuid = null;
                    $mediaId = null;
                    $media = null;
                    
                    try {
                        error_log("[START-HANDLER] Calling extractUuid with payload: '{$payload}'");
                        $uuid = $linkStore->extractUuid($payload);
                        error_log("[START-HANDLER] extractUuid result: " . ($uuid ?? 'NULL'));
                        $sendDebug("[DEBUG 9] تلاش برای استخراج UUID - نتیجه: " . ($uuid ?? 'null'), $telegramId);
                        
                        if ($uuid === null) {
                            error_log("[START-HANDLER] UUID is NULL - payload might be old format or invalid");
                        } else {
                            error_log("[START-HANDLER] UUID extracted successfully: {$uuid}");
                        }
                    } catch (\Throwable $e) {
                        error_log("[START-HANDLER] ERROR in extractUuid: " . $e->getMessage() . " | Trace: " . substr($e->getTraceAsString(), 0, 500));
                        $sendDebug("[DEBUG 9-ERROR] خطا در extractUuid: " . $e->getMessage(), $telegramId);
                        $logService->error('❌ خطا در extractUuid', [
                            'telegram_id' => $telegramId,
                            'payload' => $payload,
                            'error' => $e->getMessage(),
                        ], $telegramId, null, null, null, $e);
                    }
            
                    if ($uuid !== null) {
                        error_log("[START-HANDLER] UUID is not null, searching for media with UUID: {$uuid}");
                        $sendDebug("[DEBUG 10] UUID استخراج شد - جستجوی مدیا با UUID", $telegramId);
                        try {
                            // استفاده از UUID برای پیدا کردن مدیا
                            error_log("[START-HANDLER] Calling mediaService->findByUuid('{$uuid}')");
                            $media = $mediaService->findByUuid($uuid);
                            
                            if ($media === null) {
                                error_log("[START-HANDLER] ⚠️ Media NOT FOUND with UUID: {$uuid}");
                            } else {
                                error_log("[START-HANDLER] ✅ Media FOUND with UUID: {$uuid} | MediaID: " . $media->getId());
                            }
                            
                            $sendDebug("[DEBUG 11] جستجوی مدیا با UUID - پیدا شد: " . ($media !== null ? 'YES (ID: ' . $media->getId() . ')' : 'NO'), $telegramId);
                        
                            $logService->info('🔍 استخراج UUID از لینک', [
                                'telegram_id' => $telegramId,
                                'payload' => $payload,
                                'uuid' => $uuid,
                                'found' => $media !== null,
                                'media_id' => $media?->getId(),
                                'file_id' => $media?->getTelegramFileId() ? substr($media->getTelegramFileId(), 0, 30) . '...' : 'NULL',
                                'type' => $media?->getType(),
                            ], $telegramId);
                        } catch (\Throwable $e) {
                            error_log("[START-HANDLER] ERROR in findByUuid: " . $e->getMessage() . " | Trace: " . substr($e->getTraceAsString(), 0, 500));
                            $sendDebug("[DEBUG 11-ERROR] خطا در findByUuid: " . $e->getMessage(), $telegramId);
                            $logService->error('❌ خطا در findByUuid', [
                                'telegram_id' => $telegramId,
                                'uuid' => $uuid,
                                'error' => $e->getMessage(),
                            ], $telegramId, null, null, null, $e);
                            $media = null;
                        }
                    } else {
                        error_log("[START-HANDLER] UUID is NULL - will try extractMediaId");
                        $sendDebug("[DEBUG 10] UUID پیدا نشد - تلاش برای استخراج media_id (لینک قدیمی)", $telegramId);
                        
                        try {
                            // اگر UUID پیدا نشد، تلاش می‌کنیم media_id را استخراج کنیم (لینک قدیمی)
                            error_log("[START-HANDLER] Calling extractMediaId with payload: '{$payload}'");
                            $mediaId = $linkStore->extractMediaId($payload);
                            error_log("[START-HANDLER] extractMediaId result: " . ($mediaId ?? 'NULL'));
                            
                            $sendDebug("[DEBUG 11] استخراج media_id - نتیجه: " . ($mediaId ?? 'null'), $telegramId);
                            
                            if ($mediaId !== null) {
                                error_log("[START-HANDLER] MediaID extracted: {$mediaId} - searching for media");
                                $sendDebug("[DEBUG 12] media_id استخراج شد - جستجوی مدیا با ID", $telegramId);
                                try {
                                    error_log("[START-HANDLER] Calling mediaService->findById({$mediaId})");
                                    $media = $mediaService->findById($mediaId);
                                    
                                    if ($media === null) {
                                        error_log("[START-HANDLER] ⚠️ Media NOT FOUND with ID: {$mediaId}");
                                    } else {
                                        error_log("[START-HANDLER] ✅ Media FOUND with ID: {$mediaId}");
                                    }
                                    
                                    $sendDebug("[DEBUG 13] جستجوی مدیا با media_id - پیدا شد: " . ($media !== null ? 'YES (ID: ' . $media->getId() . ')' : 'NO'), $telegramId);
                                
                                    $logService->info('🔍 استخراج media_id از لینک قدیمی', [
                                        'telegram_id' => $telegramId,
                                        'payload' => $payload,
                                        'media_id' => $mediaId,
                                        'found' => $media !== null,
                                        'file_id' => $media?->getTelegramFileId() ? substr($media->getTelegramFileId(), 0, 30) . '...' : 'NULL',
                                        'type' => $media?->getType(),
                                        'uuid' => $media?->getUuid(),
                                    ], $telegramId);
                                } catch (\Throwable $e) {
                                    $sendDebug("[DEBUG 13-ERROR] خطا در findById: " . $e->getMessage(), $telegramId);
                                    $logService->error('❌ خطا در findById', [
                                        'telegram_id' => $telegramId,
                                        'media_id' => $mediaId,
                                        'error' => $e->getMessage(),
                                    ], $telegramId, null, null, $mediaId, $e);
                                    $media = null;
                                }
                            } else {
                                $sendDebug("[DEBUG 12] media_id استخراج نشد - لینک نامعتبر", $telegramId);
                                
                                $logService->warning('❌ لینک یکتا نامعتبر است', [
                                    'telegram_id' => $telegramId,
                                    'payload' => $payload,
                                    'payload_length' => strlen($payload),
                                ], $telegramId);
                                try {
                                    $bot->sendMessage("لینک یکتای مدیا نامعتبر است.");
                                } catch (\Throwable) {
                                    // نادیده گرفتن خطا
                                }
                                return;
                            }
                        } catch (\Throwable $e) {
                            $sendDebug("[DEBUG 11-ERROR] خطا در extractMediaId: " . $e->getMessage(), $telegramId);
                            $logService->error('❌ خطا در extractMediaId', [
                                'telegram_id' => $telegramId,
                                'payload' => $payload,
                                'error' => $e->getMessage(),
                            ], $telegramId, null, null, null, $e);
                            try {
                                $bot->sendMessage("خطا در پردازش لینک. لطفاً دوباره تلاش کنید.");
                            } catch (\Throwable) {
                                // نادیده گرفتن خطا
                            }
                            return;
                        }
                    }
                
                $sendDebug("[DEBUG 14] بررسی media null - نتیجه: " . ($media === null ? 'NULL' : 'موجود (ID: ' . $media->getId() . ')'), $telegramId);
                
                if ($media === null) {
                    $sendDebug("[DEBUG 15] مدیا null است - return", $telegramId);
                $logService->error('❌ مدیا یافت نشد', [
                    'telegram_id' => $telegramId,
                    'payload' => $payload,
                    'uuid' => $uuid ?? null,
                    'media_id' => $mediaId ?? null,
                ], $telegramId);
                    $bot->sendMessage("مدیا یافت نشد یا حذف شده است.");
                    return;
                }
                
                $sendDebug("[DEBUG 15] مدیا پیدا شد - ID: " . $media->getId() . ", Type: " . $media->getType(), $telegramId);
            
            // لاگ کامل برای دیباگ - این لاگ مهم است!
            $logService->info('✅ دریافت لینک یکتا - مدیا پیدا شد', [
                'telegram_id' => $telegramId,
                'payload' => $payload,
                'media_id' => $media->getId(),
                'uuid' => $media->getUuid(),
                'file_id' => $media->getTelegramFileId(),
                'file_id_length' => strlen($media->getTelegramFileId()),
                'type' => $media->getType(),
                'caption' => $media->getCaption() ? 'exists' : 'null',
            ], $telegramId);

                // Forced-join: بررسی عضویت در کانال‌های اجباری
                $sendDebug("[DEBUG 16] شروع بررسی forced join", $telegramId);
                
                try {
                    $joinCheck = checkForcedJoinChannels(
                        $bot,
                        $telegramId,
                        $forcedJoinChannelService,
                        $logService,
                        $forcedJoinChannel
                    );
                    
                    $sendDebug("[DEBUG 17] forced join بررسی شد - hasJoined: " . ($joinCheck['hasJoined'] ? 'YES' : 'NO'), $telegramId);

                    $logService->debug('🔐 بررسی forced join', [
                        'telegram_id' => $telegramId,
                        'hasJoined' => $joinCheck['hasJoined'],
                        'channels' => $joinCheck['channels'],
                        'channels_count' => count($joinCheck['channels'] ?? []),
                    ], $telegramId);

                    $sendDebug("[DEBUG 18] بررسی مجوز دانلود - allowed: " . ($forcedJoinService->isUserAllowedToDownload($joinCheck['hasJoined']) ? 'YES' : 'NO'), $telegramId);
                } catch (\Throwable $e) {
                    // اگر خطا در بررسی forced join رخ داد، لاگ می‌کنیم و فرض می‌کنیم کاربر عضو نیست
                    $sendDebug("[DEBUG 17] خطا در بررسی forced join: " . $e->getMessage(), $telegramId);
                    
                    $logService->error('❌ خطا در بررسی forced join', [
                        'telegram_id' => $telegramId,
                        'error_message' => $e->getMessage(),
                        'error_file' => $e->getFile(),
                        'error_line' => $e->getLine(),
                        'trace' => substr($e->getTraceAsString(), 0, 500),
                    ], $telegramId, null, null, null, $e);
                    
                    // در صورت خطا، فرض می‌کنیم کاربر عضو نیست
                    $joinCheck = [
                        'hasJoined' => false,
                        'channels' => [],
                        'channelEntities' => [],
                    ];
                    
                    $sendDebug("[DEBUG 18] خطا رخ داد - فرض می‌کنیم کاربر عضو نیست", $telegramId);
                }

                // بررسی اینکه joinCheck معتبر است
                $hasJoined = $joinCheck['hasJoined'] ?? false;
                $channels = $joinCheck['channels'] ?? [];
                $channelEntities = $joinCheck['channelEntities'] ?? [];
                
                if (!$forcedJoinService->isUserAllowedToDownload($hasJoined)) {
                    $sendDebug("[DEBUG 19] کاربر مجاز به دانلود نیست - نمایش پیام عضویت", $telegramId);
                    
                    try {
                        $text = "برای دریافت مدیا، ابتدا باید در کانال(های) مورد نیاز عضو شوید.";
                        if (!empty($channels)) {
                            $text .= PHP_EOL . PHP_EOL . "کانال‌های مورد نیاز:" . PHP_EOL;
                            foreach ($channels as $ch) {
                                $text .= "• {$ch}" . PHP_EOL;
                            }
                        }

                        // ساخت کیبورد با دکمه‌های عضویت در کانال‌ها
                        $keyboard = \SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardMarkup::make();
                
                        foreach ($channelEntities as $channelEntity) {
                            $channel = $channelEntity->getChannelUsername();
                            $inviteLink = $channelEntity->getInviteLink();
                            
                            // اگر کانال chat_id است (با - شروع می‌شود)
                            if (str_starts_with($channel, '-')) {
                                // اگر invite link وجود دارد، از آن استفاده می‌کنیم
                                if ($inviteLink !== null && $inviteLink !== '') {
                                    $keyboard->addRow(
                                        \SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton::make(
                                            "🔗 عضویت در کانال",
                                            url: $inviteLink
                                        )
                                    );
                                }
                                // اگر invite link وجود ندارد، دکمه اضافه نمی‌کنیم
                                continue;
                            }
                            
                            // ساخت لینک کانال برای username
                            $channelLink = $channel;
                            // اگر کانال با @ شروع شده باشد، @ را حذف می‌کنیم برای لینک
                            if (str_starts_with($channelLink, '@')) {
                                $channelLink = substr($channelLink, 1);
                            }
                            
                            // ساخت لینک کانال
                            $channelUrl = "https://t.me/{$channelLink}";
                            $displayName = str_starts_with($channel, '@') ? $channel : '@' . $channel;
                            $keyboard->addRow(
                                \SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton::make(
                                    "🔗 عضویت در {$displayName}",
                                    url: $channelUrl
                                )
                            );
                        }
                        
                        // دکمه برای تلاش مجدد (استفاده از deep link)
                        try {
                            $botInfo = $bot->getMe();
                            $botUsername = $botInfo->username ?? '';
                            if ($botUsername !== '') {
                                $retryUrl = "https://t.me/{$botUsername}?start={$payload}";
                                $keyboard->addRow(
                                    \SergiX44\Nutgram\Telegram\Types\Keyboard\InlineKeyboardButton::make(
                                        '🔄 تلاش مجدد',
                                        url: $retryUrl
                                    )
                                );
                            }
                        } catch (\Throwable) {
                            // اگر خطا رخ داد، دکمه را اضافه نمی‌کنیم
                        }

                        $bot->sendMessage($text, reply_markup: $keyboard);
                        $sendDebug("[DEBUG 19-END] پیام عضویت ارسال شد", $telegramId);
                        return;
                    } catch (\Throwable $e) {
                        // اگر خطا در ساخت یا ارسال پیام عضویت رخ داد
                        $sendDebug("[DEBUG 19-ERROR] خطا در ارسال پیام عضویت: " . $e->getMessage(), $telegramId);
                        
                        $logService->error('❌ خطا در ارسال پیام عضویت', [
                            'telegram_id' => $telegramId,
                            'error_message' => $e->getMessage(),
                            'error_file' => $e->getFile(),
                            'error_line' => $e->getLine(),
                        ], $telegramId, null, null, null, $e);
                        
                        // تلاش برای ارسال پیام ساده
                        try {
                            $bot->sendMessage("برای دریافت مدیا، ابتدا باید در کانال(های) مورد نیاز عضو شوید.");
                        } catch (\Throwable) {
                            // اگر این هم خطا داد، نادیده می‌گیریم
                        }
                        return;
                    }
                }

                $sendDebug("[DEBUG 19] کاربر مجاز به دانلود است - ادامه پردازش", $telegramId);

                // کاربر را ایجاد/بازیابی می‌کنیم
                $user = $userService->getOrCreateByTelegramId($telegramId);
                
                $sendDebug("[DEBUG 20] کاربر ایجاد/بازیابی شد - ID: " . $user->getId() . ", Mode: " . $user->getMode(), $telegramId);

                // محدودیت ساده برای حالت رایگان
                $current = $downloadCounter[$telegramId] ?? 0;
                
                $sendDebug("[DEBUG 21] بررسی محدودیت دانلود - current: " . $current . ", limit: " . $freeDownloadLimit . ", mode: " . $user->getMode(), $telegramId);
                
                if ($user->getMode() === 'free' && $current >= $freeDownloadLimit) {
                    $sendDebug("[DEBUG 22] محدودیت دانلود تمام شده - return", $telegramId);
                    $bot->sendMessage("سهمیه دانلود شما در حالت رایگان تمام شده است. برای استفاده نامحدود، حساب خود را به پریمیوم ارتقا دهید.");
                    return;
                }

                $downloadCounter[$telegramId] = $current + 1;
                
                $sendDebug("[DEBUG 22] محدودیت بررسی شد - دانلود مجاز - counter: " . $downloadCounter[$telegramId], $telegramId);

                // ارسال مدیا بر اساس نوع ثبت‌شده در مدیا
                $caption = $media->getCaption();
                // برای ارسال به کاربر، همیشه از telegram_id استفاده می‌کنیم
                $chatId = $telegramId;
                
                $fileId = $media->getTelegramFileId();
                $mediaType = $media->getType();
                
                $sendDebug("[DEBUG 23] آماده‌سازی ارسال - fileId length: " . strlen($fileId) . ", type: " . $mediaType . ", chatId: " . $chatId, $telegramId);
                
                // لاگ کامل قبل از ارسال - این لاگ خیلی مهم است!
                $logService->info('📤 در حال ارسال مدیا', [
                    'telegram_id' => $telegramId,
                    'chat_id' => $chatId,
                    'media_id' => $media->getId(),
                    'type' => $mediaType,
                    'file_id' => $fileId,
                    'file_id_length' => strlen($fileId),
                    'file_id_first_chars' => substr($fileId, 0, 20),
                    'caption' => $caption ? 'exists (' . strlen($caption) . ' chars)' : 'null',
                ], $telegramId);
                
                // بررسی اینکه file_id خالی نباشد
                $sendDebug("[DEBUG 24] بررسی file_id خالی - empty: " . (empty($fileId) ? 'YES' : 'NO'), $telegramId);
                
                if (empty($fileId) || trim($fileId) === '') {
                    $sendDebug("[DEBUG 25] file_id خالی است - return", $telegramId);
                    $logService->error('❌ file_id خالی یا نامعتبر است', [
                        'telegram_id' => $telegramId,
                        'media_id' => $media->getId(),
                        'type' => $mediaType,
                        'file_id' => $fileId,
                        'file_id_empty' => empty($fileId),
                        'file_id_trimmed_empty' => trim($fileId ?? '') === '',
                    ], $telegramId);
                    $bot->sendMessage("خطا: فایل یافت نشد. لطفاً با پشتیبانی تماس بگیرید.");
                    return;
                }
                
                $sendDebug("[DEBUG 25] file_id معتبر است", $telegramId);
                
                // بررسی نوع فایل - اگر نوع نامعتبر یا غیرقابل ارسال باشد
                $sendDebug("[DEBUG 26] بررسی نوع فایل - type: " . $mediaType . ", isSendable: " . (MediaType::isSendable($mediaType) ? 'YES' : 'NO'), $telegramId);
                
                if (!MediaType::isSendable($mediaType)) {
                    $sendDebug("[DEBUG 27] نوع فایل غیرقابل ارسال است - return", $telegramId);
                    $logService->error('❌ نوع فایل نامعتبر یا غیرقابل ارسال است', [
                        'telegram_id' => $telegramId,
                        'media_id' => $media->getId(),
                        'type' => $mediaType,
                        'valid_sendable_types' => MediaType::getSendableTypes(),
                    ], $telegramId);
                    $bot->sendMessage("خطا: نوع فایل نامعتبر است. لطفاً با پشتیبانی تماس بگیرید.");
                    return;
                }
                
                $sendDebug("[DEBUG 27] نوع فایل معتبر است - شروع ارسال", $telegramId);
                
                try {
                    // برای اطمینان از اینکه اگر مدیا نال بود، لاگ ارور ندهد
                    $mediaId = $media?->getId() ?? 'unknown';
                
                    $sendDebug("[DEBUG 28] ورود به switch case - mediaType: " . $mediaType, $telegramId);
            
                    switch ($mediaType) {
                        case MediaType::PHOTO->value:
                            $sendDebug("[DEBUG 29] case PHOTO انتخاب شد", $telegramId);
                            $logService->debug('📷 ارسال Photo شروع شد', [
                                'telegram_id' => $telegramId,
                                'chat_id' => $chatId,
                                'file_id' => substr($fileId, 0, 30) . '...',
                            ], $telegramId);
                            
                            $sendDebug("[DEBUG 30] فراخوانی sendPhoto - fileId: " . substr($fileId, 0, 30) . "...", $telegramId);
                            
                            $bot->sendPhoto(
                                $fileId, // ✅ Positional
                                chat_id: $chatId,
                                caption: $caption
                            );
                            
                            $sendDebug("[DEBUG 31] sendPhoto اجرا شد - موفق", $telegramId);
                            
                            $logService->info('✅ Photo ارسال شد', [
                                'telegram_id' => $telegramId,
                                'chat_id' => $chatId,
                                'media_id' => $mediaId,
                                'file_id' => substr($fileId, 0, 30) . '...',
                            ], $telegramId);
                            break;
                
                        case MediaType::VIDEO->value:
                            $sendDebug("[DEBUG 29] case VIDEO انتخاب شد", $telegramId);
                            $logService->debug('🎥 ارسال Video شروع شد', [
                                'telegram_id' => $telegramId,
                                'chat_id' => $chatId,
                                'file_id' => substr($fileId, 0, 30) . '...',
                            ], $telegramId);
                            
                            $sendDebug("[DEBUG 30] فراخوانی sendVideo - fileId: " . substr($fileId, 0, 30) . "...", $telegramId);
                            
                            $bot->sendVideo(
                                $fileId, // ✅ Positional
                                chat_id: $chatId,
                                caption: $caption
                            );
                            
                            $sendDebug("[DEBUG 31] sendVideo اجرا شد - موفق", $telegramId);
                            
                            $logService->info('✅ Video ارسال شد', [
                                'telegram_id' => $telegramId,
                                'chat_id' => $chatId,
                                'media_id' => $mediaId,
                                'file_id' => substr($fileId, 0, 30) . '...',
                            ], $telegramId);
                            break;
                
                        case MediaType::DOCUMENT->value:
                        default:
                            $sendDebug("[DEBUG 29] case DOCUMENT انتخاب شد", $telegramId);
                            $logService->debug('📄 ارسال Document شروع شد', [
                                'telegram_id' => $telegramId,
                                'chat_id' => $chatId,
                                'file_id' => substr($fileId, 0, 30) . '...',
                            ], $telegramId);
                            
                            $sendDebug("[DEBUG 30] فراخوانی sendDocument - fileId: " . substr($fileId, 0, 30) . "...", $telegramId);
                            
                            $bot->sendDocument(
                                $fileId, // ✅ Positional
                                chat_id: $chatId,
                                caption: $caption
                            );
                            
                            $sendDebug("[DEBUG 31] sendDocument اجرا شد - موفق", $telegramId);
                            
                            $logService->info('✅ Document ارسال شد', [
                                'telegram_id' => $telegramId,
                                'chat_id' => $chatId,
                                'media_id' => $mediaId,
                                'file_id' => substr($fileId, 0, 30) . '...',
                            ], $telegramId);
                            break;
                    }
                    
                    $sendDebug("[DEBUG 32] switch case کامل شد - ارسال موفق", $telegramId);
                } catch (\SergiX44\Nutgram\Telegram\Exceptions\TelegramException $e) {
                    $sendDebug("[DEBUG 33] CATCH TelegramException - error: " . $e->getMessage(), $telegramId);
                    // استفاده از متغیر ایمن شده mediaId
                    $mediaId = $media?->getId() ?? 'unknown';
                    
                    $errorMessage = $e->getMessage();
                    
                    $logService->error('❌ خطا در ارسال مدیا (TelegramException)', [
                        'telegram_id' => $telegramId,
                        'chat_id' => $chatId,
                        'media_id' => $mediaId,
                        'type' => $mediaType,
                        'file_id_part' => substr($fileId, 0, 30),
                        'error_message' => $errorMessage,
                        'error_code' => $e->getCode(),
                    ], $telegramId, null, null, $mediaId, $e); // در آرگومان‌های لاگر هم mediaId ایمن باشد
                
                    $userMessage = "خطا در ارسال فایل.";
                    if (str_contains(strtolower($errorMessage), 'file') || str_contains(strtolower($errorMessage), 'not found')) {
                        $userMessage = "خطا: فایل یافت نشد یا منقضی شده است.";
                    } elseif (str_contains(strtolower($errorMessage), 'chat not found')) {
                        $userMessage = "خطا: کاربر یافت نشد.";
                    }
                    
                    $bot->sendMessage($userMessage);
                    return;
                
                } catch (\Throwable $e) {
                    $sendDebug("[DEBUG 34] CATCH Throwable - error: " . $e->getMessage() . " | file: " . $e->getFile() . " | line: " . $e->getLine(), $telegramId);
                    
                    $mediaId = $media?->getId() ?? 'unknown';
                
                    $logService->error('❌ خطا در ارسال مدیا (Throwable)', [
                        'telegram_id' => $telegramId,
                        'chat_id' => $chatId,
                        'media_id' => $mediaId,
                        'error_message' => $e->getMessage(),
                        'file' => $e->getFile(),
                        'line' => $e->getLine(),
                    ], $telegramId, null, null, $mediaId, $e);
                    
                    $bot->sendMessage("خطا در ارسال فایل. لطفاً دوباره تلاش کنید.");
                    return;
                }

                $sendDebug("[DEBUG 35] ارسال موفق - شروع ثبت آنالیتیکس", $telegramId);

                // ثبت آنالیتیکس ساده
                try {
                    $sendDebug("[DEBUG 36] ثبت آنالیتیکس - user_id: " . $user->getId() . ", media_id: " . $media->getId(), $telegramId);
                    $analyticsService->recordDownload($user->getId(), $media->getId());
                    
                    $sendDebug("[DEBUG 37] آنالیتیکس ثبت شد - موفق", $telegramId);
                    
                    $logService->info('دانلود مدیا موفق', [
                        'user_id' => $user->getId(),
                        'telegram_id' => $telegramId,
                        'media_id' => $media->getId(),
                        'media_type' => $media->getType(),
                    ], $telegramId);
                } catch (Throwable $e) {
                    $sendDebug("[DEBUG 37] CATCH در ثبت آنالیتیکس - error: " . $e->getMessage(), $telegramId);
                    
                    $logService->error('خطا در ثبت آنالیتیکس دانلود', [
                        'user_id' => $user->getId(),
                        'media_id' => $media->getId(),
                    ], $telegramId, $user->getId(), null, $media->getId(), $e);
                }
                
                    $sendDebug("[DEBUG 38] پردازش لینک یکتا کامل شد - return", $telegramId);
                    error_log("[START-HANDLER] Payload processing completed successfully - TelegramID: {$telegramId}");
                    
                    // ارسال اطلاعات نهایی به owner
                    if ($ownerTelegramId !== null && $ownerTelegramId > 0) {
                        try {
                            $finalInfo = "✅ <b>پردازش لینک یکتا کامل شد</b>" . PHP_EOL . PHP_EOL;
                            $finalInfo .= "👤 <b>کاربر:</b> <code>{$telegramId}</code>" . PHP_EOL;
                            $finalInfo .= "📝 <b>Text:</b> <code>" . htmlspecialchars($text) . "</code>" . PHP_EOL;
                            $finalInfo .= "🎯 <b>Payload:</b> <code>" . htmlspecialchars($payload) . "</code>" . PHP_EOL;
                            $finalInfo .= "🔑 <b>UUID:</b> <code>" . htmlspecialchars($uuid ?? 'null') . "</code>" . PHP_EOL;
                            $finalInfo .= "🆔 <b>Media ID:</b> <code>" . htmlspecialchars($mediaId ?? 'null') . "</code>" . PHP_EOL;
                            $finalInfo .= "📦 <b>Media پیدا شد:</b> " . ($media !== null ? 'بله (ID: ' . $media->getId() . ')' : 'خیر') . PHP_EOL;
                            
                            $bot->sendMessage($finalInfo, chat_id: $ownerTelegramId, parse_mode: 'HTML');
                        } catch (\Throwable) {
                            // نادیده می‌گیریم
                        }
                    }
                    
                    return;
                } catch (\Throwable $e) {
                    // اگر خطای غیرمنتظره در کل بخش payload رخ داد
                    error_log("[START-HANDLER] ===== FATAL ERROR IN PAYLOAD PROCESSING ===== TelegramID: {$telegramId}");
                    error_log("[START-HANDLER] Error Message: " . $e->getMessage());
                    error_log("[START-HANDLER] Error File: " . $e->getFile());
                    error_log("[START-HANDLER] Error Line: " . $e->getLine());
                    error_log("[START-HANDLER] Error Trace: " . substr($e->getTraceAsString(), 0, 2000));
                    
                    try {
                        $sendDebug("[DEBUG-PAYLOAD-ERROR] خطای fatal در پردازش payload: " . $e->getMessage(), $telegramId);
                    } catch (\Throwable) {
                        error_log("[START-HANDLER] Could not send debug message");
                    }
                    
                    try {
                        $logService->error('❌ خطای fatal در پردازش payload', [
                            'telegram_id' => $telegramId,
                            'payload' => $payload ?? 'unknown',
                            'error_message' => $e->getMessage(),
                            'error_file' => $e->getFile(),
                            'error_line' => $e->getLine(),
                            'trace' => substr($e->getTraceAsString(), 0, 1000),
                        ], $telegramId, null, null, null, $e);
                    } catch (\Throwable $logError) {
                        error_log("[START-HANDLER] Could not log to logService: " . $logError->getMessage());
                    }
                    
                    // تلاش برای ارسال پیام خطا به کاربر
                    try {
                        $bot->sendMessage("خطا در پردازش لینک. لطفاً دوباره تلاش کنید یا با پشتیبانی تماس بگیرید.");
                        error_log("[START-HANDLER] Error message sent to user");
                    } catch (\Throwable $sendError) {
                        error_log("[START-HANDLER] Could not send error message to user: " . $sendError->getMessage());
                    }
                    return;
                }
            }

            $sendDebug("[DEBUG 7] payload خالی است - شروع معمولی", $telegramId);

            // ارسال اطلاعات به owner
            if ($ownerTelegramId !== null && $ownerTelegramId > 0) {
                try {
                    $bot->sendMessage("✅ <b>شروع معمولی (بدون payload)</b>" . PHP_EOL . PHP_EOL . "👤 <b>کاربر:</b> <code>{$telegramId}</code>" . PHP_EOL . "📝 <b>Text:</b> <code>" . htmlspecialchars($text) . "</code>", chat_id: $ownerTelegramId, parse_mode: 'HTML');
                } catch (\Throwable) {
                    // نادیده می‌گیریم
                }
            }

            // شروع معمولی - نمایش منوی اصلی
            $logService->info('🏠 شروع معمولی - بدون payload', [
                'telegram_id' => $telegramId,
                'payload' => $payload,
            ], $telegramId);
            
            $user = $userService->getOrCreateByTelegramId($telegramId);
            
            $sendDebug("[DEBUG 39] کاربر برای منوی اصلی ایجاد/بازیابی شد - ID: " . $user->getId(), $telegramId);
            
            // اگر owner است، به صورت خودکار در دیتابیس ذخیره می‌شود
            if ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) {
                $sendDebug("[DEBUG 40] کاربر owner است - ایجاد/بازیابی owner", $telegramId);
                $adminService->getOrCreateOwner($telegramId);
            } else {
                $sendDebug("[DEBUG 40] کاربر owner نیست", $telegramId);
            }
            
            $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
                || $adminService->getByTelegramId($telegramId) !== null;
            
            $sendDebug("[DEBUG 41] بررسی admin - isAdmin: " . ($isAdmin ? 'YES' : 'NO'), $telegramId);
            
            $modeText = $user->getMode() === 'premium' ? '⭐ پریمیوم' : '🆓 رایگان';
            $welcomeText = "👋 سلام! به ربات آپلودر خوش آمدید." . PHP_EOL . PHP_EOL;
            $welcomeText .= "وضعیت حساب: {$modeText}";
            
            $sendDebug("[DEBUG 42] آماده ارسال منوی اصلی - mode: " . $user->getMode(), $telegramId);
            
            $bot->sendMessage(
                $welcomeText,
                reply_markup: buildMainMenu($isAdmin)
            );
            
            $sendDebug("[DEBUG 43] منوی اصلی ارسال شد - پایان", $telegramId);
            
            $logService->info('✅ منوی اصلی نمایش داده شد', [
                'telegram_id' => $telegramId,
                'is_admin' => $isAdmin,
            ], $telegramId);
            
        } catch (\Throwable $e) {
            // لاگ خطای کلی - این باید همیشه کار کند
            error_log("[START-HANDLER] ===== FATAL ERROR IN START HANDLER ===== ");
            error_log("[START-HANDLER] Error Message: " . $e->getMessage());
            error_log("[START-HANDLER] Error File: " . $e->getFile());
            error_log("[START-HANDLER] Error Line: " . $e->getLine());
            error_log("[START-HANDLER] Error Trace: " . substr($e->getTraceAsString(), 0, 2000));
            
            // تلاش برای لاگ در LogService
            try {
                $telegramId = $bot->user()?->id ?? 0;
                error_log("[START-HANDLER] Attempting to log to LogService - TelegramID: {$telegramId}");
                $logService->error('❌ خطای fatal در هندلر /start', [
                    'telegram_id' => $telegramId,
                    'error_message' => $e->getMessage(),
                    'error_file' => $e->getFile(),
                    'error_line' => $e->getLine(),
                ], $telegramId, null, null, null, $e);
                error_log("[START-HANDLER] LogService->error executed successfully");
            } catch (\Throwable $logError) {
                error_log("[START-HANDLER] Could not log to LogService: " . $logError->getMessage());
            }
            
            // تلاش برای ارسال پیام خطا به کاربر
            try {
                $telegramId = $bot->user()?->id ?? null;
                error_log("[START-HANDLER] Attempting to send error message - TelegramID: " . ($telegramId ?? 'NULL'));
                if ($telegramId !== null && $telegramId > 0) {
                    $bot->sendMessage("خطا در پردازش درخواست. لطفاً دوباره تلاش کنید.", chat_id: $telegramId);
                    error_log("[START-HANDLER] Error message sent to user successfully");
                } else {
                    error_log("[START-HANDLER] TelegramID is null or invalid, cannot send message");
                }
            } catch (\Throwable $sendError) {
                error_log("[START-HANDLER] Could not send error message to user: " . $sendError->getMessage());
            }
        }
    });
}


