<?php

declare(strict_types=1);

namespace Bot\Domain\Service;

use Bot\Domain\Repository\LogRepositoryInterface;

final class LogService
{
    public function __construct(
        private readonly LogRepositoryInterface $logRepository,
    ) {
    }

    /**
     * لاگ با سطح دلخواه
     */
    public function log(
        string $level,
        string $message,
        array $context = [],
        ?int $telegramId = null,
        ?int $userId = null,
        ?int $adminId = null,
        ?int $mediaId = null,
        ?\Throwable $exception = null,
    ): void {
        $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 1);
        $file = $backtrace[0]['file'] ?? null;
        $line = $backtrace[0]['line'] ?? null;

        if ($file !== null) {
            // تبدیل مسیر کامل به مسیر نسبی
            $file = str_replace(__DIR__ . '/../../..', '', $file);
        }

        $this->logRepository->log(
            level: $level,
            message: $message,
            context: $context,
            file: $file,
            line: $line,
            telegramId: $telegramId,
            userId: $userId,
            adminId: $adminId,
            mediaId: $mediaId,
            exception: $exception,
        );
    }

    public function debug(
        string $message,
        array $context = [],
        ?int $telegramId = null,
        ?int $userId = null,
        ?int $adminId = null,
        ?int $mediaId = null,
        ?\Throwable $exception = null,
    ): void {
        $this->log('DEBUG', $message, $context, $telegramId, $userId, $adminId, $mediaId, $exception);
    }

    public function info(
        string $message,
        array $context = [],
        ?int $telegramId = null,
        ?int $userId = null,
        ?int $adminId = null,
        ?int $mediaId = null,
        ?\Throwable $exception = null,
    ): void {
        $this->log('INFO', $message, $context, $telegramId, $userId, $adminId, $mediaId, $exception);
    }

    public function warning(
        string $message,
        array $context = [],
        ?int $telegramId = null,
        ?int $userId = null,
        ?int $adminId = null,
        ?int $mediaId = null,
        ?\Throwable $exception = null,
    ): void {
        $this->log('WARNING', $message, $context, $telegramId, $userId, $adminId, $mediaId, $exception);
    }

    public function error(
        string $message,
        array $context = [],
        ?int $telegramId = null,
        ?int $userId = null,
        ?int $adminId = null,
        ?int $mediaId = null,
        ?\Throwable $exception = null,
    ): void {
        $this->log('ERROR', $message, $context, $telegramId, $userId, $adminId, $mediaId, $exception);
    }

    public function critical(
        string $message,
        array $context = [],
        ?int $telegramId = null,
        ?int $userId = null,
        ?int $adminId = null,
        ?int $mediaId = null,
        ?\Throwable $exception = null,
    ): void {
        $this->log('CRITICAL', $message, $context, $telegramId, $userId, $adminId, $mediaId, $exception);
    }

    /**
     * دریافت لاگ‌ها با فیلتر
     */
    public function getLogs(
        ?string $level = null,
        ?\DateTimeImmutable $startDate = null,
        ?\DateTimeImmutable $endDate = null,
        ?int $telegramId = null,
        int $limit = 100,
        int $offset = 0,
    ): array {
        return $this->logRepository->findLogs($level, $startDate, $endDate, $telegramId, $limit, $offset);
    }

    /**
     * شمارش لاگ‌ها
     */
    public function countLogs(
        ?string $level = null,
        ?\DateTimeImmutable $startDate = null,
        ?\DateTimeImmutable $endDate = null,
        ?int $telegramId = null,
    ): int {
        return $this->logRepository->countLogs($level, $startDate, $endDate, $telegramId);
    }

    /**
     * حذف لاگ‌های قدیمی
     */
    public function deleteOldLogs(\DateTimeImmutable $beforeDate): int
    {
        return $this->logRepository->deleteOldLogs($beforeDate);
    }
}

