<?php

declare(strict_types=1);

namespace Bot\Infrastructure\Repository;

use Bot\Domain\Entity\Media;
use Bot\Domain\Repository\MediaRepositoryInterface;

/**
 * پیاده‌سازی in-memory برای مدیاها.
 * توجه: شناسه‌ها در دامین مقداردهی می‌شوند و این ریپوزیتوری فقط اشیاء را در حافظه نگه می‌دارد.
 */
final class InMemoryMediaRepository implements MediaRepositoryInterface
{
    /** @var array<int,Media> keyed by id */
    private array $mediaById = [];

    /** @var array<int,list<Media>> keyed by folder_id */
    private array $mediaByFolderId = [];

    private int $nextId = 1;

    public function findById(int $id): ?Media
    {
        return $this->mediaById[$id] ?? null;
    }

    public function findByUuid(string $uuid): ?Media
    {
        foreach ($this->mediaById as $media) {
            if ($media->getUuid() === $uuid) {
                return $media;
            }
        }
        return null;
    }

    /**
     * @return Media[]
     */
    public function findByFolderId(int $folderId): array
    {
        return array_values($this->mediaByFolderId[$folderId] ?? []);
    }

    public function save(Media $media): void
    {
        $id       = $media->getId();
        $folderId = $media->getFolderId();

        // اگر media جدید است (id=0)، Media جدید با ID جدید می‌سازیم
        if ($id === 0) {
            $media = new Media(
                $this->nextId++,
                $media->getFolderId(),
                $media->getOwnerAdminId(),
                $media->getTelegramFileId(),
                $media->getType(),
                $media->getUuid(),
                $media->getThumbnailFileId(),
                $media->isThumbnailEnabled(),
                $media->getCaption(),
                $media->getAutoDeleteSeconds(),
                $media->isLinkInCaptionEnabled()
            );
            $id = $media->getId();
        }

        $this->mediaById[$id] = $media;

        if (!isset($this->mediaByFolderId[$folderId])) {
            $this->mediaByFolderId[$folderId] = [];
        }

        $this->mediaByFolderId[$folderId][$id] = $media;
    }

    public function delete(int $id): void
    {
        $media = $this->mediaById[$id] ?? null;
        if ($media === null) {
            return;
        }

        $folderId = $media->getFolderId();
        unset($this->mediaById[$id]);
        
        if (isset($this->mediaByFolderId[$folderId][$id])) {
            unset($this->mediaByFolderId[$folderId][$id]);
        }
    }
}


