#!/bin/bash
# -*- coding: utf-8 -*-

# Script to check Telegram Bot status
# Usage: ./status.sh

export LANG=en_US.UTF-8
export LC_ALL=en_US.UTF-8

cd "$(dirname "$0")"

PID_FILE="bot.pid"
FOREVER_PID_FILE="bot.forever.pid"

# بررسی اینکه آیا forever script در حال اجراست
if [ -f "$FOREVER_PID_FILE" ]; then
    FOREVER_PID=$(cat "$FOREVER_PID_FILE")
    if ps -p "$FOREVER_PID" > /dev/null 2>&1; then
        echo "ℹ️  Forever script is running (PID: $FOREVER_PID)"
        echo ""
        # بررسی اینکه آیا bot process واقعاً در حال اجراست
        BOT_PIDS=$(ps aux | grep -E "[p]hp.*public/index\.php" | awk '{print $2}')
        if [ -n "$BOT_PIDS" ]; then
            BOT_PID=$(echo "$BOT_PIDS" | head -n 1)
            echo "✅ Bot is running under forever script"
            echo "Bot PID: $BOT_PID"
            echo ""
            echo "Process information:"
            ps -p "$BOT_PID" -o pid,ppid,cmd,etime,pcpu,pmem,stat 2>/dev/null || echo "Process not found"
            echo ""
            if [ -f "bot.log" ]; then
                echo "Last log lines:"
                tail -n 5 bot.log
            fi
            exit 0
        else
            echo "⚠️  Forever script is running but no bot process found"
            echo "Forever script will restart the bot automatically"
            exit 0
        fi
    else
        rm -f "$FOREVER_PID_FILE"
    fi
fi

if [ ! -f "$PID_FILE" ]; then
    echo "❌ Error: Bot is not running (PID file not found)"
    exit 1
fi

PID=$(cat "$PID_FILE")

if ! ps -p "$PID" > /dev/null 2>&1; then
    echo "❌ Error: Bot is not running (Process not found)"
    rm -f "$PID_FILE"
    exit 1
fi

# بررسی وضعیت پروسه
STATUS=$(ps -p "$PID" -o stat= 2>/dev/null)
if [ -z "$STATUS" ]; then
    echo "❌ Error: Cannot get process status"
    rm -f "$PID_FILE"
    exit 1
fi

# بررسی zombie process (Z)
if echo "$STATUS" | grep -q "Z"; then
    echo "⚠️  Warning: Bot process is ZOMBIE (PID: $PID)"
    echo "   This means the process is dead but not cleaned up."
    echo "   Run ./restart.sh to fix this."
    exit 1
fi

# بررسی stopped process (T)
if echo "$STATUS" | grep -q "T"; then
    echo "⚠️  Warning: Bot process is STOPPED (PID: $PID)"
    echo "   The process exists but is not running."
    echo "   Run ./restart.sh to fix this."
    exit 1
fi

echo "✅ Bot is running"
echo "PID: $PID"
echo "Status: $STATUS"
echo ""

echo "Process information:"
ps -p "$PID" -o pid,ppid,cmd,etime,pcpu,pmem,stat
echo ""

# بررسی آخرین فعالیت در لاگ
if [ -f "bot.log" ]; then
    echo "=== Last 10 log lines ==="
    tail -n 10 bot.log
    echo ""
    
    # بررسی زمان آخرین لاگ
    if command -v stat >/dev/null 2>&1; then
        LOG_MTIME=$(stat -c %Y bot.log 2>/dev/null || stat -f %m bot.log 2>/dev/null)
        CURRENT_TIME=$(date +%s)
        LOG_AGE=$((CURRENT_TIME - LOG_MTIME))
        
        if [ $LOG_AGE -gt 300 ]; then
            echo "⚠️  Warning: Last log entry is more than 5 minutes old ($LOG_AGE seconds)"
            echo "   The bot might be hung or not processing updates."
        else
            echo "✅ Log is recent (last update: $LOG_AGE seconds ago)"
        fi
    fi
else
    echo "⚠️  Warning: Log file not found"
fi

echo ""
echo "=== Checking for other bot processes ==="
OTHER_PIDS=$(ps aux | grep -E "[p]hp.*public/index\.php" | grep -v "$PID" | awk '{print $2}')
if [ -n "$OTHER_PIDS" ]; then
    echo "⚠️  Warning: Other bot processes found: $OTHER_PIDS"
    echo "   Consider running ./killall.sh to clean up"
else
    echo "✅ No other bot processes found"
fi

