<?php

declare(strict_types=1);

use Bot\Domain\Service\AdminService;
use Bot\Domain\Service\LogService;
use Bot\Domain\Service\MediaService;
use Bot\Infrastructure\Link\InMemoryLinkStore;
use SergiX44\Nutgram\Nutgram;

// بارگذاری helper functions
require_once __DIR__ . '/../helpers.php';

/**
 * هندلر آپلود مدیا توسط ادمین
 */
function registerMediaUploadHandler(
    Nutgram $bot,
    AdminService $adminService,
    MediaService $mediaService,
    InMemoryLinkStore $linkStore,
    ?int $ownerTelegramId,
    array &$userStates,
    LogService $logService,
): void {
    $bot->onMessage(function (Nutgram $bot) use (
        $adminService,
        $mediaService,
        $linkStore,
        $ownerTelegramId,
        &$userStates,
        $logService,
    ) {
        $from = $bot->user();
        if ($from === null) {
            return;
        }

        $telegramId = (int) $from->id;

        $message = $bot->message();
        if ($message === null) {
            return;
        }

        // بررسی اینکه آیا پیام واقعاً مدیا است یا نه
        $type   = null;
        $fileId = null;
        $caption = $message->caption ?? null;

        if (!empty($message->photo)) {
            $photo  = end($message->photo);
            $type   = 'photo';
            $fileId = $photo?->file_id;
        } elseif ($message->video !== null) {
            $type   = 'video';
            $fileId = $message->video->file_id;
        } elseif ($message->document !== null) {
            $type   = 'document';
            $fileId = $message->document->file_id;
        }

        // اگر پیام مدیا نیست، نادیده بگیر (به handler بعدی برو)
        if ($fileId === null || $type === null) {
            return;
        }

        // بررسی دسترسی ادمین
        $isAdmin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId) 
            || $adminService->getByTelegramId($telegramId) !== null;
        
        if (!$isAdmin) {
            // فقط ادمین‌ها اجازه آپلود دارند
            return;
        }

        // اگر کاربر در state است و state برای ورودی متنی است، state را cancel کن و فایل را پردازش کن
        // چون آپلود فایل اولویت بالاتری دارد
        if (isset($userStates[$telegramId]) && in_array($userStates[$telegramId]['action'], ['add_admin', 'remove_admin', 'create_ad', 'set_autodelete', 'broadcast'], true)) {
            unset($userStates[$telegramId]);
            // ادامه می‌دهیم تا فایل را پردازش کنیم
        }

        // دریافت یا ایجاد admin
        $admin = ($ownerTelegramId !== null && $telegramId === $ownerTelegramId)
            ? $adminService->getOrCreateOwner($telegramId)
            : $adminService->getByTelegramId($telegramId);
        
        if ($admin === null) {
            return;
        }

        // در نسخه مینیمال، همه چیز در یک پوشه پیش‌فرض با شناسه 1 ذخیره می‌شود.
        $folderId = 1;

        $media = $mediaService->createMedia(
            folderId: $folderId,
            ownerAdminId: $admin->getId(),
            telegramFileId: $fileId,
            type: $type,
            caption: $caption,
        );

        $logService->info('مدیا ایجاد شد', [
            'media_id' => $media->getId(),
            'type' => $media->getType(),
            'file_id' => substr($media->getTelegramFileId(), 0, 20) . '...',
            'admin_id' => $admin->getId(),
        ], $telegramId, adminId: $admin->getId(), mediaId: $media->getId());

        $code = $linkStore->createForMedia($media);
        
        $logService->info('لینک مدیا ایجاد شد', [
            'code' => $code,
            'media_id' => $media->getId(),
        ], $telegramId, adminId: $admin->getId(), mediaId: $media->getId());

        // دریافت username ربات برای ساخت لینک کامل
        try {
            $botInfo = $bot->getMe();
            $botUsername = $botInfo->username ?? '';
            $deepLink = $botUsername !== '' 
                ? "https://t.me/{$botUsername}?start={$code}"
                : "/start {$code}";
        } catch (Throwable) {
            $deepLink = "/start {$code}";
        }

        // ارسال پیام مجزا (ریپلای به مدیای آپلود شده) تا همیشه قابل مشاهده باشد.
        // پیام‌های مدیا قابل ویرایش به متن دلخواه نیستند، بنابراین sendMessage مطمئن‌تر است.
        $bot->sendMessage(
            "مدیا ذخیره شد ✅" . PHP_EOL . PHP_EOL .
            "🔗 <b>کد لینک یکتا:</b>" . PHP_EOL .
            "<pre>{$code}</pre>" . PHP_EOL . PHP_EOL .
            "🌐 <b>لینک دیپ‌لینک:</b>" . PHP_EOL .
            "<pre>{$deepLink}</pre>",
            parse_mode: 'HTML',
            reply_to_message_id: $message->message_id ?? null,
        );
    });
}

