<?php

declare(strict_types=1);

namespace Bot\Infrastructure\Link;

use Bot\Domain\Entity\Media;

/**
 * ذخیره‌ساز in-memory برای لینک‌های یکتای مدیا.
 * در نسخه مینیمال، داده‌ها فقط در طول اجرای پروسه نگه‌داری می‌شوند.
 * 
 * توجه: این کلاس از UUID برای ساخت لینک استفاده می‌کند.
 * برای استفاده پایدار، باید از DatabaseLinkStore استفاده شود.
 */
final class InMemoryLinkStore
{
    /**
     * ساخت لینک یکتا برای مدیا
     * 
     * @return string کد لینک (به صورت get_{uuid})
     */
    public function createForMedia(Media $media): string
    {
        $uuid = $media->getUuid();
        
        if ($uuid !== null && $uuid !== '') {
            // استفاده مستقیم از UUID
            return 'get_' . $uuid;
        }
        
        // اگر UUID وجود ندارد (مدیا قدیمی)، از کد تصادفی استفاده می‌کنیم
        return 'get_' . bin2hex(random_bytes(8));
    }

    /**
     * استخراج UUID از کد لینک
     * 
     * @return string|null UUID در صورت موفقیت، null در غیر این صورت
     */
    public function extractUuid(string $code): ?string
    {
        // حذف پیشوند get_ اگر وجود دارد
        $payload = str_starts_with($code, 'get_') ? substr($code, 4) : $code;
        
        if (empty($payload)) {
            return null;
        }
        
        // اگر payload یک UUID معتبر است (36 کاراکتر با فرمت استاندارد)
        if (preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $payload)) {
            return $payload;
        }
        
        // اگر payload فقط عدد است (لینک قدیمی)، null برمی‌گردانیم
        // چون دیگر از media_id استفاده نمی‌کنیم
        return null;
    }

    /**
     * استخراج media_id از کد لینک (برای سازگاری با لینک‌های قدیمی)
     * 
     * @deprecated استفاده از extractUuid توصیه می‌شود
     * @return int|null media_id در صورت موفقیت، null در غیر این صورت
     */
    public function extractMediaId(string $code): ?int
    {
        // حذف پیشوند get_ اگر وجود دارد
        $payload = str_starts_with($code, 'get_') ? substr($code, 4) : $code;
        
        if (empty($payload)) {
            return null;
        }
        
        // اگر payload فقط عدد است (لینک قدیمی)، مستقیماً برمی‌گردانیم
        if (ctype_digit($payload)) {
            return (int) $payload;
        }
        
        // برای لینک‌های جدید با UUID، null برمی‌گردانیم
        return null;
    }

}


