<?php

declare(strict_types=1);

namespace Bot\Tests\Integration;

use Bot\Domain\Entity\User;
use Bot\Domain\Repository\MediaRepositoryInterface;
use Bot\Domain\Repository\UserRepositoryInterface;
use Bot\Domain\Service\ForcedJoinService;
use Bot\Domain\Service\MediaService;
use Bot\Domain\Service\UserService;
use Bot\Infrastructure\Link\InMemoryLinkStore;
use Bot\Infrastructure\Repository\InMemoryMediaRepository;
use Bot\Infrastructure\Repository\InMemoryUserRepository;
use PHPUnit\Framework\TestCase;

/**
 * تست یکپارچه سبک برای سناریو تحویل مدیا
 */
final class MediaDeliveryTest extends TestCase
{
    private UserRepositoryInterface $userRepository;
    private MediaRepositoryInterface $mediaRepository;
    private UserService $userService;
    private MediaService $mediaService;
    private ForcedJoinService $forcedJoinService;
    private InMemoryLinkStore $linkStore;

    protected function setUp(): void
    {
        $this->userRepository = new InMemoryUserRepository();
        $this->mediaRepository = new InMemoryMediaRepository();
        $this->userService = new UserService($this->userRepository);
        $this->mediaService = new MediaService($this->mediaRepository);
        $this->forcedJoinService = new ForcedJoinService();
        $this->linkStore = new InMemoryLinkStore();
    }

    public function testMediaDeliveryToUserAfterForcedJoin(): void
    {
        // ایجاد مدیا
        $media = $this->mediaService->createMedia(
            1,
            10,
            'file_123',
            'video',
            'Test video caption',
            'thumb_123'
        );

        // ایجاد لینک یکتا
        $uniqueLink = $this->linkStore->createForMedia($media);

        // کاربر لینک را باز می‌کند
        $user = $this->userService->getOrCreateByTelegramId(12345);

        // بررسی جوین اجباری
        $hasJoined = true;
        $isAllowed = $this->forcedJoinService->isUserAllowedToDownload($hasJoined);
        $this->assertTrue($isAllowed);

        // بازیابی مدیا از لینک یکتا
        $linkData = $this->linkStore->resolve($uniqueLink);
        $this->assertNotNull($linkData);
        $this->assertEquals('file_123', $linkData['file_id']);
        $this->assertEquals('video', $linkData['type']);
        $this->assertEquals('Test video caption', $linkData['caption']);

        // مدیا باید قابل بازیابی باشد
        $deliveredMedia = $this->mediaService->findById($media->getId());
        $this->assertNotNull($deliveredMedia);
        $this->assertEquals($media->getTelegramFileId(), $deliveredMedia->getTelegramFileId());
        $this->assertEquals('video', $deliveredMedia->getType());
        $this->assertEquals('Test video caption', $deliveredMedia->getCaption());
    }

    public function testMediaDeliveryWithThumbnail(): void
    {
        // ایجاد مدیا با thumbnail
        $media = $this->mediaService->createMedia(
            1,
            10,
            'file_456',
            'video',
            'Video with thumbnail',
            'thumb_456'
        );

        $this->assertNotNull($media->getThumbnailFileId());
        $this->assertEquals('thumb_456', $media->getThumbnailFileId());
        $this->assertTrue($media->isThumbnailEnabled());
    }

    public function testMediaDeliveryWithAutoDelete(): void
    {
        // ایجاد مدیا با auto-delete timer
        $media = $this->mediaService->createMedia(1, 10, 'file_789', 'photo', 'Photo with auto-delete');
        $media->setAutoDeleteSeconds(300);
        $this->mediaService->updateMedia($media);

        $this->assertEquals(300, $media->getAutoDeleteSeconds());
    }
}

