<?php
/**
 * ربات جدید آپلودر – اسکلت پایه
 * این فایل ورودی وبهوک تلگرام است.
 *
 * نکته:
 * - از همان config و دیتابیس فعلی استفاده می‌کنیم.
 * - به تدریج منطق دستورات، آپلود، پرداخت و ... را اینجا اضافه می‌کنیم.
 */

declare(strict_types=1);

require __DIR__ . '/config.php';
require __DIR__ . '/lib/Telegram.php';
require __DIR__ . '/lib/Features.php';
require __DIR__ . '/lib/Storage.php';
require __DIR__ . '/lib/StorageJson.php';
require __DIR__ . '/lib/StorageMysql.php';

use NewUploader\Telegram;
use NewUploader\Features;
use NewUploader\Storage;

// خواندن ورودی خام از تلگرام
$raw = file_get_contents('php://input');
if (!$raw) {
    exit('NO_INPUT');
}

$update = json_decode($raw, true);
if (!is_array($update)) {
    exit('BAD_JSON');
}

$tg = new Telegram(API_KEY);

// تشخیص نوع آپدیت (message / callback_query / ...)
if (isset($update['message'])) {
    $message = $update['message'];
    $chatId  = $message['chat']['id'] ?? null;
    $text    = trim($message['text'] ?? '');
    $fromId  = $message['from']['id'] ?? null;

    if ($chatId && $fromId) {
        // اوامر متنی اصلی
        if ($text === '/start') {
            handleStart($tg, $chatId, $fromId);
            exit;
        }

        if ($text === '/panel') {
            handleAdminPanel($tg, $chatId, $fromId);
            exit;
        }

        if ($text === '/cancel') {
            setUserStep($fromId, '');
            $tg->sendMessage($chatId, "✅ عملیات لغو شد.");
            exit;
        }

        // اگر در حالت انتظار آپلود هستیم و کاربر فایل فرستاد
        $step = getUserStep($fromId);
        if ($step === 'upload_wait_file') {
            if (isset($message['document']) || isset($message['video']) || isset($message['photo'])) {
                handleFileUpload($tg, $chatId, $fromId, $message);
                exit;
            } else {
                $tg->sendMessage($chatId, "لطفاً یک فایل (سند، ویدیو یا عکس) ارسال کنید.\nبرای لغو، /cancel را بفرستید.");
                exit;
            }
        }

        // فعلاً برای سایر پیام‌ها واکنش خاصی نداریم
    }
}
elseif (isset($update['callback_query'])) {
    $cb        = $update['callback_query'];
    $data      = $cb['data'] ?? '';
    $fromId    = $cb['from']['id'] ?? null;
    $message   = $cb['message'] ?? null;
    $chatId    = $message['chat']['id'] ?? null;
    $messageId = $message['message_id'] ?? null;

    if (strpos($data, 'admin:toggle:') === 0 && $fromId && $chatId && $messageId) {
        handleAdminToggle($tg, $chatId, $messageId, $fromId, $data, $cb['id']);
        exit;
    }

    if ($data === 'admin:storage' && $fromId && $chatId && $messageId) {
        handleAdminStorageMenu($tg, $chatId, $messageId, $fromId, $cb['id']);
        exit;
    }

    if (strpos($data, 'user:') === 0 && $fromId && $chatId && $messageId) {
        handleUserMenuCallback($tg, $chatId, $messageId, $fromId, $data, $cb['id']);
        exit;
    }
}

// --- توابع هندلر ساده اولیه ---

/**
 * /start
 */
function handleStart(Telegram $tg, int $chatId, ?int $fromId): void
{
    // ثبت کاربر در لایه ذخیره‌سازی اگر وجود ندارد
    if ($fromId) {
        Storage::instance()->ensureUser((int)$fromId, null);
    }

    // فعلاً یک متن ساده – بعداً از جدول settings خوانده می‌شود
    $text = "سلام 👋\n"
        ."به ربات آپلودر خوش آمدی.\n\n"
        ."از منوی زیر می‌تونی آپلود، مدیریت فایل‌ها و خرید اشتراک رو انجام بدی.";

    $keyboard = [
        'inline_keyboard' => [
            [
                ['text' => '📤 آپلود فایل',   'callback_data' => 'user:upload'],
            ],
            [
                ['text' => '📁 فایل‌های من',  'callback_data' => 'user:my_files'],
            ],
            [
                ['text' => '💳 خرید اشتراک',  'callback_data' => 'user:buy'],
            ],
            [
                ['text' => '⚙️ تنظیمات',      'callback_data' => 'user:settings'],
            ],
        ],
    ];

    $tg->sendMessage($chatId, $text, [
        'reply_markup' => json_encode($keyboard, JSON_UNESCAPED_UNICODE),
    ]);
}

/**
 * /panel – پنل پایه ادمین
 * فعلاً فقط نمایش چند فلگ ساده برای تست.
 */
function handleAdminPanel(Telegram $tg, int $chatId, ?int $fromId): void
{
    global $admins;

    // فقط ادمین‌ها
    if (!$fromId || !in_array($fromId, $admins, true)) {
        $tg->sendMessage($chatId, "❌ شما ادمین نیستید.");
        return;
    }

    // فلگ‌هایی که می‌خواهیم در منو نشان دهیم
    $flags = [
        'payments_enabled'          => 'پرداخت کلی',
        'payments_zarinpal_enabled' => 'درگاه زرین‌پال',
        'payments_zibal_enabled'    => 'درگاه زیبال',
        'vip_enabled'               => 'اشتراک VIP',
    ];

    $rows = [];
    foreach ($flags as $key => $label) {
        $on = Features::isEnabled($key);
        $emoji = $on ? '✅' : '❌';
        $rows[] = [
            [
                'text'          => "{$emoji} {$label}",
                'callback_data' => "admin:toggle:{$key}",
            ],
        ];
    }

    $keyboard = [
        'inline_keyboard' => array_merge(
            $rows,
            [
                [
                    [
                        'text'          => '💾 نوع ذخیره‌سازی: ' . strtoupper(getCurrentStorageDriver()),
                        'callback_data' => 'admin:storage',
                    ],
                ],
            ]
        ),
    ];

    $text = "⚙️ پنل مدیریت – برای روشن/خاموش کردن هر قابلیت روی دکمه‌اش بزن:";

    $tg->sendMessage($chatId, $text, [
        'reply_markup' => json_encode($keyboard, JSON_UNESCAPED_UNICODE),
    ]);
}

/**
 * هندل کردن کلیک روی دکمه‌های شیشه‌ای پنل ادمین
 */
function handleAdminToggle(Telegram $tg, int $chatId, int $messageId, int $fromId, string $data, string $callbackId): void
{
    global $admins;

    if (!in_array($fromId, $admins, true)) {
        $tg->answerCallbackQuery($callbackId, 'شما ادمین نیستید.', true);
        return;
    }

    // data به شکل admin:toggle:KEY
    $parts = explode(':', $data, 3);
    if (count($parts) !== 3) {
        return;
    }
    $key = $parts[2];

    // وضعیت فعلی
    $current = Features::isEnabled($key);
    $newVal  = $current ? 'off' : 'on';
    Features::set($key, $newVal);

    $tg->answerCallbackQuery($callbackId, $newVal === 'on' ? 'قابلیت فعال شد ✅' : 'قابلیت غیرفعال شد ❌');

    // بازسازی متن و کیبورد پنل
    handleAdminPanel($tg, $chatId, $fromId);
}

/**
 * منوی انتخاب نوع ذخیره‌سازی (JSON / MySQL)
 */
function handleAdminStorageMenu(Telegram $tg, int $chatId, int $messageId, int $fromId, string $callbackId): void
{
    global $admins;

    if (!in_array($fromId, $admins, true)) {
        $tg->answerCallbackQuery($callbackId, 'شما ادمین نیستید.', true);
        return;
    }

    $current = getCurrentStorageDriver();

    $keyboard = [
        'inline_keyboard' => [
            [
                [
                    'text'          => ($current === 'json' ? '✅ ' : '') . 'JSON (فایل)',
                    'callback_data' => 'admin:storage:set:json',
                ],
            ],
            [
                [
                    'text'          => ($current === 'mysql' ? '✅ ' : '') . 'MySQL (دیتابیس)',
                    'callback_data' => 'admin:storage:set:mysql',
                ],
            ],
        ],
    ];

    $text = "💾 نوع ذخیره‌سازی فعلی: <b>" . strtoupper($current) . "</b>\n"
        ."لطفاً یکی را انتخاب کنید.\n\n"
        ."توجه: برای استفاده از MySQL باید اتصال دیتابیس در config.php به درستی تنظیم شده باشد.";

    $tg->answerCallbackQuery($callbackId);
    $tg->editMessageText($chatId, $messageId, $text, [
        'reply_markup' => json_encode($keyboard, JSON_UNESCAPED_UNICODE),
    ]);
}

/**
 * هندل کردن دکمه‌های شیشه‌ای کاربر (منوی اصلی)
 */
function handleUserMenuCallback(Telegram $tg, int $chatId, int $messageId, int $fromId, string $data, string $callbackId): void
{
    // user:upload | user:my_files | user:buy | user:settings
    if ($data === 'user:upload') {
        setUserStep($fromId, 'upload_wait_file');
        $tg->answerCallbackQuery($callbackId, 'لطفاً فایل خود را ارسال کنید.');
        $tg->sendMessage($chatId, "📤 لطفاً فایل خود را ارسال کنید.\nبرای لغو، /cancel را بفرستید.");
        return;
    }

    if ($data === 'user:my_files') {
        $tg->answerCallbackQuery($callbackId);
        // در آینده لیست فایل‌ها از جدول userfiles/	files خوانده می‌شود
        $tg->sendMessage($chatId, "📁 بخش «فایل‌های من» به‌زودی تکمیل می‌شود.");
        return;
    }

    if ($data === 'user:buy') {
        $tg->answerCallbackQuery($callbackId);
        $tg->sendMessage($chatId, "💳 بخش خرید اشتراک (درگاه‌ها) را در مرحله بعد به این ربات وصل می‌کنیم.");
        return;
    }

    if ($data === 'user:settings') {
        $tg->answerCallbackQuery($callbackId);
        $tg->sendMessage($chatId, "⚙️ بخش تنظیمات کاربر به‌زودی اضافه می‌شود.");
        return;
    }

    // تنظیم نوع ذخیره‌سازی
    if ($data === 'admin:storage:set:json' || $data === 'admin:storage:set:mysql') {
        global $admins;
        if (!in_array($fromId, $admins, true)) {
            $tg->answerCallbackQuery($callbackId, 'شما ادمین نیستید.', true);
            return;
        }

        $driver = str_ends_with($data, ':mysql') ? 'mysql' : 'json';
        setStorageDriver($driver);

        $tg->answerCallbackQuery($callbackId, 'ذخیره‌سازی روی ' . strtoupper($driver) . ' تنظیم شد.');
        // برگشت به پنل اصلی
        handleAdminPanel($tg, $chatId, $fromId);
        return;
    }
}

/**
 * گرفتن step کاربر از جدول user
 */
function getUserStep(int $userId): string
{
    return Storage::instance()->getUserStep($userId);
}

/**
 * تنظیم step کاربر در جدول user
 */
function setUserStep(int $userId, string $step): void
{
    Storage::instance()->setUserStep($userId, $step);
}

/**
 * ذخیره اولیه فایل در دیتابیس (جدول files و userfiles)
 */
function handleFileUpload(Telegram $tg, int $chatId, int $fromId, array $message): void
{
    // تعیین نوع و file_id و سایز
    $type      = '';
    $fileId    = '';
    $fileSize  = 0;
    $thumbnail = null;

    if (isset($message['document'])) {
        $type     = 'document';
        $fileId   = $message['document']['file_id'];
        $fileSize = (int)($message['document']['file_size'] ?? 0);
    } elseif (isset($message['video'])) {
        $type     = 'video';
        $fileId   = $message['video']['file_id'];
        $fileSize = (int)($message['video']['file_size'] ?? 0);
        if (isset($message['video']['thumbnail']['file_id'])) {
            $thumbnail = $message['video']['thumbnail']['file_id'];
        }
    } elseif (isset($message['photo'])) {
        // بزرگ‌ترین سایز را انتخاب می‌کنیم
        $photoSizes = $message['photo'];
        $last       = end($photoSizes);
        $type       = 'photo';
        $fileId     = $last['file_id'];
        $fileSize   = (int)($last['file_size'] ?? 0);
    }

    if ($fileId === '' || $type === '') {
        $tg->sendMessage($chatId, "❌ نوع فایل پشتیبانی نمی‌شود.");
        return;
    }

    $caption = $message['caption'] ?? '';

    // تولید کد یکتا برای لینک
    $code = generateFileCode();

    $userId = (int)$fromId;

    // ذخیره در لایه Storage (خودش بسته به JSON یا MySQL تصمیم می‌گیرد)
    Storage::instance()->saveFile([
        'code'      => $code,
        'user_id'   => $userId,
        'file_id'   => $fileId,
        'file_size' => $fileSize,
        'caption'   => $caption,
        'type'      => $type,
        'thumbnail' => $thumbnail,
    ]);

    Storage::instance()->saveUserFile($userId, $code, $fileId, $caption, $type);

    // پاک کردن step
    setUserStep($userId, '');

    $tg->sendMessage($chatId,
        "✅ فایل با موفقیت ذخیره شد.\n"
        ."کد فایل شما: <code>{$code}</code>\n"
        ."می‌تونی بعداً با این کد لینک دانلود بگیری (این بخش را مرحله بعد اضافه می‌کنیم).",
        ['parse_mode' => 'HTML']
    );
}

/**
 * تولید کد یکتای فایل (مثلاً ۶ کاراکتر حروف/عدد)
 */
function generateFileCode(int $length = 6): string
{
    return Storage::instance()->generateUniqueFileCode($length);
}

/**
 * خواندن نوع ذخیره‌سازی فعلی از فایل config مخصوص Storage
 */
function getCurrentStorageDriver(): string
{
    $baseDir  = __DIR__;
    $confPath = $baseDir . '/storage/storage_config.json';

    if (is_file($confPath)) {
        $raw = file_get_contents($confPath);
        $cfg = json_decode($raw, true);
        if (is_array($cfg) && !empty($cfg['driver'])) {
            return $cfg['driver'] === 'mysql' ? 'mysql' : 'json';
        }
    }

    return 'json';
}

/**
 * تنظیم نوع ذخیره‌سازی (json / mysql) در فایل config
 */
function setStorageDriver(string $driver): void
{
    $driver = $driver === 'mysql' ? 'mysql' : 'json';
    $baseDir  = __DIR__;
    $storageDir = $baseDir . '/storage';
    if (!is_dir($storageDir)) {
        mkdir($storageDir, 0775, true);
    }
    $confPath = $storageDir . '/storage_config.json';
    file_put_contents($confPath, json_encode(['driver' => $driver], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}


