<?php
declare(strict_types=1);

namespace NewUploader;

/**
 * درایور ذخیره‌سازی مبتنی بر JSON
 */
class StorageJson implements StorageDriverInterface
{
    private string $dir;

    public function __construct(string $baseDir)
    {
        $this->dir = rtrim($baseDir, '/');
        if (!is_dir($this->dir)) {
            mkdir($this->dir, 0775, true);
        }
    }

    private function load(string $name): array
    {
        $file = "{$this->dir}/{$name}.json";
        if (!is_file($file)) {
            return [];
        }
        $raw = file_get_contents($file);
        $data = json_decode($raw, true);
        return is_array($data) ? $data : [];
    }

    private function save(string $name, array $data): void
    {
        $file = "{$this->dir}/{$name}.json";
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }

    public function ensureUser(int $id, ?string $name = null): void
    {
        $all = $this->load('users');
        $key = (string)$id;
        if (!isset($all[$key])) {
            $all[$key] = [
                'id'       => $id,
                'name'     => $name,
                'step'     => '',
                'timejoin' => date('Y-m-d H:i:s'),
            ];
            $this->save('users', $all);
        }
    }

    public function getUserStep(int $id): string
    {
        $all = $this->load('users');
        $key = (string)$id;
        return isset($all[$key]['step']) ? (string)$all[$key]['step'] : '';
    }

    public function setUserStep(int $id, string $step): void
    {
        $all = $this->load('users');
        $key = (string)$id;
        if (!isset($all[$key])) {
            $this->ensureUser($id, null);
            $all = $this->load('users');
        }
        $all[$key]['step'] = $step;
        $this->save('users', $all);
    }

    public function generateUniqueFileCode(int $length = 6): string
    {
        $chars = 'ABCDEFGHJKMNPQRSTUVWXYZ23456789';

        $files = $this->load('files');
        $existingCodes = array_column($files, 'code');

        while (true) {
            $code = '';
            for ($i = 0; $i < $length; $i++) {
                $code .= $chars[random_int(0, strlen($chars) - 1)];
            }
            if (!in_array($code, $existingCodes, true)) {
                return $code;
            }
        }
    }

    public function saveFile(array $data): void
    {
        $files = $this->load('files');
        $files[] = $data;
        $this->save('files', $files);
    }

    public function saveUserFile(int $userId, string $code, string $fileId, string $caption, string $type): void
    {
        $userFiles = $this->load('userfiles');
        $userFiles[] = [
            'user_id' => $userId,
            'code'    => $code,
            'file_id' => $fileId,
            'caption' => $caption,
            'type'    => $type,
        ];
        $this->save('userfiles', $userFiles);
    }
}


